/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor
 */
define(['jquery', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/config/ComplexCompositeBuilder'], function ($, CompositeEditor, ComplexCompositeBuilder) {
  'use strict';

  /**
   * A field editor for editing multiple slots on a `Complex`.
   *
   * If you extend from this editor, please read some very important
   * documentation on the `ComplexCompositeBuilder` class. Reading and
   * validating subclasses of this editor will, by default, be working with
   * `ComplexDiff` instances.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor
   * @see module:nmodule/webEditors/rc/fe/config/ComplexCompositeBuilder#readAll
   * @see module:nmodule/webEditors/rc/fe/baja/util/ComplexDiff
   */
  var ComplexCompositeEditor = function ComplexCompositeEditor() {
    CompositeEditor.apply(this, arguments);
  };
  ComplexCompositeEditor.prototype = Object.create(CompositeEditor.prototype);
  ComplexCompositeEditor.prototype.constructor = ComplexCompositeEditor;

  /**
   * Defines what slots this composite editor should show. This should return
   * one of two values:
   *
   * - An `Array` of slot names, or config Objects. Each config Object must
   *   have a `slot` property.
   * - A `Function` that accepts a `baja.Slot` parameter and returns a truthy
   *   value, or a config object, if that Slot should be included in the
   *   composite editor.
   *
   * See examples for usage.
   *
   * This function will always throw an error if not implemented by a subclass
   * or instance.
   *
   * @abstract
   * @returns {Array|Function} a slot filter array or function
   * @throws {Error} if not implemented by a subclass or instance
   *
   * @example
   *   <caption>Create and register a composite editor that always edits a
   *   particular group of slots</caption>
   *
   *   var StatusValueEditor = // subclass ComplexCompositeEditor
   *   StatusValueEditor.prototype.getSlotFilter = function () {
   *     return [ 'status', 'value' ];
   *   };
   *
   *   fe.buildFor({
   *     type: StatusValueEditor,
   *     value: baja.$('baja:StatusString'),
   *     dom: $('#myStatusStringEditor')
   *   })
   *     .then(function (ed) {
   *       //dom now contains a composite field editor that shows the default
   *       //field editors for the status and value slots
   *     });
   *
   * @example
   *   <caption>Define a composite editor with extra configuration defined for
   *   individual slots. This config data will be passed through to
   *   fe.buildFor().</caption>
   *
   *   var OverrideEditor = // subclass ComplexCompositeEditor
   *   OverrideEditor.prototype.getSlotFilter = function () {
   *     return [
   *       { slot: 'duration', type: OverrideRelTimeEditor },
   *       'value'
   *     ];
   *   };
   *
   *   fe.buildFor({
   *     type: OverrideEditor,
   *     value: baja.$('control:NumericOverride'),
   *     dom: $('#myOverrideEditor')
   *   })
   *     .then(function (ed) {
   *       //dom now contains a composite field editor that shows an
   *       //OverrideRelTimeEditor for the duration slot, and the default
   *       //field editor for the value slot
   *     });
   *
   * @example
   *   <caption>Overriding `getSlotFilter` allows you to use existing composite
   *   editor types, like a `PropertySheet`, and only edit slots that match
   *   your particular needs.</caption>
   *
   *   var stringPropSheet = new PropertySheet();
   *   stringPropSheet.getSlotFilter = function () {
   *     return function (slot) {
   *       return slot.getType().is('baja:String');
   *     };
   *   };
   *
   *   stringPropSheet.initialize($('#myElem')).then(function () {
   *     //only strings will be shown
   *     return stringPropSheet.load(myComponent);
   *   });
   */
  ComplexCompositeEditor.prototype.getSlotFilter = function () {
    throw new Error('getSlotFilter not implemented');
  };

  /**
   * Load individual editors for all slots that match this editor's slot filter.
   *
   * @param {baja.Complex} value
   * @returns {Promise}
   * @see #getSlotFilter
   */
  ComplexCompositeEditor.prototype.doLoad = function (value) {
    var builder = this.getBuilder();
    builder.setSlots(this.getSlotFilter(value));
    return CompositeEditor.prototype.doLoad.apply(this, arguments);
  };

  /**
   * Create the `CompositeBuilder` that will be used to manage the sub-editors
   * in this composite editor.
   *
   * @returns {module:nmodule/webEditors/rc/fe/config/ComplexCompositeBuilder}
   */
  ComplexCompositeEditor.prototype.makeBuilder = function () {
    var ed = this,
      builder = new ComplexCompositeBuilder(),
      setDataSource = builder.setDataSource;
    builder.setDataSource = function (complex) {
      var that = this;
      return setDataSource.apply(that, arguments).then(function () {
        return ed.getSlotFilter(complex);
      }).then(function (slotFilter) {
        return that.setSlots(slotFilter);
      });
    };
    builder.getDomFor = function (key) {
      return ed.$buildSubEditorDom(key);
    };
    return builder;
  };

  //TODO: add tests for when this method does something more complicated than just appending a div
  /**
   * Will be called during `load` once for each key to be edited. Ensure that
   * the element you create gets appended to `this.jq()`.
   *
   * @private
   * @param {String} key
   * @returns {*|HTMLElement}
   */
  ComplexCompositeEditor.prototype.$buildSubEditorDom = function (key) {
    return $('<div/>').appendTo(this.jq());
  };
  return ComplexCompositeEditor;
});
