/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/wb/mgr/BacnetPointManagerModel
 */
define(['baja!', 'lex!', 'lex!bacnet,driver', 'Promise', 'nmodule/driver/rc/wb/mgr/PointMgrModel', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/wb/mgr/model/columns/NameMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/columns/PathMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyPathMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/columns/IconMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/columns/TypeMgrColumn', 'nmodule/webEditors/rc/wb/table/model/Column', 'baja!bacnet:BacnetPointFolder,' + 'bacnet:BacnetObjectIdentifier,' + 'bacnet:BacnetProxyExt,' + 'bacnet:BacnetBooleanProxyExt,' + 'bacnet:BacnetEnumProxyExt,' + 'bacnet:BacnetNumericProxyExt,' + 'bacnet:BacnetStringProxyExt,' + 'control:ControlPoint,' + 'driver:ProxyExt'], function (baja, lexjs, lexs, Promise, PointMgrModel, typeUtils, NameMgrColumn, PathMgrColumn, PropertyMgrColumn, PropertyPathMgrColumn, IconMgrColumn, TypeMgrColumn, Column) {
  'use strict';

  var bacnetLex = lexs[0],
    driverLex = lexs[1],
    CONTROL_POINT_TYPE = baja.lt('control:ControlPoint'),
    PROXY_EXT_TYPE = baja.lt('driver:ProxyExt'),
    BACNET_PROXY_EXT_TYPE = baja.lt('bacnet:BacnetProxyExt'),
    BACNET_POINT_FOLDER_TYPE = baja.lt('bacnet:BacnetPointFolder'),
    UNSEEN = Column.flags.UNSEEN,
    EDITABLE = Column.flags.EDITABLE,
    READONLY = Column.flags.READONLY;
  function lex(key) {
    return bacnetLex.get(key);
  }

  ////////////////////////////////////////////////////////////////
  // Present Value Column
  ////////////////////////////////////////////////////////////////

  var OutColumn = function OutColumn(name, params) {
    Column.apply(this, arguments);
  };
  OutColumn.prototype = Object.create(Column.prototype);
  OutColumn.prototype.constructor = OutColumn;
  OutColumn.prototype.getValueFor = function (row) {
    return String(row.getSubject());
  };

  ////////////////////////////////////////////////////////////////
  // Proxy Conversion Column
  ////////////////////////////////////////////////////////////////

  var ConversionColumn = function ConversionColumn(name, params) {
    PropertyPathMgrColumn.apply(this, arguments);
  };
  ConversionColumn.prototype = Object.create(PropertyPathMgrColumn.prototype);
  ConversionColumn.prototype.constructor = ConversionColumn;
  ConversionColumn.prototype.buildCell = function (row, dom) {
    var conversion = this.getProposedValueFor(row);
    return typeUtils.getTypeDisplayName(conversion.getType()).then(function (name) {
      dom.text(name);
    });
  };

  ////////////////////////////////////////////////////////////////
  // BACnet Property Id Column
  ////////////////////////////////////////////////////////////////

  var PropertyIdColumn = function PropertyIdColumn(name, params) {
    PropertyPathMgrColumn.apply(this, arguments);
  };
  PropertyIdColumn.prototype = Object.create(PropertyPathMgrColumn.prototype);
  PropertyIdColumn.prototype.constructor = PropertyIdColumn;

  /**
   * Build the cell's content from the property id enumeration. If the value has an
   * ordinal for one of the values in the frozen range, we can trivially build it
   * from the display tag on the frozen type. If the value has an ordinal in the
   * dynamic range, it will see if it can obtain a display value from a lexicon
   * specified in the range's options.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {JQuery} dom
   * @returns {Promise.<*>}
   */
  PropertyIdColumn.prototype.buildCell = function (row, dom) {
    var value = this.getProposedValueFor(row),
      ordinal = value.getOrdinal(),
      range = value.getRange(),
      facets;
    if (range.isFrozenOrdinal(ordinal)) {
      dom.text(range.getFrozenType().getDisplayTag(ordinal));
    } else {
      facets = row.getSubject().getProxyExt().getFacets('propertyId');
      range = facets && facets.get('range');
      if (range && range.isFrozenOrdinal(ordinal)) {
        dom.text(range.getFrozenType().getDisplayTag(ordinal));
      } else if (range && range.isDynamicOrdinal(ordinal)) {
        return this.buildCellFromRangeFacets(range, ordinal, dom);
      } else {
        dom.text(String(ordinal));
      }
    }
    return Promise.resolve();
  };

  /**
   * This is used in the case where the property id enum has a dynamic range. Taking
   * the range from the facets obtained via the proxy ext, it will look for a module
   * lexicon to be specified in the options. If a lexicon is found, it will try to
   * build the cell from an entry corresponding to the tag. If not, it will build the
   * cell's content from the tag value.
   *
   * @param {baja.EnumRange} range - the enum range obtained from the proxy ext's facets
   * @param {Number} ordinal - the ordinal value of the property id
   * @param {JQuery} dom - the dom element
   * @returns {Promise.<*>}
   */
  PropertyIdColumn.prototype.buildCellFromRangeFacets = function (range, ordinal, dom) {
    var tag = range.getTag(ordinal),
      options = range.getOptions(),
      lexModule = options && options.get('lexicon');
    if (lexModule) {
      return this.getDisplayTagFromLexicon(lexModule, tag).then(function (str) {
        dom.text(str || baja.SlotPath.unescape(tag));
      });
    } else {
      dom.text(baja.SlotPath.unescape(tag));
      return Promise.resolve();
    }
  };

  /**
   * Returns a promise that will try to resolve to the tag name to a lexiconized
   * string. The name of the lexicon will have been obtained from the enum range's
   * options facet.
   *
   * @param {String} lexModule - the name of a module to use for the lexicon.
   * @param {String} tag - the tag of the property id enum.
   * @returns {Promise.<String>}
   */
  PropertyIdColumn.prototype.getDisplayTagFromLexicon = function (lexModule, tag) {
    return lexjs.module(lexModule).then(function (lex) {
      return lex.get(tag);
    });
  };

  ////////////////////////////////////////////////////////////////
  // BacnetPointManagerModel
  ////////////////////////////////////////////////////////////////

  /**
   * BacnetPointManagerModel constructor.
   *
   * @class
   * @alias module:nmodule/bacnet/rc/wb/mgr/model/BacnetPointManagerModel
   * @extends module:nmodule/driver/rc/wb/mgr/PointMgrModel
   * @param {Object} params
   */
  var BacnetPointManagerModel = function BacnetPointManagerModel(params) {
    PointMgrModel.call(this, {
      columns: makeColumns(this),
      component: params.component,
      newTypes: params.newTypes,
      folderType: BACNET_POINT_FOLDER_TYPE
    });
  };
  BacnetPointManagerModel.prototype = Object.create(PointMgrModel.prototype);
  BacnetPointManagerModel.prototype.constructor = BacnetPointManagerModel;

  /**
   * Return an array of `Column` instances for the main database table.
   *
   * @returns {Array.<module:nmodule/webEditors/rc/wb/table/model/Column>}
   */
  function makeColumns() {
    return [new IconMgrColumn(), new PathMgrColumn({
      flags: UNSEEN | READONLY
    }), new NameMgrColumn({
      flags: EDITABLE
    }), new TypeMgrColumn({
      flags: EDITABLE | UNSEEN
    }), new OutColumn('out', {
      displayName: driverLex.get('out'),
      flags: READONLY
    }), new PropertyPathMgrColumn('proxyExt/enabled', {
      type: PROXY_EXT_TYPE,
      flags: EDITABLE | UNSEEN
    }), new PropertyPathMgrColumn('proxyExt/objectId', {
      displayName: lex('learn.objectId'),
      type: BACNET_PROXY_EXT_TYPE,
      flags: EDITABLE
    }), new PropertyIdColumn('proxyExt/propertyId', {
      displayName: lex('pointManager.property'),
      type: BACNET_PROXY_EXT_TYPE,
      flags: EDITABLE
    }), new PropertyPathMgrColumn('proxyExt/propertyArrayIndex', {
      displayName: lex('pointManager.index'),
      type: BACNET_PROXY_EXT_TYPE,
      flags: EDITABLE
    }), new PropertyPathMgrColumn('proxyExt/tuningPolicyName', {
      type: PROXY_EXT_TYPE,
      flags: EDITABLE | UNSEEN
    }), new PropertyPathMgrColumn('proxyExt/dataType', {
      displayName: lex('pointManager.dataType'),
      type: BACNET_PROXY_EXT_TYPE,
      flags: EDITABLE | READONLY | UNSEEN
    }), new PropertyPathMgrColumn('proxyExt/readStatus', {
      displayName: lex('pointManager.read'),
      type: BACNET_PROXY_EXT_TYPE,
      flags: EDITABLE | READONLY
    }), new PropertyPathMgrColumn('proxyExt/writeStatus', {
      displayName: lex('pointManager.write'),
      type: BACNET_PROXY_EXT_TYPE,
      flags: EDITABLE | READONLY
    }), new PropertyPathMgrColumn('proxyExt/deviceFacets', {
      type: BACNET_PROXY_EXT_TYPE,
      flags: EDITABLE | UNSEEN | READONLY
    }), new PropertyMgrColumn('facets', {
      type: CONTROL_POINT_TYPE,
      flags: EDITABLE | UNSEEN
    }), new ConversionColumn('proxyExt/conversion', {
      type: PROXY_EXT_TYPE,
      flags: EDITABLE | UNSEEN
    }), new PropertyPathMgrColumn('proxyExt/readValue', {
      type: PROXY_EXT_TYPE,
      flags: UNSEEN
    }), new PropertyPathMgrColumn('proxyExt/writeValue', {
      type: PROXY_EXT_TYPE,
      flags: UNSEEN
    }), new PropertyPathMgrColumn('proxyExt/faultCause', {
      type: PROXY_EXT_TYPE,
      flags: UNSEEN
    })];
  }

  /**
   * Create the default facets to be used for a ControlPoint instance. The contents
   * of the returned facets instance will depend on the point's data type.
   *
   * @param {Type} pointType
   * @returns {baja.Facets}
   */
  function makeControlPointFacets(pointType) {
    if (pointType.is('control:BooleanPoint')) {
      return baja.Facets.make({
        trueText: 'true',
        falseText: 'false'
      });
    } else if (pointType.is('control:NumericPoint')) {
      return baja.Facets.make({
        units: baja.Unit.NULL,
        precision: 1,
        min: Number.NEGATIVE_INFINITY,
        max: Number.POSITIVE_INFINITY
      });
    } else if (pointType.is('control:EnumPoint')) {
      return baja.Facets.make({
        range: baja.EnumRange.DEFAULT
      });
    }
    return baja.Facets.DEFAULT;
  }

  /**
   * Create a new control point instance from the provided `MgrTypeInfo`,
   * setting the correct type of proxy extension on it according to the
   * data type of the control point.
   *
   * @param {module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo} typeInfo
   */
  BacnetPointManagerModel.prototype.newInstance = function (typeInfo) {
    return typeInfo.newInstance().then(function (point) {
      var pointType = point.getType(),
        facets = makeControlPointFacets(pointType),
        proxyExt;
      if (pointType.is('control:BooleanPoint')) {
        proxyExt = baja.$('bacnet:BacnetBooleanProxyExt');
      } else if (pointType.is('control:NumericPoint')) {
        proxyExt = baja.$('bacnet:BacnetNumericProxyExt');
      } else if (pointType.is('control:EnumPoint')) {
        proxyExt = baja.$('bacnet:BacnetEnumProxyExt');
      } else if (pointType.is('control:StringPoint')) {
        proxyExt = baja.$('bacnet:BacnetStringProxyExt');
      } else {
        proxyExt = baja.$('control:NullProxyExt');
      }
      point.set({
        slot: 'facets',
        value: facets
      });
      point.setProxyExt(proxyExt);
      if (proxyExt.getType().is('driver:ProxyExt')) {
        proxyExt.setDeviceFacets(facets.newCopy());
      }
      return point;
    });
  };
  return BacnetPointManagerModel;
});
