/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/BacnetWeekNDayEditor
 */
define(['baja!', 'lex!bacnet', 'underscore', 'nmodule/bacnet/rc/baja/datatypes/BacnetOctetString', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/baja/OrderedMapEditor', 'nmodule/webEditors/rc/fe/config/CompositeBuilder', 'css!nmodule/bacnet/rc/bacnet'], function (baja, lexs, _, BacnetOctetString, CompositeEditor, OrderedMapEditor, CompositeBuilder) {
  'use strict';

  var lex = lexs[0],
    KEYS = ['month', 'week', 'day'],
    getMap = function getMap(key) {
      var octetValues;
      if (key === 'month') {
        octetValues = [-1].concat(_.range(1, 15));
      }
      if (key === 'week') {
        octetValues = [-1].concat(_.range(1, 10));
      }
      if (key === 'day') {
        octetValues = [-1].concat(_.range(1, 8));
      }
      var octetNum = KEYS.indexOf(key);
      var map = new baja.OrderedMap();
      _.each(octetValues, function (octetValue) {
        map.put(lex.get('weekNDay.b' + octetNum + '.' + octetValue), octetValue);
      });
      return map;
    },
    getMonthMap = _.once(function () {
      return getMap('month');
    }),
    getWeekMap = _.once(function () {
      return getMap('week');
    }),
    getDayMap = _.once(function () {
      return getMap('day');
    });

  /**
   * Editor for handling `bacnet:BacnetOctetString` values in the Week N Day
   * part of a `bacnet:BacnetCalendarEntry`. The editor consists of three
   * drop down boxes.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   * @alias module:nmodule/bacnet/rc/fe/BacnetWeekNDayEditor
   */
  var BacnetWeekNDayEditor = function BacnetWeekNDayEditor() {
    CompositeEditor.apply(this, arguments);
  };
  BacnetWeekNDayEditor.prototype = Object.create(CompositeEditor.prototype);
  BacnetWeekNDayEditor.prototype.constructor = BacnetWeekNDayEditor;

  /**
   * Configures the composite builder for the loaded `bacnet:BacnetOctetString`
   * instance.
   *
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */
  BacnetWeekNDayEditor.prototype.makeBuilder = function () {
    var that = this,
      builder = new CompositeBuilder();
    builder.getDomFor = function (key) {
      return that.jq().children('.js-' + key);
    };
    builder.getKeys = function () {
      return KEYS.slice();
    };
    builder.getValueFor = function (key) {
      switch (key) {
        case 'month':
          return getMonthMap();
        case 'week':
          return getWeekMap();
        case 'day':
          return getDayMap();
      }
    };
    builder.getConfigFor = function (key) {
      return {
        type: OrderedMapEditor
      };
    };
    return builder;
  };

  /**
   * Create the DOM structure for housing the each of the drop down boxes.
   *
   * @param {JQuery} dom
   * @returns {Promise}
   */
  BacnetWeekNDayEditor.prototype.doInitialize = function (dom) {
    dom.html('<span class="js-month"/>' + '<span class="js-week"/>' + '<span class="js-day"/>');
    return CompositeEditor.prototype.doInitialize.apply(this, arguments);
  };

  /**
   * Select an item in each drop down box that corresponds to one octet in
   * the backing octet string.
   *
   * @param {module:nmodule/bacnet/rc/baja/datatypes/BacnetOctetString} octetString
   * @returns {Promise}
   */
  BacnetWeekNDayEditor.prototype.doLoad = function (octetString) {
    var that = this,
      len = octetString.getBytes().length;
    if (len < 3) {
      throw new Error('Invalid bit string length for WeekNDay: ' + len);
    }
    return CompositeEditor.prototype.doLoad.apply(that, arguments).then(function () {
      var kids = that.getChildWidgets(),
        bytes = octetString.getBytes();
      _.each(kids, function (kid, index) {
        kid.setSelectedValue(bytes[index]);
      });
    });
  };

  /**
   * Read a new `bacnet:BacnetOctetString` instance from the currently selected
   * items in each drop down box.
   *
   * @returns {Promise.<module:nmodule/bacnet/rc/baja/datatypes/BacnetOctetString>}
   */
  BacnetWeekNDayEditor.prototype.doRead = function () {
    return this.getChildWidgets().readAll().then(BacnetOctetString.make);
  };
  return BacnetWeekNDayEditor;
});
