/**
 * @copyright 2025 Tridium, Inc. All Rights Reserved.
 * @author Uday Rapuru on 25-Mar-2025
 *
 * @since Niagara 4.14u2
 * @since Niagara 4.15u1
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/BacnetOptionalValueEditor
 */
define(['baja!', 'bajaux/events', 'Promise', 'nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor', 'css!nmodule/bacnet/rc/bacnet'], function (baja, events, Promise, ComplexCompositeEditor) {
  'use strict';

  var DESTROY_EVENT = events.DESTROY_EVENT,
    DISABLE_EVENT = events.DISABLE_EVENT,
    ENABLE_EVENT = events.ENABLE_EVENT,
    INITIALIZE_EVENT = events.INITIALIZE_EVENT,
    LOAD_EVENT = events.LOAD_EVENT,
    MODIFY_EVENT = events.MODIFY_EVENT,
    READONLY_EVENT = events.READONLY_EVENT,
    WRITABLE_EVENT = events.WRITABLE_EVENT;

  /**
   * Editor for working with bacnet optional values.
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor
   * @alias module:nmodule/bacnet/rc/fe/BBacnetOptionalValueEditor
   */
  var BacnetOptionalValueEditor = function BacnetOptionalValueEditor() {
    ComplexCompositeEditor.apply(this, arguments);
  };
  BacnetOptionalValueEditor.prototype = Object.create(ComplexCompositeEditor.prototype);
  BacnetOptionalValueEditor.prototype.constructor = BacnetOptionalValueEditor;

  /**
   * Initialize editors for the `isNull` and `value` slots.
   *
   * @returns {Array}
   */
  BacnetOptionalValueEditor.prototype.getSlotFilter = function () {
    return [{
      slot: 'isNull',
      properties: {
        trueText: 'null',
        falseText: 'null'
      }
    }, {
      slot: 'value',
      readonly: true
    }];
  };

  /**
   * Get the boolean editor for the 'null' checkbox.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/BooleanEditor}
   */
  BacnetOptionalValueEditor.prototype.$getNullEditor = function () {
    return this.getBuilder().getEditorFor('isNull');
  };

  /**
   * Get the editor for the optional value.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/NumericEditor}
   */
  BacnetOptionalValueEditor.prototype.$getValueEditor = function () {
    return this.getBuilder().getEditorFor('value');
  };

  /**
   * Update the readonly status of the 'isNull' checkbox and the value editor as appropriate.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  BacnetOptionalValueEditor.prototype.$updateReadonly = function (readonly) {
    var nullEditor = this.$getNullEditor(),
      valueEditor = this.$getValueEditor();
    return nullEditor && nullEditor.read().then(function (isNull) {
      return Promise.all([nullEditor.setReadonly(readonly), valueEditor.setReadonly(isNull || readonly)]);
    });
  };

  /**
   * Arm event handlers to toggle the value editor's readonly status when
   * 'isNull' is checked and unchecked.
   *
   * @param {jQuery} dom
   */
  BacnetOptionalValueEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.on([DESTROY_EVENT, DISABLE_EVENT, ENABLE_EVENT, INITIALIZE_EVENT, LOAD_EVENT, READONLY_EVENT, WRITABLE_EVENT].join(' '), '.editor', false);
    dom.on(MODIFY_EVENT, '.key-isNull', function (e, ed) {
      that.$updateReadonly(that.isReadonly());
      that.setModified(true);
      return false;
    });
  };

  /**
   * Load in the `bacnet:BacnetOptionalValue` and ensure the readonly status of the
   * value editor is correct.
   *
   * @param {baja.Struct} optionalValue a `bacnet:BacnetOptionalValue` instance
   * @returns {Promise}
   */
  BacnetOptionalValueEditor.prototype.doLoad = function (optionalValue) {
    var that = this;
    return ComplexCompositeEditor.prototype.doLoad.apply(this, arguments).then(function () {
      return that.$updateReadonly(that.isReadonly());
    });
  };

  /**
   * Update the readonly status of the 'isNull' checkbox and the value editor as appropriate.
   * The value editor will always be readonly if 'isNull' is checked.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  BacnetOptionalValueEditor.prototype.doReadonly = function (readonly) {
    return this.$updateReadonly(readonly);
  };
  return BacnetOptionalValueEditor;
});
