/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/BacnetDateEditor
 */
define(['baja!', 'jquery', 'Promise', 'nmodule/bacnet/rc/baja/datatypes/BacnetDate', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/fe', 'bajaux/events', 'bajaux/Widget', 'bajaux/commands/Command', 'bajaux/util/CommandButtonGroup', 'nmodule/webEditors/rc/wb/commands/DatePickerPopupCommand', 'hbs!nmodule/bacnet/rc/fe/templates/BacnetDateEditor', 'nmodule/bacnet/rc/baja/util/BacnetDateUtil', 'css!nmodule/bacnet/rc/bacnet', 'baja!baja:Weekday'], function (baja, $, Promise, BacnetDate, BaseEditor, fe, events, Widget, Command, CommandButtonGroup, DatePickerPopupCommand, templateEditor, BacnetDateUtil) {
  'use strict';

  var DESTROY_EVENT = events.DESTROY_EVENT,
    INITIALIZE_EVENT = events.INITIALIZE_EVENT,
    LOAD_EVENT = events.LOAD_EVENT,
    ENABLE_EVENT = events.ENABLE_EVENT,
    DISABLE_EVENT = events.DISABLE_EVENT,
    MODIFY_EVENT = events.MODIFY_EVENT,
    READONLY_EVENT = events.READONLY_EVENT,
    WRITABLE_EVENT = events.WRITABLE_EVENT,
    BACDATE_FORMAT = "YYYY-MM-DD-W";

  ////////////////////////////////////////////////////////////////
  // Commands
  ////////////////////////////////////////////////////////////////

  /**
   * Command to reset the editor to default BacnetDate.
   *
   * @inner
   * @class
   * @extends {module:bajaux/commands/Command}
   * @param {module:nmodule/bacnet/rc/fe/BacnetDateEditor} ed
   */
  var AsteriskCommand = function AsteriskCommand(ed) {
    Command.call(this, {
      module: 'bacnet',
      lex: "BacnetEd.asteriskCommand",
      func: function func() {
        ed.setModified(true);
        return ed.doLoad(BacnetDate.DEFAULT);
      }
    });
  };
  AsteriskCommand.prototype = Object.create(Command.prototype);
  AsteriskCommand.prototype.constructor = AsteriskCommand;

  /**
   * Command to pick a date from a calendar.
   *
   * @inner
   * @class
   * @extends {module:bajaux/commands/Command}
   * @param {module:nmodule/bacnet/rc/fe/BacnetDateEditor} ed
   */
  var CalendarCommand = function CalendarCommand(ed) {
    DatePickerPopupCommand.call(this);
    var func = this.getFunction();
    this.setFunction(function () {
      return func().then(function (date) {
        var bDt = baja.Date.make({
          jsDate: date
        });
        return bDt.toDateString({
          textPattern: BACDATE_FORMAT
        }).then(function (dtStr) {
          ed.setModified(true);
          return ed.$getDateInput().load(dtStr);
        });
      });
    });
  };
  CalendarCommand.prototype = Object.create(Command.prototype);
  CalendarCommand.prototype.constructor = CalendarCommand;

  /**
   * Editor for handling `bacnet:BacnetDate` values.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/bacnet/rc/fe/BacnetDateEditor
   */
  var BacnetDateEditor = function BacnetDateEditor() {
    var that = this;
    BaseEditor.apply(this, arguments);
    that.getCommandGroup().add(new AsteriskCommand(that), new CalendarCommand(that));
  };
  BacnetDateEditor.prototype = Object.create(BaseEditor.prototype);
  BacnetDateEditor.prototype.constructor = BacnetDateEditor;

  /**
   * Return the input dom selector that stores the date
   * @returns {module:bajaux/Widget} the dom selector
   */
  BacnetDateEditor.prototype.$getDateInput = function () {
    return Widget["in"](this.jq().find('.js-bacDate'));
  };

  /**
   * @private
   * @returns {module:bajaux/commands/Command}
   */
  BacnetDateEditor.prototype.$getAsteriskCommand = function () {
    return this.getCommandGroup().get(0);
  };

  /**
   * @private
   * @returns {module:bajaux/commands/Command}
   */
  BacnetDateEditor.prototype.$getCalendarCommand = function () {
    return this.getCommandGroup().get(1);
  };

  /**
   * Get the date string to load into the input widget
   * 
   * @param {module:nmodule/bacnet/rc/baja/datatypes/BacnetDate} bacnetDate
   * @returns {string}
   */
  BacnetDateEditor.prototype.$getBacnetDateDisplayString = function (bacnetDate) {
    return BacnetDateUtil.getDisplayBacnetDateString(bacnetDate.encodeToString());
  };

  /**
   * Initialise the Bacnet Date widget
   * @param {JQuery} dom
   */
  BacnetDateEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.html(templateEditor());
    //Disable events from bubbling up
    dom.on([DESTROY_EVENT, INITIALIZE_EVENT, LOAD_EVENT, ENABLE_EVENT, DISABLE_EVENT, READONLY_EVENT, WRITABLE_EVENT].join(' '), '.editor', false);
    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });
    return Promise.all([fe.buildFor({
      dom: $('.js-bacDate', that.jq()),
      value: BacnetDate.DEFAULT.encodeToString(),
      properties: {
        uxFieldEditor: "webEditors:StringEditor"
      },
      formFactor: 'mini'
    }), fe.buildFor({
      dom: $('.commands', that.jq()),
      type: CommandButtonGroup,
      value: this.getCommandGroup(),
      properties: {
        toolbar: true
      },
      formFactor: 'mini'
    })]);
  };

  /**
   * @param {baja.Simple} bacnetDate the `bacnet:BacnetDate` value to load
   */
  BacnetDateEditor.prototype.doLoad = function (bacnetDate) {
    return this.$getDateInput().load(this.$getBacnetDateDisplayString(bacnetDate));
  };
  BacnetDateEditor.prototype.doRead = function () {
    var that = this;
    return that.$getDateInput().read().then(function (inputDate) {
      return inputDate ? BacnetDate.DEFAULT.decodeFromString(BacnetDateUtil.getEncodedBacnetDateString(inputDate)) : BacnetDate.DEFAULT;
    });
  };

  /**
   * Enables or disables the date field.
   *
   * @param {Boolean} enabled
   */
  BacnetDateEditor.prototype.doEnabled = function (enabled) {
    return this.getChildWidgets().setAllEnabled(enabled);
  };

  /**
   * Disables or enables the date field.
   *
   * @param {Boolean} readonly
   */
  BacnetDateEditor.prototype.doReadonly = function (readonly) {
    return this.getChildWidgets().setAllReadonly(readonly);
  };
  BacnetDateEditor.prototype.doDestroy = function () {
    return this.getChildWidgets().destroyAll();
  };
  return BacnetDateEditor;
});
