/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 * @since Niagara 4.2
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/BacnetAddressEditor
 */
define(['baja!', 'baja!bacnet:BacnetAddress,bacnet:BacnetOctetString', 'bajaux/events', 'bajaux/Widget', 'hbs!nmodule/bacnet/rc/fe/templates/BacnetAddressEditor', 'lex!bacnet', 'nmodule/bacnet/rc/baja/datatypes/BacnetAddress', 'nmodule/bacnet/rc/fe/MacAddressEditor', 'nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor', 'nmodule/webEditors/rc/fe/baja/OrderedMapEditor', 'Promise', 'underscore'], function (baja, types, events, Widget, template, lexs, BacnetAddress, MacAddressEditor, ComplexCompositeEditor, OrderedMapEditor, Promise, _) {
  'use strict';

  var lex = lexs[0],
    lexNetNum = lex.get({
      key: "bacnetAddress.networkNumber",
      def: 'Network Number'
    }),
    lexMacAddr = lex.get({
      key: 'bacnetAddress.macAddress',
      def: 'MAC Address'
    }),
    lexMacStyle = lex.get({
      key: 'bacnetAddress.macStyle',
      def: 'MAC Address Style'
    }),
    NETWORK_NUMBER_LABEL_SELECTOR = ".netLabel",
    MAC_ADDRESS_LABEL_SELECTOR = ".macLabel",
    STYLE_LABEL_SELECTOR = ".styleLabel",
    NETWORK_NUMBER_EDITOR_CLASS = "networkNumber",
    MAC_ADDRESS_EDITOR_CLASS = "macAddress",
    MAC_STYLE_CLASS = "addressType",
    NETWORK_NUMBER_EDITOR_SELECTOR = '.' + NETWORK_NUMBER_EDITOR_CLASS,
    MAC_ADDRESS_EDITOR_SELECTOR = '.' + MAC_ADDRESS_EDITOR_CLASS,
    MAC_STYLE_SELECTOR = '.' + MAC_STYLE_CLASS,
    UNKNOWN = BacnetAddress.MAC_TYPE_UNKNOWN,
    MSTP = BacnetAddress.MAC_TYPE_MSTP,
    // index = choice
    // key = lex lookup key
    // def = default text if no lex entry is found
    addressTypesLex = [{
      key: 'bacnetAddress.macType.Unknown',
      def: 'Unknown'
    }, {
      key: 'bacnetAddress.macType.Ethernet',
      def: 'Ethernet'
    }, {
      key: 'bacnetAddress.macType.IP',
      def: 'IP'
    }, {
      key: 'bacnetAddress.macType.MSTP',
      def: 'MSTP/Other'
    }, {
      key: 'bacnetAddress.macType.SC',
      def: 'SC'
    }],
    addressTypes = _.map(addressTypesLex, function (entry) {
      return lex.get(entry);
    });

  /**
   * BacnetAddressEditor is a field editor for a bacnet:BacnetAddress type
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/bacnet/rc/fe/BacnetAddressEditor
   */
  var BacnetAddressEditor = function BacnetAddressEditor() {
    ComplexCompositeEditor.apply(this, arguments);
  };

  //extend and set up prototype chain
  BacnetAddressEditor.prototype = Object.create(ComplexCompositeEditor.prototype);
  BacnetAddressEditor.prototype.constructor = BacnetAddressEditor;

  /**
   * Returns the text of the Network Number label
   * @private
   * @returns {String}
   */
  BacnetAddressEditor.prototype.$getNetworkNumberLabel = function () {
    return this.jq().find(NETWORK_NUMBER_LABEL_SELECTOR).text();
  };

  /**
   * Returns the text of the MAC Address label
   * @private
   * @returns {String}
   */
  BacnetAddressEditor.prototype.$getMacAddressLabel = function () {
    return this.jq().find(MAC_ADDRESS_LABEL_SELECTOR).text();
  };

  /**
   * Returns the text of the MAC Address Style label
   * @private
   * @returns {String}
   */
  BacnetAddressEditor.prototype.$getMacStyleLabel = function () {
    return this.jq().find(STYLE_LABEL_SELECTOR).text();
  };

  /**
   * Returns the IntegerEditor for the Network Number field
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/IntegerEditor}
   */
  BacnetAddressEditor.prototype.$getNetworkNumberEditor = function () {
    return Widget["in"](this.jq().find(NETWORK_NUMBER_EDITOR_SELECTOR));
  };

  /**
   * Returns the MacAddressEditor for the MAC Address field
   * @private
   * @returns {module:nmodule/bacnet/rc/fe/MacAddressEditor}
   */
  BacnetAddressEditor.prototype.$getMacAddressEditor = function () {
    return Widget["in"](this.jq().find(MAC_ADDRESS_EDITOR_SELECTOR));
  };

  /**
   * Returns the OrderedMapEditor for the MAC Address style field
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/OrderedMapEditor}
   */
  BacnetAddressEditor.prototype.$getMacStyleEditor = function () {
    return Widget["in"](this.jq().find(MAC_STYLE_SELECTOR));
  };

  /**
   * Initializes the DOM with fields for network number, MAC address and address style
   *
   * @param dom The jQuery DOM element to be initialized
   * @returns {Promise} Promise resolved when editor is initialized
   */
  BacnetAddressEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.addClass('BacnetAddressEditor').html(template({
      netLabel: lexNetNum,
      macLabel: lexMacAddr,
      styleLabel: lexMacStyle,
      networkNumber: NETWORK_NUMBER_EDITOR_CLASS,
      macAddress: MAC_ADDRESS_EDITOR_CLASS,
      addressType: MAC_STYLE_CLASS
    }));

    //arm the address style event handler to set the 'addressType' property on the MacAddressEditor
    //this property is used to access the current addressType when calling MacAddressEditor.doRead()
    dom.on(events.MODIFY_EVENT, MAC_STYLE_SELECTOR, function (e, ed) {
      var macStyleEd = ed,
        selectedValue = macStyleEd && macStyleEd.getSelectedValue(),
        macAddrEd = that.$getMacAddressEditor(),
        macAddress = macAddrEd && macAddrEd.$getMacAddress(),
        translatedMacAddr;
      if (macAddress !== null && selectedValue !== null) {
        macAddrEd.properties().setValue('addressType', selectedValue);
        translatedMacAddr = BacnetAddress.convertMacAddress(selectedValue, macAddress);
        macAddrEd.$getTextElement().val(translatedMacAddr);
        macAddrEd.setModified(true);
      }
    });
    return ComplexCompositeEditor.prototype.doInitialize.apply(this, arguments);
  };

  /**
   * Loads the editor with a bacnet:BacnetAddress
   * @param address An instance of bacnet:BacnetAddress
   * @returns {Promise} Promise resolved when the editor has been loaded with the address
   */
  BacnetAddressEditor.prototype.doLoad = function (address) {
    if (!baja.hasType(address, 'bacnet:BacnetAddress')) {
      throw new Error('editor must be loaded with bacnet:BacnetAddress');
    }
    var that = this,
      macAddress = address.getMacAddress(),
      addressType = address.getAddressType(),
      bytes = macAddress.getBytes(),
      macAddrStyle = !bytes.length ? UNKNOWN : bytes.length === 1 ? MSTP : addressType;
    return ComplexCompositeEditor.prototype.doLoad.apply(this, arguments).then(function () {
      that.$getMacStyleEditor().setSelectedValue(macAddrStyle);
    });
  };

  /**
   * Reads a new `BacnetAddress` with current values.
   *
   * @returns {Promise.<baja.Struct>}
   */
  BacnetAddressEditor.prototype.doRead = function () {
    var that = this;
    return ComplexCompositeEditor.prototype.doRead.apply(this, arguments).then(function (diff) {
      return diff.apply(that.value().newCopy());
    });
  };

  ///////////////////////////////////////////////////////////////////
  //ComplexCompositeEditor overrides
  //////////////////////////////////////////////////////////////////
  BacnetAddressEditor.prototype.getSlotFilter = function () {
    return ['networkNumber', {
      slot: 'macAddress',
      type: MacAddressEditor
    }, {
      slot: 'addressType',
      type: OrderedMapEditor
    }];
  };
  BacnetAddressEditor.prototype.makeBuilder = function () {
    var that = this,
      builder = ComplexCompositeEditor.prototype.makeBuilder.apply(this, arguments),
      valueFor = builder.getValueFor,
      map = new baja.OrderedMap();
    _.each(addressTypes, function (item, index) {
      map.put(item, index);
    });
    builder.getDomFor = function (key) {
      return that.jq().find('.' + key);
    };
    builder.getValueFor = function (key) {
      switch (key) {
        case 'addressType':
          return map;
        case 'macAddress':
          var ed = this.getEditorFor(key),
            ds = this.getDataSource(),
            addressType = ds.getAddressType(),
            macAddress = ds.getMacAddress();
          if (ed) {
            ed.properties().add('addressType', addressType);
          }
          return macAddress;
        default:
          return valueFor.apply(this, arguments);
      }
    };
    return builder;
  };
  return BacnetAddressEditor;
});
