/*
 * Copyright 2000 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.web;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import javax.servlet.http.HttpServletRequest;

import javax.baja.util.Version;

import com.tridium.nre.util.NUserAgent;

/**
 * UserAgent is used to parse the user-agent HTTP
 * header field to provide information about the
 * browser (or other software) making an HTTP request.
 * <p>
 * This class is based on the following BNF grammar
 * defined for the User-Agent header:
 * <pre>
 *   User-Agent  = "User-Agent" ":" 1*( product | comment )
 *   product  = token ["/" product-version ]
 *   product-version  = token
 *   comment  = "(" *token ")"
 * </pre>
 * <p>
 * Refactored:
 *   @author Bill Smith on 17 Jan 2024
 *   @since Niagara 4.15
 *
 * Originally:
 *   @since Baja 1.0
 *   @author Brian Frank
 *   @version $Revision: 5$ $Date: 4/5/05 8:30:55 AM EDT$
 *   @creation 16 Nov 00
 */
public class UserAgent
{

  /**
   * Constructor parses the user-agent header value.
   */
  public UserAgent(String value)
  {
    this.value = value;

    NUserAgent nuserAgent = new NUserAgent(value);
    product = new Product(nuserAgent.getProduct());

    List<Product> products = new ArrayList<>();
    for (NUserAgent.NProduct coreNProduct : nuserAgent.getProducts())
    {
      products.add(new Product(coreNProduct));
    }
    this.products = products.toArray(PRODUCT_ARRAY_DEFINITION);

    comments = nuserAgent.getComments().clone();
  }

  /**
   * Parse a product token into an instance of Product.
   */
  public static Product parseProduct(String token)
  {
    return new Product(NUserAgent.parseProduct(token));
  }

  /**
   * Convenience for <code>getProduct().isIE()</code>.
   *
   * @deprecated IE11 is no longer supported in Niagara. Will be removed in Niagara 5.0
   */
  @Deprecated
  public boolean isIE()
  {
    return getProduct().isIE();
  }

  /**
   * Convenience for <code>getProduct().isMozilla()</code>.
   */
  public boolean isMozilla()
  {
    return getProduct().isMozilla();
  }

  /**
   * Tests the user agent string to determine whether the HTTP request
   * was made by the Niagara AX Web Start application.
   *
   * @since Niagara 3.8U1
   * @deprecated since Niagara 4.13 - will be removed in Niagara 5.0
   */
  @Deprecated
  public boolean isNiagaraAxWebStart()
  {
    return value.toLowerCase().contains(AX_WEBSTART);
  }

  /**
   * Static helper method that will inspect the user-agent header string to determine
   * whether the HTTP request was made by the Niagara AX Web Start application.
   *
   * @param req An HTTP request instance
   * @return true if the user agent header indicates the request came from the Web Start app.
   * @since Niagara 3.8U1
   * @deprecated since Niagara 4.13 - will be removed in Niagara 5.0
   */
  @Deprecated
  public static boolean isNiagaraAxWebStart(HttpServletRequest req)
  {
    return hasUserAgent(req, AX_WEBSTART);
  }

  /**
   * Tests the user agent string to determine whether the HTTP request
   * was made by the Niagara 4 Web Start application.
   *
   * @since Niagara 4.2
   * @deprecated since Niagara 4.13 - will be removed in Niagara 5.0
   */
  @Deprecated
  public boolean isNiagara4WebStart()
  {
    return value.toLowerCase().contains(N4_WEBSTART);
  }

  /**
   * Static helper method that will inspect the user-agent header string to determine
   * whether the HTTP request was made by the Niagara 4 Web Start application.
   *
   * @param req An HTTP request instance
   * @return true if the user agent header indicates the request came from the Web Start app.
   * @since Niagara 4.2
   * @deprecated since Niagara 4.13 - will be removed in Niagara 5.0
   */
  @Deprecated
  public static boolean isNiagara4WebStart(HttpServletRequest req)
  {
    return hasUserAgent(req, N4_WEBSTART);
  }

  /**
   * Tests the user agent string to determine whether the HTTP request
   * was made by the Niagara Web Launcher application.
   *
   * @since Niagara 4.4 Update 3 patch (4.4.94.12.2)
   * @deprecated since Niagara 4.13 - will be removed in Niagara 5.0
   */
  @Deprecated
  public boolean isNiagaraWebLauncher()
  {
    return value.toLowerCase().contains(NIAGARA_WEB_LAUNCHER);
  }

  /**
   * Static helper method that will inspect the user-agent header string to determine
   * whether the HTTP request was made by the Niagara Web Launcher application.
   *
   * @param req An HTTP Request instance
   * @return true if the user agent indicates the request came from Niagara Web Launcher app
   * @since Niagara 4.4 Update 3 patch (4.4.94.12.2)
   * @deprecated since Niagara 4.13 - will be removed in Niagara 5.0
   */
  @Deprecated
  public static boolean isNiagaraWebLauncher(HttpServletRequest req)
  {
    return hasUserAgent(req, NIAGARA_WEB_LAUNCHER);
  }

  /**
   * Helper to check if the header contains the passed in user agent identifier
   *
   * @param req     An HTTP request instance
   * @param agentId The user agent identifier to check the header against
   * @return true if the the User-Agent header contains the agentId
   */
  private static boolean hasUserAgent(HttpServletRequest req, String agentId)
  {
    String header = req.getHeader("User-Agent");
    return header != null ? header.toLowerCase().contains(agentId) : false;
  }

  /**
   * Get the primary product of the user agent.  In
   * most cased this is the first product listed.
   * The exception is Microsoft IE which lists its
   * product incorrectly as a comment.
   */
  public Product getProduct()
  {
    return product;
  }

  /**
   * Get the list of all the product tokens
   * sent in the user agent header.
   */
  public Product[] getProducts()
  {
    return products.clone();
  }

  /**
   * Get the list of parsed comments which
   * is all the tokens found inside a "(..)"
   * separated by semicolons.
   */
  public String[] getComments()
  {
    return comments.clone();
  }

  /**
   * To string returns the unparsed header value.
   */
  public String toString()
  {
    return value;
  }

  /**
   * Dump to standard output.
   */
  public void dump()
  {
    System.out.println("User-Agent: " + value);
    System.out.println("  Primary Product: " + product);
    System.out.println("  Products:");
    for (Product item : products)
    {
      System.out.println("    " + item);
    }
    System.out.println("  Comments:");
    for (String comment : comments)
    {
      System.out.println("    " + comment);
    }
  }

  /**
   * The Product class stores the name an optional
   * version of a product token in a User-Agent header.
   * If the version not included, then version and
   * versionString are null.  If it is included then
   * versionString is the String value.  If the version
   * can be parsed as a Version, then version contains
   * that instance.
   */
  public static class Product
  {
    /**
     * Construct a new Product instance.
     */
    public Product(String name, String versionString, Version version)
    {
      nproduct = new NUserAgent.NProduct(name, versionString, new com.tridium.nre.util.Version(version.toString()));
    }

    private Product(NUserAgent.NProduct nproduct)
    {
      this.nproduct = nproduct;
    }

    /**
     * Return true if the product name equals
     * UserAgent.MSIE = "MSIE", indicating
     * Microsoft Internet Explorer.
     *
     * @deprecated IE is no longer supported in Niagara. Will be removed in Niagara 5.0
     */
    @Deprecated
    public boolean isIE()
    {
      return nproduct.getName().equals(MSIE);
    }

    /**
     * Return true if the product name equals
     * UserAgent.MOZILLA = "Mozilla", indicating
     * a Netscape product.  Versions 5.0 and
     * after indicate the open source version
     * of Netscape based on Gecko.
     */
    public boolean isMozilla()
    {
      return nproduct.getName().equals(MOZILLA);
    }

    /**
     * Checks for equality between two products
     */
    @Override
    public boolean equals(Object obj)
    {
      if (obj == null)
      {
        return false;
      }

      if (obj.getClass() != getClass())
      {
        return false;
      }

      Product product = (Product) obj;
      if (!Objects.equals(product.nproduct, nproduct))
      {
        return false;
      }

      return true;
    }

    /**
     * Computes the hashcode of a product
     */
    @Override
    public int hashCode()
    {
      return Objects.hash(nproduct);
    }

    /**
     * Is a string representation of the Product object.
     */
    public String toString()
    {
      if (nproduct.getVersionString() == null)
      {
        return nproduct.getName();
      }
      else if (nproduct.getVersion() == null)
      {
        return nproduct.getName() + '/' + nproduct.getVersionString();
      }
      else
      {
        return nproduct.getName() + '/' + nproduct.getVersion();
      }
    }

    private final NUserAgent.NProduct nproduct;
  }

  private final String value;
  private final Product product;
  private final Product[] products;
  private final String[] comments;

  public static final String MSIE = "MSIE";
  public static final String MOZILLA = "Mozilla";
  private static final String AX_WEBSTART = "niagaraax/webstart";
  private static final String N4_WEBSTART = "niagara4/webstart";
  private static final String NIAGARA_WEB_LAUNCHER = "niagara/weblauncher";

  private static final Product[] PRODUCT_ARRAY_DEFINITION = new Product[0];
}

