/*
 * Copyright 2023 Tridium, Inc. All Rights Reserved.
 */

package javax.baja.test;

import static org.testng.Assert.assertTrue;

import static javax.baja.fox.FoxConnectionTypeEnum.FOX;

import java.io.IOException;
import java.util.Collections;
import java.util.Set;

import javax.baja.alarm.BAlarmService;
import javax.baja.app.BAppContainer;
import javax.baja.category.BCategoryService;
import javax.baja.driver.BDriverContainer;
import javax.baja.fox.BFoxProxySession;
import javax.baja.fox.FoxConnectionTypeEnum;
import javax.baja.naming.BLocalHost;
import javax.baja.naming.BOrd;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.role.BRole;
import javax.baja.role.BRoleService;
import javax.baja.search.BSearchService;
import javax.baja.security.BAes256Pbkdf2HmacSha256PasswordEncoder;
import javax.baja.security.BCertificateAliasCredential;
import javax.baja.security.BIPreconnectCredentials;
import javax.baja.security.BIUserCredentials;
import javax.baja.security.BPassword;
import javax.baja.security.BPasswordAuthenticator;
import javax.baja.security.BPermissions;
import javax.baja.security.BPermissionsMap;
import javax.baja.security.BUsernameAndPassword;
import javax.baja.sys.BComponent;
import javax.baja.sys.BStation;
import javax.baja.sys.BString;
import javax.baja.sys.Context;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.test.permissions.PermissionsScenario;
import javax.baja.user.BUser;
import javax.baja.user.BUserService;
import javax.baja.util.BServiceContainer;
import javax.baja.web.BWebServer;
import javax.baja.web.BWebService;

import org.testng.annotations.AfterTest;
import org.testng.annotations.BeforeTest;

import com.tridium.authn.BAuthenticationService;
import com.tridium.crypto.core.cert.CertUtils;
import com.tridium.fox.sys.BFoxService;
import com.tridium.fox.sys.BFoxSession;
import com.tridium.fox.sys.BFoxwssScheme;
import com.tridium.jetty.BJettyWebServer;
import com.tridium.sys.Nre;
import com.tridium.sys.station.Station;
import com.tridium.testng.TestAuthenticationClient;
import com.tridium.testng.TestCertAuthenticationClient;
import com.tridium.testng.TestUtil;

/**
 * Classes can extend from BTestNgStation to implement tests
 * that execute in a running station. History warmup and the web
 * and search services are disabled by default to speed up station start.
 * See {@link #configureTestStation(BStation, String, int, int)}
 * to enable history warmup. See {@link #isWebServiceEnabled()} and
 * {@link #isSearchServiceEnabled()} to enable the web and search service.
 *
 * @author Tim Urenda on Mar 7 2023
 * @since Niagara 4.14
 */
@NiagaraType
public abstract class BTestNgStation
  extends BTestNg
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.test.BTestNgStation(2979906276)1.0$ @*/
/* Generated Tue Mar 07 14:35:37 EST 2023 by Slot-o-Matic (c) Tridium, Inc. 2012-2023 */

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BTestNgStation.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

//region setup/teardown

  /**
   * Create the test station for this test class.
   *
   * @throws Exception
   */
  @BeforeTest(alwaysRun = true, description = "Setup and start test station")
  public void setupStation()
    throws Exception
  {
    cleanStationListeners();

    // Create basic station
    if (stationHandler == null)
    {
      System.out.println("Creating test station for " + getClass());
      makeStationHandler();
      System.out.println("Done creating station for " + getClass());
    }

    BStation station = stationHandler.getStation();

    // Override this method for custom station configuration if needed
    try
    {
      configureTestStation(station, testStationName, webPort, foxPort);
    }
    catch (Exception e)
    {
      // If the configuration fails, we need to ensure that the half-created
      // station is released so that subsequent tests can still run
      stationHandler.stopStation();
      stationHandler.releaseStation();
      stationHandler = null;
      System.out.println("Unable to set up station for " + getClass());
      throw new Exception("Unable to set up station for " + getClass(), e);
    }

    // Start the station
    startStation(station);
    Thread.sleep(500);
    Station.atSteadyState = true;
    Nre.getEngineManager().atSteadyState(station);
  }

  /**
   * Tear down this test class's test station.
   *
   * @throws Exception
   */
  @AfterTest(alwaysRun = true, description = "Teardown test station")
  public void teardownStation()
    throws Exception
  {
    if (stationHandler != null)
    {
      stationHandler.stopStation();

      // Wait until the jetty server stops since this occurs asynchronously. We do not want to
      // proceed to the next test until the webserver has stopped. See BWebServer stopWebServer()
      // function for transitions.
      if (webService != null &&
        webService.getWebServer() != null)
      {
        String[] serverState = { null };
        TestUtil.assertWillBeTrue(
          () -> {
            try
            {
              serverState[0] = webService.getWebServer().getServerState();
              return serverState[0].equals("stopped") ||
                serverState[0].equals("failed");
            }
            catch (Exception ignored)
            {
              // Any exception retrieving the server probably means it shut down, ignore any errors
              return true;
            }
          },
          () -> "Failed to shutdown webserver in a timely fashion, state is '" + serverState[0] + '\''
        );
      }

      //TODO: It would be marvelous if the Station class could understand if no tests were run and print
      //      a warning here indicating that the Station was started for no reason, i.e. automate NCCB-51960

      stationHandler.releaseStation();
      stationHandler = null;
      cleanStationListeners();
      Station.station = null;
      Station.stationStarted = false;
      Station.atSteadyState = false;
    }
  }

  /**
   * Start a station that has a Niagara Network and make sure both station and the Fox service
   * are running prior to return.
   *
   * @throws Exception
   */
  protected void startStation(BStation station)
    throws Exception
  {
    BFoxService stationFoxService = station.getChildren(BServiceContainer.class)[0]
      .getChildren(BFoxService.class)[0];

    if (!station.isRunning())
    {
      Nre.clearPlatform();
      Nre.loadPlatform();
      Nre.getServiceManager().startAllServices();
      station.start();
      Station.stationStarted = true;

      Nre.getEngineManager().stationStarted(station);
    }

    int maxAttempts = 10;
    while (maxAttempts-- >= 0 && !station.isRunning())
    {
      Thread.sleep(100);
    }
    if (!station.isRunning())
    {
      throw new Exception("Station did not start in time");
    }

    // Increased attempts from 10 to 50 to prevent intermittent CI test failures
    maxAttempts = 50;
    while (maxAttempts-- >= 0 && !stationFoxService.isServing())
    {
      Thread.sleep(100);
    }
    if (!stationFoxService.isServing())
    {
      throw new Exception("Fox service did not start in time");
    }
  }

  /**
   * Create an empty test station that is later configured by
   * {@link BTestNgStation#configureTestStation(BStation, String, int, int)}. Can be overridden by subclasses to
   * call a different version of createTestStation that, for example, takes a file ord to an xml
   * definition of the station (see {@link TestStationHandler#createTestStation(BOrd)}) or
   * {@link TestStationHandler#createTestStation(String)}.
   * For example:
   * <pre><code>
   *  protected void makeStationHandler() throws Exception
   *  {
   *    stationHandler = createTestStation(BOrd.make("module://myModule/rc/myTestStation.xml"));
   *  }
   * </code></pre>
   * <p>
   * Note: If the supplied station definition contains services configured in configureTestStation,
   * that method should probably also be overridden to avoid duplicate slot exceptions.
   * </p>
   */
  protected void makeStationHandler()
    throws Exception
  {
    stationHandler = createTestStation();
  }

  private void cleanStationListeners()
  {
    for (Station.SaveListener listener : Station.getSaveListeners())
    {
      Station.removeSaveListener(listener);
    }

    for (Station.RemoteListener listener : Station.getRemoteListeners())
    {
      Station.removeRemoteListener(listener);
    }
  }

//endregion setup/teardown

  /**
   * Connect to the local (in-process) test station via Fox using the specified user credentials.
   *
   * @return Fox proxy session
   * @throws Exception
   */
  protected BFoxProxySession connect(String userName, String password)
    throws Exception
  {
    return connect(userName, password, MAX_FOX_CONNECTION_ATTEMPTS);
  }

  /**
   * Connect to the local (in-process) test station via Fox using the specified user credentials
   * with the specified number of attempts.
   *
   * @return Fox proxy session
   * @throws Exception
   */
  protected BFoxProxySession connect(String userName, String password, int maxAttempts)
    throws Exception
  {
    return connect(userName, password, foxPort, FOX, BFoxwssScheme.DEFAULT_PORT, maxAttempts);
  }

  /**
   * Connect to the local (in-process) test station via the given Fox connection type using the
   * specified user credentials, fox/foxs port, and webSocket port with the specified number
   * of attempts.
   *
   * @return The connected {@link BFoxProxySession} instance
   * @throws Exception If there were any problems while establishing and connecting a
   * BFoxProxySession instance with the given parameters.
   *
   * @since Niagara 4.15
   */
  protected BFoxProxySession connect(String userName, String password, int port, FoxConnectionTypeEnum connectionType, int webSocketPort, int maxAttempts)
    throws Exception
  {
    return connect(new BUsernameAndPassword(userName, password), port, connectionType, webSocketPort, maxAttempts);
  }

  /**
   * Connect to the local (in-process) test station via the given Fox connection type using the
   * specified authentication client, fox/foxs port, and webSocket port with the specified number
   * of attempts.
   *
   * @return The connected {@link BFoxProxySession} instance
   * @throws Exception If there were any problems while establishing and connecting a
   * BFoxProxySession instance with the given parameters.
   *
   * @since Niagara 4.15
   */
  protected BFoxProxySession connect(BIUserCredentials credentials, int port, FoxConnectionTypeEnum connectionType, int webSocketPort, int maxAttempts)
    throws Exception
  {
    BFoxSession foxSession =
      BFoxSession.make(null, BLocalHost.INSTANCE, port, connectionType, webSocketPort);
    if (credentials instanceof BUsernameAndPassword)
    {
      BUsernameAndPassword usernameAndPassword = (BUsernameAndPassword) credentials;
      foxSession.getConnection().setAuthenticationClient(
        new TestAuthenticationClient(usernameAndPassword.getUsername(),
          usernameAndPassword.getPassword().getValue()));
    }
    else if (credentials instanceof BCertificateAliasCredential)
    {
      BCertificateAliasCredential certificateAliasCredential = (BCertificateAliasCredential) credentials;
      foxSession.getConnection().setAuthenticationClient(
        new TestCertAuthenticationClient(certificateAliasCredential.getUsername(),
          certificateAliasCredential.getCertificateAlias(),
          certificateAliasCredential.getCertificatePassword()));
    }
    else
    {
      foxSession.setCredentials(credentials);
      foxSession.getConnection().setCredentials(credentials);
      if (credentials instanceof BIPreconnectCredentials)
      {
        foxSession.getConnection().setPreconnectCredentials((BIPreconnectCredentials)credentials);
      }
    }

    // Attempt the FOX client connection, and if there are any failures, retry
    // up to the maximum number of connection attempts allowed
    Exception e = null;
    int attempts = 0;
    while (!foxSession.getConnection().isConnected() && attempts < maxAttempts)
    {
      try
      {
        ++attempts;
        foxSession = BFoxSession.connect(foxSession);
        e = null;
      }
      catch (Exception ex)
      {
        // Stash the exception to be thrown later if all connection attempts fail
        e = ex;
      }

      if (!foxSession.getConnection().isConnected() && attempts < maxAttempts)
      {
        // Give some time for the server side to settle out before attempting
        // the next FOX client connection retry
        Thread.sleep(Long.getLong("SERVER_STARTUP_SLEEP", DEFAULT_FOX_CONNECTION_REATTEMPT_DELAY));
      }
    }

    if (!foxSession.getConnection().isConnected() && e != null)
    {
      throw e;
    }

    assertTrue(
      foxSession.getConnection().isConnected(),
      "failed to connect fox client connection (" + foxSession.getLastFailureCause() + ")"
    );

    return foxSession;
  }

//region configuration

  /**
   * Configures a standard test station that provides at least a Niagara Network.
   * The default implementation here also (for convenience) provides several other
   * common services. However, subclasses can override this behavior if desired.
   * <p>
   * It is common to override this method to include additional setup on the default test station.
   * To do this, the override method would first call this method, then add the needed elements
   * to the test station.
   * For example:
   * <pre><code>
   *  protected void configureTestStation(BStation station, String stationName, int webPort, int foxPort)
   *    throws Exception
   *  {
   *    super.configureTestStation(station, stationName, webPort, foxPort);
   *    // Add a test NumericWritable
   *    numericWritable = new BNumericWritable();
   *    station.add("testPoint", numericWritable);
   *  }
   * </code></pre>
   * <p>
   * If history warmup is required for test methods, set enableHistoryWarmup = true
   * before calling configureTestStation(...):
   * <pre><code>
   *  protected void configureTestStation(BStation station, String stationName, int webPort, int foxPort)
   *    throws Exception
   *  {
   *    enableHistoryWarmup = true;
   *    super.configureTestStation(station, stationName, webPort, foxPort);
   *  }
   * </code></pre>
   *
   * @param station     The test station instance
   * @param stationName The test station name
   * @param webPort     The port number for the web service
   * @param foxPort     The port number for the Fox service
   * @throws Exception
   */
  protected void configureTestStation(BStation station, String stationName, int webPort, int foxPort)
    throws Exception
  {
    // Create station db with two containers
    BComponent drivers = addOrSetContainer(station, DRIVERS, BDriverContainer.TYPE);
    addOrSetContainer(station, APPS, BAppContainer.TYPE);

    // Add non-driver services
    BComponent services = station.getServices();

    services.add(ROLE_SERVICE, makeRoleService());
    services.add(USER_SERVICE, makeUserService());

    // Add the minimum services
    services.add(ALARM_SERVICE, newInstance("alarm:AlarmService"));
    services.add(HISTORY_SERVICE, newInstance("history:HistoryService"));
    services.add(JOB_SERVICE, newInstance("baja:JobService"));
    services.add(CATEGORY_SERVICE, newInstance("baja:CategoryService"));
    services.add(BOX_SERVICE, newInstance("box:BoxService"));
    if (makeDefaultAuthFoxServices)
    {
      services.add(AUTH_SERVICE, makeDefaultAuthService());
      services.add(FOX_SERVICE, makeDefaultFoxService(foxPort));
    }

    if (isSearchServiceEnabled())
    {
      services.add(SEARCH_SERVICE, newInstance("search:SearchService"));
    }

    if (isWebServiceEnabled())
    {
      webService = makeWebService(webPort);
      services.add(WEB_SERVICE, webService);
    }

    // Add drivers
    if (drivers.get(NIAGARA_NETWORK) == null)
    {
      drivers.add(NIAGARA_NETWORK, newInstance("niagaraDriver:NiagaraNetwork"));
    }

    // Establish the history warmup property
    // Users should set enableHistoryWarmup = true before calling configureTestStation(...)
    // if their tests require history warmup to be active.
    System.setProperty("niagara.history.warmup", String.valueOf(enableHistoryWarmup));

    // Name the station
    station.setStationName(stationName);
  }

  /**
   * Create an instance of a component based on a given type.
   * The type string format is {@code module:type}, for example:
   * {@code newInstance("control:NumericPoint");}
   *
   * @param type The string representation of a type
   * @return A new component instance.
   * @throws Exception
   */
  public BComponent newInstance(String type)
    throws Exception
  {
    return (BComponent)Sys.getType(type).getInstance();
  }

  /**
   * Create a role service with roles for default test users.
   * When overriding this method, call {@link #addRole(BRoleService, String, BPermissionsMap)}
   * to add custom roles to the new role service.
   *
   * @return BRoleService with 'superUser', 'TestAdmin, 'TestOperator'.
   * @throws Exception
   */
  public BComponent makeRoleService()
    throws Exception
  {
    BRoleService roleService = new BRoleService();

    addRole(roleService, TEST_SUPER_USER, BPermissionsMap.SUPER_USER);
    addRole(roleService, TEST_ADMIN_USER, map(BPermissions.make("rwiRWI")));
    addRole(roleService, TEST_OPERATOR_USER, map(BPermissions.make("rwi")));

    return roleService;
  }

  /**
   * Create a user service with test users of various privileges.
   * When overriding this method, call {@link #addUser(BUserService, String)}
   * or {@link #addUser(BUserService, String, String)} to add custom users
   * to the new user service.
   *
   * @return BUserService with 'superUser', 'TestAdmin, 'TestOperator'.
   * All passwords are set to 'Test@1234_5678'.
   * @throws Exception
   */
  public BComponent makeUserService()
    throws Exception
  {
    // Create service
    BUserService userService = new BUserService();

    addUser(userService, TEST_SUPER_USER);
    addUser(userService, TEST_ADMIN_USER);
    addUser(userService, TEST_OPERATOR_USER);

    return userService;
  }

  private static BPermissionsMap map(BPermissions permissions)
  {
    // The null likely serves to offset the index mismatch between zero-based java arrays and
    // one-based category indexes. Yes, you can create a category with a "zero" index but
    // the BPermissionsMap only encodes from "i=1" so using one is probably not a good idea.
    return BPermissionsMap.make(new BPermissions[]{ null, permissions });
  }

  /**
   * Create a permissions map populated with the provided permissions.
   *
   * @param permissions A varargs set of permissions
   * @return A permissions map
   */
  protected static BPermissionsMap map(BPermissions... permissions)
  {
    BPermissions[] mappedPermissions = new BPermissions[permissions.length + 1];
    mappedPermissions[0] = null;
    System.arraycopy(permissions, 0, mappedPermissions, 1, permissions.length);
    return BPermissionsMap.make(mappedPermissions);
  }

  /**
   * Create a permissions map populated with the provided permission strings
   * representing read, write, and invoke permissions. For example:
   * {@code map("rwiRWI");}
   *
   * @param permissions A varargs set of permission strings
   * @return A permissions map
   */
  protected static BPermissionsMap map(String... permissions)
  {
    BPermissions[] mappedPermissions = new BPermissions[permissions.length + 1];
    mappedPermissions[0] = null;
    for (int i = 0; i < permissions.length; ++i)
    {
      BPermissions permissionsValue = BPermissions.none;
      try
      {
        permissionsValue = BPermissions.make(permissions[i]);
      }
      catch (IOException ignore)
      {
      }
      mappedPermissions[i + 1] = permissionsValue;
    }
    return BPermissionsMap.make(mappedPermissions);
  }

  // Create an authentication service with a Digest, LegacyDigest, and Basic authentication scheme.
  private BAuthenticationService makeDefaultAuthService()
  {
    BAuthenticationService authService = new BAuthenticationService();
    authService.get("authenticationSchemes");
    return authService;
  }

  /**
   * Create a web service.
   *
   * @param port The port number to use for the web HTTP service (not HTTPS)
   * @return A web service instance
   * @throws Exception
   */
  protected BWebService makeWebService(int port)
    throws Exception
  {
    BWebService service = new BWebService();

    service.set("httpsCert", BString.make(CertUtils.FACTORY_CERT_ALIAS));
    service.getMainCertAliasAndPassword().resetAliasAndPassword();
    service.getHttpPort().setPublicServerPort(port);

    BWebServer server = new BJettyWebServer();
    service.add("JettyWebServer", server);

    return service;
  }

  private BComponent makeDefaultFoxService(int foxPort)
    throws Exception
  {
    BFoxService foxSvc = new BFoxService();

    foxSvc.getFoxPort().setPublicServerPort(foxPort);

    return foxSvc;
  }

  private static BComponent addOrSetContainer(BStation station, String name, Type type)
  {
    BComponent container = (BComponent)station.get(name);

    if (container == null)
    {
      // container is not present already
      container = (BComponent)type.getInstance();
      station.add(name, container);
    }
    else if (!container.getType().is(type))
    {
      // container that is present is not the correct type; replace it
      container = (BComponent)type.getInstance();
      station.set(name, container, null);
    }

    return container;
  }

  /**
   * Add a user to the given user service with the given name and a default test password.
   * This method can be called when overriding the makeUserService() method to create custom users.
   *
   * @param userService The user service
   * @param name        The name of the user
   */
  protected void addUser(BUserService userService, String name)
  {
    addUser(userService, name, TEST_PASSWORD);
  }

  /**
   * Add a user to the given user service with the given name and password.
   * This method can be called when overriding the makeUserService() method to create custom users.
   *
   * @param userService The user service
   * @param name        The name of the user
   * @param password    The password for the user
   */
  protected void addUser(BUserService userService, String name, String password)
  {
    userService.add(name, makeUser(Collections.singleton(name), password));
  }

  /**
   * Create a user with the given username and role.
   * The new user is added to the station's existing user service.
   *
   * @param userName The name of the user
   * @param role     The role for this user
   * @return A new user instance
   */
  protected BUser addUser(String userName, String role)
  {
    return addUser(userName, Collections.singleton(role), TEST_PASSWORD);
  }

  /**
   * Create a user with the given username and roles.
   * The new user is added to the station's existing user service.
   *
   * @param userName  The name of the user
   * @param roleNames The set of roles for this user
   * @return A new user instance
   */
  protected BUser addUser(String userName, Set<String> roleNames)
  {
    return addUser(userName, roleNames, TEST_PASSWORD);
  }

  /**
   * Create a user with the given username and password with the given role.
   * The new user is added to the station's existing user service.
   *
   * @param userName The name of the user
   * @param role     The role for this user
   * @param password The password for the user
   * @return A new user instance
   */
  protected BUser addUser(String userName, String role, String password)
  {
    return addUser(userName, Collections.singleton(role), password);
  }

  /**
   * Create a user with the given username and password with the given roles.
   * The new user is added to the station's existing user service.
   *
   * @param userName  The name of the user
   * @param roleNames The set of roles for this user
   * @param password  The password for the user
   * @return A new user instance
   */
  protected BUser addUser(String userName, Set<String> roleNames, String password)
  {
    BUser user = makeUser(roleNames, password);
    getUserService().add(userName, user);
    return user;
  }

  private static BUser makeUser(Set<String> roleNames, String password)
  {
    BUser user = new BUser();
    roleNames.forEach(roleName -> user.addRole(roleName, Context.skipValidate));
    BPassword encodedPassword = BPassword.make(password, BAes256Pbkdf2HmacSha256PasswordEncoder.ENCODING_TYPE);
    BPasswordAuthenticator pAuth = new BPasswordAuthenticator(encodedPassword);
    user.setAuthenticator(pAuth);
    return user;
  }

  /**
   * Add a role to the given role service with the given permissions.
   * This method can be called when overriding the makeRoleService() method to create custom roles.
   *
   * @param roleService The role service
   * @param name        The name of the new role
   * @param permissions The permissions for the new role
   */
  protected void addRole(BRoleService roleService, String name, BPermissionsMap permissions)
  {
    BRole role = new BRole();
    role.setPermissions(permissions);
    roleService.add(name, role);
  }

  /**
   * Create a role with the given permissions.
   * The new role is added to the station's existing role service.
   *
   * @param name           The name of the role
   * @param permissionsMap The permissions to apply to the role
   * @return A new role instance
   */
  protected BRole addRole(String name, BPermissionsMap permissionsMap)
  {
    BRole role = new BRole();
    role.setPermissions(permissionsMap);
    getRoleService().add(name, role);
    return role;
  }

  /**
   * <p>
   * Conducts a scenario specifically for testing permissions. Will create a test user and role
   * with the permissions as configured given the setup steps, and clean up afterwards.
   * </p>
   * <p>
   * You can set up tests by assigning ORDs or Components to specified categories:
   * </p>
   * <pre><code>
   * import static javax.baja.test.permissions.PermissionsScenario.withOrd;
   * import static javax.baja.test.permissions.PermissionsScenario.withPermissions;
   * //...
   * userPermissionsScenario(
   *   withPermissions("R").onCategory(5),
   *   withOrd("station:|slot:/MyComponent").onlyInCategory(5)
   * )
   *   .actAsUser((user) -&lt; {
   *     OrdTarget target = BOrd.make("station:|slot:/MyComponent").resolve(stationHandler.getStation(), user);
   *     assertTrue(target.canRead());
   *   });
   * </code></pre>
   * <p>
   * Or, if the actual permissions are the only interesting thing about the test scenario, you
   * can specify permissions only. A "throwaway" category will be created just for this scenario
   * and cleaned up afterwards.
   * </p>
   * <pre><code>
   * import static javax.baja.test.permissions.PermissionsScenario.withPermissions;
   * //...
   * userPermissionsScenario(withPermissions("R").onOrd("station:|slot:/Services"))
   *   .actAsUser((user) -&lt; {
   *     OrdTarget target = BOrd.make("station:|slot:/Services").resolve(stationHandler.getStation(), user);
   *     assertFalse(target.canWrite());
   *   });
   * </code></pre>
   *
   * @param setupSteps permissions setup steps
   * @return a permissions test scenario, ready to run with its own test user
   */
  protected PermissionsScenario userPermissionsScenario(PermissionsScenario.SetupStep... setupSteps)
  {
    return new PermissionsScenario(station, setupSteps);
  }

  /**
   * Make WebService opt-in to improve Station startup times and reduce noise.
   * Override this method and return true if you need access to a running web service.
   *
   * @return true if the web service is to be initialized
   */
  protected boolean isWebServiceEnabled()
  {
    return false;
  }

  /**
   * Make SearchService opt-in since it prints a warning each time it is started.
   * Override this method and return true if you need access to a running search service.
   *
   * @return true if the search service is to be initialized
   */
  protected boolean isSearchServiceEnabled()
  {
    return false;
  }

//endregion configuration

//region access

  /**
   * Get the test superuser.
   *
   * @return The test superuser
   */
  protected BUser getTestSuperUser()
  {
    return getUserService().getUser(TEST_SUPER_USER);
  }

  /**
   * Get the test admin user.
   *
   * @return The test admin user
   */
  protected BUser getTestAdminUser()
  {
    return getUserService().getUser(TEST_ADMIN_USER);
  }

  /**
   * Get the test operator user.
   *
   * @return The test operator user
   */
  protected BUser getTestOperatorUser()
  {
    return getUserService().getUser(TEST_OPERATOR_USER);
  }

  /**
   * Get the name of the test superuser.
   *
   * @return The test superuser name
   */
  protected String getSuperUsername()
  {
    return TEST_SUPER_USER;
  }

  /**
   * Get the password string of the test superuser
   *
   * @return The test superuser password string
   */
  protected String getSuperUserPassword()
  {
    return TEST_PASSWORD;
  }

  /**
   * Get the driver container from the test station.
   *
   * @return The driver container
   */
  protected BDriverContainer getDrivers()
  {
    return (BDriverContainer)stationHandler.getStation().get(DRIVERS);
  }

  /**
   * Get the service container from the test station.
   *
   * @return The service container
   */
  protected BServiceContainer getServices()
  {
    return stationHandler.getStation().getServices();
  }

  /**
   * Get the role service from the test station.
   *
   * @return The role service
   */
  protected BRoleService getRoleService()
  {
    return (BRoleService)getServices().get(ROLE_SERVICE);
  }

  /**
   * Get the role service from the test station.
   *
   * @return The role service
   */
  protected BUserService getUserService()
  {
    return (BUserService)getServices().get(USER_SERVICE);
  }

  /**
   * Get the role service from the test station.
   *
   * @return The role service
   */
  protected BWebService getWebService()
  {
    if (isWebServiceEnabled())
    {
      return (BWebService)getServices().get(WEB_SERVICE);
    }

    return null;
  }

  /**
   * Get the search service from the test station.
   *
   * @return The search service
   */
  protected BSearchService getSearchService()
  {
    if (isSearchServiceEnabled())
    {
      return (BSearchService)getServices().get(SEARCH_SERVICE);
    }

    return null;
  }

  /**
   * Get the category service from the test station.
   *
   * @return The category service
   */
  protected BCategoryService getCategoryService()
  {
    return (BCategoryService)getServices().get(CATEGORY_SERVICE);
  }

  /**
   * Get the alarm service from the test station.
   *
   * @return The alarm service
   */
  protected BAlarmService getAlarmService()
  {
    return (BAlarmService)getServices().get(ALARM_SERVICE);
  }

  /**
   * Get the Fox service component from the test station.
   *
   * @return The Fox service component
   */
  protected BComponent getFoxService()
  {
    return (BComponent)getServices().get(FOX_SERVICE);
  }

  /**
   * Get the authentication service component from the test station.
   *
   * @return The authentication service component
   */
  protected BComponent getAuthService()
  {
    return (BComponent)getServices().get(AUTH_SERVICE);
  }

  /**
   * Get the Niagara network component from the test station.
   *
   * @return The Niagara network component
   */
  protected BComponent getNiagaraNetwork()
  {
    return (BComponent)getDrivers().get(NIAGARA_NETWORK);
  }

//endregion access

//region variables

  protected static final String TEST_SUPER_USER = "TestSuper";
  protected static final String TEST_ADMIN_USER = "TestAdmin";
  protected static final String TEST_OPERATOR_USER = "TestOperator";
  protected static final String TEST_PASSWORD = "Test@1234_567890!";
  protected static final String testStationName = "test";

  protected boolean enableHistoryWarmup = false;
  protected boolean makeDefaultAuthFoxServices = true;

  protected static final String USER_SERVICE = "UserService";
  protected static final String ROLE_SERVICE = "RoleService";
  protected static final String AUTH_SERVICE = "AuthenticationService";
  protected static final String CATEGORY_SERVICE = "CategoryService";
  protected static final String ALARM_SERVICE = "AlarmService";
  protected static final String WEB_SERVICE = "WebService";
  protected static final String SEARCH_SERVICE = "SearchService";
  protected static final String HISTORY_SERVICE = "History";
  protected static final String JOB_SERVICE = "Job";
  protected static final String BOX_SERVICE = "BoxService";
  protected static final String FOX_SERVICE = "FoxService";
  protected static final String DRIVERS = "Drivers";
  protected static final String NIAGARA_NETWORK = "NiagaraNetwork";
  protected static final String APPS = "Apps";

  private BWebService webService = null;

  protected static final int MAX_FOX_CONNECTION_ATTEMPTS = 3;
  protected static final long DEFAULT_FOX_CONNECTION_REATTEMPT_DELAY = 1000L;

  protected TestStationHandler stationHandler;
  protected int foxPort = 1911;
  protected int webPort = 9090;

//endregion variables
}
