/*
 * Copyright 2024 Tridium, Inc. All Rights Reserved.
 */

package com.tridium.testng;

import java.util.Optional;

import javax.baja.security.AuthenticationRealm;
import javax.baja.security.BCertificateAliasCredential;
import javax.baja.security.BICredentials;
import javax.baja.security.BIPreconnectCredentials;
import javax.baja.security.BIUserCredentials;
import javax.baja.security.BPassword;
import javax.baja.security.BUsernameCredential;
import javax.baja.sys.BIObject;

import com.tridium.authn.AuthenticationClient;

/**
 * An Client Cert AuthenticationClient for test classes.
 *
 * @author Scott Hoye on 6/19/2024
 * @since Niagara 4.15
 */
public class TestCertAuthenticationClient
  implements AuthenticationClient
{
  public TestCertAuthenticationClient(String username, String clientCertAlias, BPassword certPassword)
  {
    this.username = username;
    this.clientCertAlias = clientCertAlias;
    this.certPassword = certPassword;
  }

  @Override
  public BIUserCredentials requestUsername(AuthenticationRealm realm)
  {
    return new BUsernameCredential(username);
  }

  @Override
  public BICredentials requestInformation(AuthenticationRealm realm, String schemeName, int step, BIObject seedInfo)
  {
    BCertificateAliasCredential certCred = new BCertificateAliasCredential();
    certCred.setUsername(username);
    certCred.setCertificateAlias(clientCertAlias);
    certCred.setCertificatePassword(certPassword);
    return certCred;
  }

  @Override
  public void setPreconnectCredentials(BIPreconnectCredentials credentials)
  {
    preconnectCredentials = credentials;
  }

  @Override
  public Optional<BIPreconnectCredentials> getPreconnectCredentials()
  {
    if (preconnectCredentials == null)
    {
      BCertificateAliasCredential credential = new BCertificateAliasCredential();
      credential.setUsername(username);
      credential.setCertificateAlias(clientCertAlias);
      credential.setCertificatePassword(certPassword);
      preconnectCredentials = credential;
    }
    return Optional.of(preconnectCredentials);
  }

  @Override
  public String toString()
  {
    String pswd = "UNKNOWN";
    try
    {
      pswd = certPassword.getValue();
    }
    catch (Throwable ignore) { }

    return super.toString() + " [username=" + username + ", clientCertAlias=" + clientCertAlias +
      ", certPassword=" + pswd + ']';
  }

  private final String username;
  private final String clientCertAlias;
  private final BPassword certPassword;
  private BIPreconnectCredentials preconnectCredentials;
}
