/*
 * Copyright 2014 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.tagdictionary;

import static com.tridium.tagdictionary.util.ImportExportConst.JSON_NAME;
import static com.tridium.tagdictionary.util.ImportExportConst.JSON_NAMESPACE;
import static com.tridium.tagdictionary.util.ImportExportConst.JSON_TYPE;
import static com.tridium.tagdictionary.util.ImportUtil.decodeType;
import static com.tridium.tagdictionary.util.TagDictionaryUtil.handleIllegalChild;

import java.util.Iterator;
import java.util.Optional;

import javax.baja.collection.SlotCursorIterator;
import javax.baja.naming.SlotPath;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.registry.TypeInfo;
import javax.baja.sys.BFacets;
import javax.baja.sys.BValue;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.IllegalNameException;
import javax.baja.sys.LocalizableRuntimeException;
import javax.baja.sys.Property;
import javax.baja.sys.SlotCursor;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.tag.Id;
import javax.baja.tag.TagInfo;

import com.tridium.json.JSONArray;
import com.tridium.json.JSONObject;
import com.tridium.json.JSONWriter;
import com.tridium.tagdictionary.util.ExportUtil;

/**
 * BTagInfoList is a list of {@code TagInfo}.
 *
 * @author John Sublett
 * @creation 2/20/14
 * @since Niagara 4.0
 */
@NiagaraType
public class BTagInfoList
  extends BInfoList
  implements Iterable<TagInfo>
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.tagdictionary.BTagInfoList(2979906276)1.0$ @*/
/* Generated Tue Jan 25 17:26:55 CST 2022 by Slot-o-Matic (c) Tridium, Inc. 2012-2022 */

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BTagInfoList.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  /**
   * Returns an iterator over the tags in this list.
   *
   * @return iterator of tags
   */
  @Override
  public Iterator<TagInfo> iterator()
  {
    return SlotCursorIterator.iterator(getProperties(), TagInfo.class);
  }

  /**
   * Test whether this list includes a tag with the specified id.
   *
   * @param id tag id to search for
   * @return {@code true} if the list contains a tag with the specified id;
   *   {@code false} otherwise.
   */
  public boolean containsTagId(Id id)
  {
    return getTag(id).isPresent();
  }

  /**
   * Get the tag for the specified id.
   *
   * @param id id of the tag
   * @return an {@code Optional} that contains the {@code TagInfo} with the
   * specified id, if it exists in the list; an empty {@code Optional}
   * otherwise.
   */
  public Optional<TagInfo> getTag(Id id)
  {
    for (SlotCursor<Property> tagInfos = getProperties(); tagInfos.next(TagInfo.class);)
    {
      TagInfo tagInfo = (TagInfo)tagInfos.get();
      if (tagInfo.getTagId().equals(id))
      {
        return Optional.of(tagInfo);
      }
    }

    return Optional.empty();
  }

  /**
   * First, checks that the child is a TagInfo. Then, disallows adding child objects when under a
   * frozen {@link BTagDictionary}.
   *
   * <p>Exception: even if the dictionary is frozen, removal is allowed if the
   * parent of this list is a {@link BTagGroupInfo}.</p>
   *
   * <p>Additionally, if the property is a {@link BTagInfo} and the dictionary
   * is not frozen, the new tag id must not match an existing tag group id.</p>
   *
   * @param name name of the child object being added
   * @param value child object being added
   * @param flags {@link Flags} to be added to the child object
   * @param facets {@link BFacets} to be added to the child object
   * @param context execution context
   */
  @Override
  public void checkAdd(String name, BValue value, int flags, BFacets facets, Context context)
  {
    // Only allow TagInfos to be added to a BTagInfoList
    if (!(value instanceof TagInfo))
    {
      handleIllegalChild(this, value, context);
    }

    if (checkContext(context))
    {
      return;
    }

    if (getParent() instanceof BTagGroupInfo)
    {
      return;
    }

    if (isDictionaryFrozen())
    {
      throw new LocalizableRuntimeException("tagdictionary", "frozenDictionary.checkAdd");
    }

    BTagDictionary td = getTagDictionary();
    if (td == null)
    {
      return;
    }

    Id newId = Id.newId(td.getNamespace(), name);

    // TODO do we also want to search the tag definitions?
    // TODO what about the tags within the tag group definitions?
    BTagGroupInfoList tagGroupInfos = td.getTagGroupDefinitions();
    if(tagGroupInfos.containsTagId(newId))
    {
      throw new IllegalNameException("tagdictionary", "tagInfo.duplicateTagId", new String[] { newId.toString() });
    }
  }

  /**
   * Disallows renaming slots of a frozen {@link BTagDictionary}.
   *
   * <p>Exception: whether the dictionary is frozen is only evaluated if the
   * property is a {@link BTagInfo}.</p>
   *
   * <p>Additionally, if the property is a {@link BTagInfo} and the dictionary
   * is not frozen, the new tag id must not match an existing tag group id.</p>
   *
   * @param property slot to be renamed
   * @param newName new name
   * @param context execution context
   */
  @Override
  public void checkRename(Property property, String newName, Context context)
  {
    if (checkContext(context))
    {
      return;
    }

    BValue bValue = get(property);
    if (bValue instanceof BTagInfo)
    {
      if (isDictionaryFrozen())
      {
        throw new IllegalNameException("tagdictionary", "frozenDictionary.checkRename", new String[] {newName});
      }

      // TODO null check the BTagDictionary?
      BTagDictionary td = getTagDictionary();

      Id newId = Id.newId(td.getNamespace(), newName);

      // TODO do we also want to search the tag definitions?
      // TODO what about the tags within the tag group definitions?
      BTagGroupInfoList tagGroupInfos = td.getTagGroupDefinitions();
      if (tagGroupInfos.containsTagId(newId))
      {
        throw new IllegalNameException("tagdictionary", "tagInfo.duplicateTagId", new String[] { newId.toString() });
      }
    }
  }

  /**
   * Disallows removing slots when under a frozen {@link BTagDictionary}.
   *
   * <p>Exception: even if the dictionary is frozen, removal is allowed if the
   * parent of this list is a {@link BTagGroupInfo}.</p>
   *
   * @param property slot to be removed
   * @param context execution context
   */
  @Override
  public void checkRemove(Property property, Context context)
  {
    if (checkContext(context))
    {
      return;
    }

    if (getParent()instanceof BTagGroupInfo)
    {
      return;
    }

    if (isDictionaryFrozen())
    {
      throw new LocalizableRuntimeException("tagdictionary", "frozenDictionary.checkRemove");
    }
  }

  /**
   * If the property is a {@link BTagInfo}, call {@link BTagInfo#tagRenamed()} on it.
   *
   * @param property property being renamed
   * @param oldName old name
   * @param context execution context
   */
  @Override
  public void renamed(Property property, String oldName, Context context)
  {
    BValue value = get(property);
    if (value instanceof BTagInfo)
    {
      ((BTagInfo)value).tagRenamed();
    }
  }

  /**
   * Encode to a JSON representation
   * @since Niagara 4.14
   */
  public void encodeToJson(JSONWriter writer)
  {
    writer.array();
    for (TagInfo tag : this)
    {
      BTagInfo tagInfo = (BTagInfo) tag;
      writer.object();
      ExportUtil.encodeName(tagInfo.getName(), writer);
      if (!tagInfo.getType().equals(BSimpleTagInfo.TYPE))
      {
        writer.key(JSON_TYPE).value(tagInfo.getType());
      }
      tagInfo.encodeToJson(writer);
      writer.endObject();
    }
    writer.endArray();
  }

  /**
   * Decode a Tag Info List from a JSON representation
   *
   * @param tagsJson the JSON for the tag info list
   * @since Niagara 4.14
   */
  public void decodeFromJson(JSONArray tagsJson)
  {
    for (Object o : tagsJson)
    {
      JSONObject tagJson = (JSONObject) o;
      String tagType = tagJson.optString(JSON_TYPE, "tagdictionary:SimpleTagInfo");
      TypeInfo type = decodeType(tagType, BTagInfo.TYPE);
      BTagInfo tagInfo = (BTagInfo)type.getInstance();
      tagInfo.decodeFromJson(tagJson);
      // NAME is unescaped when generating TagInfo json
      String namespace = tagJson.optString(JSON_NAMESPACE, "");
      if (!namespace.isEmpty())
      {
        add(SlotPath.escape(namespace + ':' + tagJson.getString(JSON_NAME)), tagInfo);
      }
      else
      {
        add(SlotPath.escape(tagJson.getString(JSON_NAME)), tagInfo);
      }
    }
  }
}
