/*
 * Copyright 2003, Tridium, Inc. All Rights Reserved.
 */
package javax.baja.rdb.history;

import static com.tridium.rdb.history.RdbmsHistoryUtil.fixQuotes;
import static com.tridium.rdb.history.RdbmsHistoryUtil.isNewSchema;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.baja.driver.history.BHistoryExport;
import javax.baja.driver.util.BDescriptorState;
import javax.baja.history.BHistoryConfig;
import javax.baja.history.BHistoryId;
import javax.baja.history.BHistoryRecord;
import javax.baja.history.BHistoryService;
import javax.baja.history.BIHistory;
import javax.baja.history.BTrendRecord;
import javax.baja.history.HistoryNotFoundException;
import javax.baja.history.HistoryQuery;
import javax.baja.history.HistorySpaceConnection;
import javax.baja.history.db.BHistoryDatabase;
import javax.baja.naming.SlotPath;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.nre.util.Array;
import javax.baja.nre.util.TextUtil;
import javax.baja.rdb.BRdbms;
import javax.baja.rdb.BRdbmsTimestampStorage;
import javax.baja.security.BPassword;
import javax.baja.sys.Action;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BDouble;
import javax.baja.sys.BFacets;
import javax.baja.sys.BFloat;
import javax.baja.sys.BLong;
import javax.baja.sys.BObject;
import javax.baja.sys.BString;
import javax.baja.sys.BValue;
import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.Context;
import javax.baja.sys.Cursor;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.SlotCursor;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.timezone.BTimeZone;
import javax.baja.util.IFuture;
import javax.baja.util.Invocation;

import com.tridium.nre.util.tuple.Pair;
import com.tridium.rdb.BRdbmsDeprecatedDialect;
import com.tridium.rdb.history.RdbmsHistoryUtil;
import com.tridium.rdb.jdbc.RdbmsDialect;
import com.tridium.rdb.jdbc.trans.BSqlType;

//NOTE: Re-slotting this code will result in a redundant BBoolean cast of 'exportInvalidValues' as defined by
//      NCCB-38318. Please be sure to manually correct the cast prior to commitment to avoid a compile warning.

/**
 * BRdbmsHistoryExport defines an archive action for transferring
 * one or more histories from the local source to a relational database.
 *
 * @author Mike Jarmy
 * @version $Revision: 28$ $Date: 1/12/11 9:20:53 AM EST$
 * @creation 24 Jul 03
 * @since Baja 1.0
 */
@NiagaraType
/*
 The user name that is used to login to the database.
 If defined, this will be used instead of the loginName
 that is defined on the relational database.
 */
@NiagaraProperty(
  name = "userName",
  type = "String",
  defaultValue = ""
)
/*
 The password that is used to login to the database.
 If defined, this will be used instead of the loginPassword
 that is defined on the relational database.
 */
@NiagaraProperty(
  name = "password",
  type = "BPassword",
  defaultValue = "BPassword.DEFAULT"
)
/*
 Represents the maximum timestamp value of any exported record.
 If this value is null, then all the history records will be exported.
 Otherwise, only those history records with a timestamp newer than this
 value will be exported.
 After a successful export, this value is set to the maximum
 timestamp value of any of the records that were exported.
 */
@NiagaraProperty(
  name = "lastTimestamp",
  type = "BAbsTime",
  defaultValue = "BAbsTime.NULL",
  flags = Flags.READONLY | Flags.DEFAULT_ON_CLONE
)
/*
 A flag that allows skipping of nan and +/- inf values from being exported
 in most cases if an infinite value is exported the database will store the max or min possible value
 if nan is exported it will go into the database as a null
 */
@NiagaraProperty(
  name = "exportInvalidValues",
  type = "boolean",
  defaultValue = "true"
)
public abstract class BRdbmsHistoryExport
  extends BHistoryExport
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.rdb.history.BRdbmsHistoryExport(4236025402)1.0$ @*/
/* Generated Mon Oct 07 10:30:16 EDT 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Property "userName"

  /**
   * Slot for the {@code userName} property.
   * The user name that is used to login to the database.
   * If defined, this will be used instead of the loginName
   * that is defined on the relational database.
   * @see #getUserName
   * @see #setUserName
   */
  @Generated
  public static final Property userName = newProperty(0, "", null);

  /**
   * Get the {@code userName} property.
   * The user name that is used to login to the database.
   * If defined, this will be used instead of the loginName
   * that is defined on the relational database.
   * @see #userName
   */
  @Generated
  public String getUserName() { return getString(userName); }

  /**
   * Set the {@code userName} property.
   * The user name that is used to login to the database.
   * If defined, this will be used instead of the loginName
   * that is defined on the relational database.
   * @see #userName
   */
  @Generated
  public void setUserName(String v) { setString(userName, v, null); }

  //endregion Property "userName"

  //region Property "password"

  /**
   * Slot for the {@code password} property.
   * The password that is used to login to the database.
   * If defined, this will be used instead of the loginPassword
   * that is defined on the relational database.
   * @see #getPassword
   * @see #setPassword
   */
  @Generated
  public static final Property password = newProperty(0, BPassword.DEFAULT, null);

  /**
   * Get the {@code password} property.
   * The password that is used to login to the database.
   * If defined, this will be used instead of the loginPassword
   * that is defined on the relational database.
   * @see #password
   */
  @Generated
  public BPassword getPassword() { return (BPassword)get(password); }

  /**
   * Set the {@code password} property.
   * The password that is used to login to the database.
   * If defined, this will be used instead of the loginPassword
   * that is defined on the relational database.
   * @see #password
   */
  @Generated
  public void setPassword(BPassword v) { set(password, v, null); }

  //endregion Property "password"

  //region Property "lastTimestamp"

  /**
   * Slot for the {@code lastTimestamp} property.
   * Represents the maximum timestamp value of any exported record.
   * If this value is null, then all the history records will be exported.
   * Otherwise, only those history records with a timestamp newer than this
   * value will be exported.
   * After a successful export, this value is set to the maximum
   * timestamp value of any of the records that were exported.
   * @see #getLastTimestamp
   * @see #setLastTimestamp
   */
  @Generated
  public static final Property lastTimestamp = newProperty(Flags.READONLY | Flags.DEFAULT_ON_CLONE, BAbsTime.NULL, null);

  /**
   * Get the {@code lastTimestamp} property.
   * Represents the maximum timestamp value of any exported record.
   * If this value is null, then all the history records will be exported.
   * Otherwise, only those history records with a timestamp newer than this
   * value will be exported.
   * After a successful export, this value is set to the maximum
   * timestamp value of any of the records that were exported.
   * @see #lastTimestamp
   */
  @Generated
  public BAbsTime getLastTimestamp() { return (BAbsTime)get(lastTimestamp); }

  /**
   * Set the {@code lastTimestamp} property.
   * Represents the maximum timestamp value of any exported record.
   * If this value is null, then all the history records will be exported.
   * Otherwise, only those history records with a timestamp newer than this
   * value will be exported.
   * After a successful export, this value is set to the maximum
   * timestamp value of any of the records that were exported.
   * @see #lastTimestamp
   */
  @Generated
  public void setLastTimestamp(BAbsTime v) { set(lastTimestamp, v, null); }

  //endregion Property "lastTimestamp"

  //region Property "exportInvalidValues"

  /**
   * Slot for the {@code exportInvalidValues} property.
   * A flag that allows skipping of nan and +/- inf values from being exported
   * in most cases if an infinite value is exported the database will store the max or min possible value
   * if nan is exported it will go into the database as a null
   * @see #getExportInvalidValues
   * @see #setExportInvalidValues
   */
  @Generated
  public static final Property exportInvalidValues = newProperty(0, true, null);

  /**
   * Get the {@code exportInvalidValues} property.
   * A flag that allows skipping of nan and +/- inf values from being exported
   * in most cases if an infinite value is exported the database will store the max or min possible value
   * if nan is exported it will go into the database as a null
   * @see #exportInvalidValues
   */
  @Generated
  public boolean getExportInvalidValues() { return getBoolean(exportInvalidValues); }

  /**
   * Set the {@code exportInvalidValues} property.
   * A flag that allows skipping of nan and +/- inf values from being exported
   * in most cases if an infinite value is exported the database will store the max or min possible value
   * if nan is exported it will go into the database as a null
   * @see #exportInvalidValues
   */
  @Generated
  public void setExportInvalidValues(boolean v) { setBoolean(exportInvalidValues, v, null); }

  //endregion Property "exportInvalidValues"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BRdbmsHistoryExport.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// BDescriptor
////////////////////////////////////////////////////////////////

  /**
   * Export the table to the target database.
   */
  @Override
  public void doExecute()
  {
    devicelet = (BRdbmsHistoryDeviceExt) getDeviceExt();
    db = (BRdbms) devicelet.getDevice();

    if (!db.getEnabled())
    {
      if (LOG.isLoggable(Level.FINE))
      {
        LOG.fine(
          "HistoryExport " + getHistoryId() + ": database disabled, on thread " + Thread.currentThread().getName()
        );
      }
      setState(BDescriptorState.idle);
      updateStatus();
      return;
    }

    if (LOG.isLoggable(Level.FINE))
    {
      LOG.fine("HistoryExport " + getHistoryId() + ": begin export on thread " + Thread.currentThread().getName());
    }

    long t0 = System.currentTimeMillis();
    executeInProgress();

    try
    {
      dialect = BRdbmsDeprecatedDialect.make(db);

      // If credentials have been supplied for this History Export, always use
      // them when creating a connection to the db.
      dmlConnection = AccessController.doPrivileged((PrivilegedExceptionAction<Connection>) () ->
        BRdbmsHistoryDeviceExt.getConnectionForDescriptor(db, this, /*requestPrivileged*/false, null));

      // Since BRdbms#getNonPrivilegedConnection(Connection, Context) can (but might not)
      // return a connection that is 'privileged', test to see if it is. If so, we can optimize
      // the number of connections needed for this export since the connection for
      // DML operations should also be able to perform DDL operations.
      dmlConnectionIsPrivileged = db.isPrivilegedConnection(dmlConnection);

      // get connection to remote database to be used for DML operations
      dmlStatement = dmlConnection.createStatement();

      // get local database
      BHistoryDatabase localDb =
        ((BHistoryService)Sys.getService(BHistoryService.TYPE)).getDatabase();

      // look up info about the history
      BHistoryId hid = getHistoryId();
      try (HistorySpaceConnection conn = localDb.getConnection(null))
      {
        history = conn.getHistory(hid);
      }
      if (history == null)
        throw new HistoryNotFoundException();

      config = history.getConfig();
      template = config.makeRecord();

      // use a different timezone, if necessary
      if (devicelet.getUseHistoryConfigTimeZone())
      {
        // If we are using station-side timestamps, then we are OK.
        if (devicelet.getUseLastTimestamp())
        {
          dialect.setTimeZone(config.getTimeZone());
        }
        // Else if we are using database-side timestamps, then we cannot proceed.
        // That is because we cannot reliably fetch the timezone data
        // from the database.
        else
        {
          throw new BajaRuntimeException(
            "Export failed: " +
              "useHistoryConfigTimeZone=true and " +
              "useLastTimestamp=false.");
        }
      }

      if (isExportNecessary())
      {
        // export the records
        exportRecords();
      }
      else
      {
        LOG.fine("No history records found for export.");
      }
      executeOk();
    }
    catch(Exception e)
    {
      Throwable cause = e;
      if (e instanceof PrivilegedActionException && e.getCause() != null)
      {
        cause = e.getCause();
      }

      if (LOG.isLoggable(Level.SEVERE))
      {
        LOG.log(Level.SEVERE, "History Export failed for " + devicelet.getName(), LOG.isLoggable(Level.SEVERE) ? cause : null);
      }
      executeFail(cause);
    }
    finally
    {
      // reset the internal state at the end of each execution so
      // we're not holding onto stale values on subsequent executions.
      metadataTableExists = false;
      historyMetaRecord = null;
      exportRangeStartTime = null;

      try
      {
        if (dmlStatement != null)
        {
          dmlStatement.close();
        }
      }
      catch (SQLException e)
      {
        if (LOG.isLoggable(Level.SEVERE))
        {
          LOG.log(
            Level.SEVERE,
            "HistoryExport " + getHistoryId() + ": Unable to close statement 'nonPrivilegedConnectionStatement': " + e.getMessage(),
            LOG.isLoggable(Level.FINE) ? e : null
          );
        }
      }

      try
      {
        if (dmlConnection != null)
        {
          dmlConnection.close();
        }
      }
      catch (SQLException e)
      {
        if (LOG.isLoggable(Level.SEVERE))
        {
          LOG.log(
            Level.SEVERE,
            "HistoryExport " + getHistoryId() + ": Unable to close connection 'nonPrivilegedConnection': " + e.getMessage(),
            LOG.isLoggable(Level.FINE) ? e : null
          );
        }
      }

      if (LOG.isLoggable(Level.FINE))
      {
        long ms = System.currentTimeMillis() - t0;
        LOG.fine("end export (" + ms + "ms)");
      }
    }
  }

  /**
   * postExecute
   */
  @Override
  protected IFuture postExecute(Action action, BValue arg, Context cx)
  {
    BRdbms db = (BRdbms)getDevice();
    if (db != null) db.getWorker().postAsync(new Invocation(this, action, arg, cx));

    return null;
  }

////////////////////////////////////////////////////////////////
// private
////////////////////////////////////////////////////////////////

  /**
   * Determines if an export is actually necessary. An export is deemed necessary if any of
   * the following conditions are true:
   * <pre>
   *   <ul>
   *     <li>The metadata table does not exist</li>
   *     <li>The metadata table exists, but does not contain an entry for the history being exported</li>
   *     <li>The metadata table exists and has an entry corresponding to the history being exported, and the timestamp of the corresponding history records
   *     in the 'local' history database is greater than the timestamp of the latest exported history record in the rdb.
   *     <li>An exception occurs when attempting to make a determination whether the export should proceed</li>
   *   </ul>
   * </pre>
   * <p>
   * The values obtained and used to determine whether an export needs to occur are stored in
   * instance variables so we won't have to issue duplicative queries during the actual export.
   *
   * @return true if an export would be attempted, false otherwise.
   */
  private boolean isExportNecessary()
  {
    // Start with the assumption that an export is necessary unless it can be determined that it isn't.
    boolean exportNecessary = true;
    try
    {
      // Check to see if the metadata table exists. If this is the first export
      // of this history/history type, it shouldn't exist, which means there's nothing else
      // that needs to be checked, so proceed with the export. Also, keep the result of this
      // check in the instance variable 'metadataTableExists' so we don't have to re-check
      // this in exportRecords().
      metadataTableExists = dialect.tableExists(db, dmlConnection, getMetaTableName());
      if (metadataTableExists)
      {
        // Next, check to see if this history's meta-record exists. If
        // it doesn't, proceed with the export. If it does, keep checking to see
        // if we can determine the value to use for the time query to the local
        // history database to see if there are records that need to be exported.
        Optional<Map<String, Object>> metaRecord = getMetaRecord();
        if (metaRecord.isPresent())
        {
          String historyTableName = (String)metaRecord.get().get(METARECORD_TABLE_NAME);
          exportRangeStartTime = lookupMaxTimestamp(historyTableName);
          // exportRangeStartTime could be null if either there was an exception looking up
          // the most recently exported history record (see SQLException catch block above),
          // or because lookupMaxTimestamp(String) returned null. In either case, we know we
          // will need to do the export anyway, so only skip this block in that case.
          if (exportRangeStartTime != null)
          {
            // use the timestamp accuracy to increment the value to make sure we don't reinsert anything.
            exportRangeStartTime = BAbsTime.make(exportRangeStartTime.getMillis() + dialect.getTimestampAccuracy());
            if (LOG.isLoggable(Level.FINE))
            {
              LOG.fine("HistoryExport " + getHistoryId() + ": maxTimestamp " + exportRangeStartTime.toString(SHOW_MILLIS));
            }

            BHistoryDatabase localHistoryDb =
              ((BHistoryService)Sys.getService(BHistoryService.TYPE)).getDatabase();
            // For Rdbms History Exports, we are always excluding archive history data
            // and only considering local history data via the Context used to retrieve
            // the HistorySpaceConnection
            try (HistorySpaceConnection historyConn =
                   localHistoryDb.getConnection(HistoryQuery.makeExcludeArchiveDataContext(null));
                 Cursor<BHistoryRecord> recordsCursor = historyConn.timeQuery(history, exportRangeStartTime, null).cursor())
            {
              exportNecessary = recordsCursor.next();
            }
          }
        }
      }
    }
    catch (Exception e)
    {
      // If anything went wrong trying to figure out whether the export needs to happen,
      // the answer is that the export _does_ need to happen, so just log the exception
      // and continue, letting the logic in exportRecords() handle explicitly checking
      // for and creating schema objects before pushing records into the history table.
      if (LOG.isLoggable(Level.SEVERE))
      {
        LOG.log(
          Level.SEVERE,
          "HistoryExport " + getHistoryId() + ": " + e.getMessage(),
          LOG.isLoggable(Level.FINE) ? e : null
        );
      }
    }

    return exportNecessary;
  }

  /**
   * exportRecords
   */
  private void exportRecords()
    throws Exception
  {
    String metaTableName = getMetaTableName();

    // Make sure the metaTable exists.  If it doesn't exist,
    // then apparently we have not yet done any exports at all.
    if (!metadataTableExists)
    {
      isNewSchema = true;
      if (LOG.isLoggable(Level.FINE))
      {
        LOG.fine("HistoryExport " + getHistoryId() + ": table '" + metaTableName + "' does not exist.");
      }
      createMetaTable();
    }
    else
    {
      isNewSchema = isNewSchema(dmlConnection, metaTableName);
    }

    // Fetch the meta record.
    Optional<Map<String, Object>> metaRecord = getMetaRecord();
    String tableName = null;
    Number tableId;
    boolean historyIdColumnAddedToExistingTable = false;

    if (metaRecord.isPresent())
    {
      Map<String, Object> meta = metaRecord.get();
      // If the meta record exists, just get the table name.
      tableName = (String)meta.get(METARECORD_TABLE_NAME);
      tableId = (Number)meta.get(METARECORD_ID);

      //at this point we need to update any fields in the meta record in case the user changed them on the
      //history config. For now we'll only check for updates to 'VALUEFACETS'
      if(columnExists(metaTableName, "VALUEFACETS"))
      {
        String dbValueFacets = (String)meta.get("VALUEFACETS");
        BValue valuefacets = config.get("valueFacets");

        if (valuefacets instanceof BFacets)
        {
          String configValueFacets = valuefacets.asSimple().encodeToString();

          if(dbValueFacets != null && !dbValueFacets.equals(configValueFacets))
          {
            String updateFacetsSql = dialect.makeUpdateSql(
              metaTableName,
              new Property[]{ config.getProperty("valueFacets") },
              new String[0],
              dialect.mangleIdentifier(BHistoryConfig.id.getName())
            );

            try (PreparedStatement stmt = dmlConnection.prepareStatement(updateFacetsSql))
            {
              stmt.setString(1, configValueFacets); //SET
              stmt.setString(2, config.getId().toString()); //WHERE
              stmt.execute();
            }
          }
        }
      }

      // If the table does not exist (but the metatable has a corresponding entry), create it.
      // This can happen if there was an error during export after the metadata entry was created
      // but before the table could successfully be created.
      if (!dialect.tableExists(db, dmlConnection, tableName))
      {
        if (LOG.isLoggable(Level.FINE))
        {
          LOG.fine("HistoryExport " + getHistoryId() + ": table '" + tableName + "' does not exist.");
        }
        createTable(tableName, tableId);
      }
      else if (db.getExportMode().getOrdinal() == BRdbmsHistoryExportMode.BY_HISTORY_TYPE
        && !columnExists(tableName, HISTORY_ID))
      {
        // Prior to 4.13.2, the HISTORY_ID column was only added to tables for instances of
        // BTrendRecord when exporting in BY_HISTORY_TYPE mode. If it doesn't exist,
        // create it (and it's associated composite index).
        addHistoryIdColumnAndIndexToTable(tableName, tableId);
        historyIdColumnAddedToExistingTable = true;
      }
    }
    // Otherwise, we must have never exported records for this
    // particular history (or history type) yet.
    // In that case, create a meta record. We may also
    // need to create a table.
    else
    {
      insertMetaRecord();
      metaRecord = getMetaRecord();

      if (!metaRecord.isPresent())
      {
        throw new BajaRuntimeException(
          "Could not create metadata record for history " +
            config.getId().getDeviceName() + " ::: " +
            config.getId().getHistoryName());
      }

      tableName = (String)metaRecord.get().get(METARECORD_TABLE_NAME);
      tableId = (Number)metaRecord.get().get(METARECORD_ID);

      // If we are exporting by history id, we will always
      // need to make a table.  If we are exporting by history
      // type, the table may already exist. Its simplest to
      // just check if the table exists, and create it if it doesn't.
      if (!dialect.tableExists(db, dmlConnection, tableName))
      {
        if (LOG.isLoggable(Level.FINE))
        {
          LOG.fine("HistoryExport " + getHistoryId() + ": table '" + tableName + "' does not exist.");
        }
        createTable(tableName, tableId);
      }
      else if (db.getExportMode().getOrdinal() == BRdbmsHistoryExportMode.BY_HISTORY_TYPE
        && !columnExists(tableName, HISTORY_ID))
      {
        // Prior to 4.13.2, the HISTORY_ID column was only added to tables for instances of
        // BTrendRecord when exporting in BY_HISTORY_TYPE mode. If it doesn't exist,
        // create it (and it's associated composite index).
        addHistoryIdColumnAndIndexToTable(tableName, tableId);
        historyIdColumnAddedToExistingTable = true;
      }
    }

    // insert the records
    BAbsTime oldStamp = getLastTimestamp();
    BAbsTime newStamp = insertRecords(oldStamp, tableName, historyIdColumnAddedToExistingTable);
    setLastTimestamp(newStamp);
  }

  /**
   * createTable
   */
  private void createTable(String tableName, Number tableId)
    throws SQLException
  {
    // get the template
    Property[] propTemplate = template.getPropertiesArray();

    // set up extra fields
    Array<String> extraFields = new Array<>(String.class);
    Array<Type> extraFieldTypes = new Array<>(Type.class);
    Array<BFacets> extraFacets = new Array<>(BFacets.class);

    //add export mode-specific columns
    if (db.getExportMode().getOrdinal() == BRdbmsHistoryExportMode.BY_HISTORY_TYPE)
    {
      extraFields.add(HISTORY_ID);
      extraFieldTypes.add(BString.TYPE);
      extraFacets.add(BFacets.NULL);
    }

    if (template instanceof BTrendRecord)
    {
      //add extra columns common to all trend records
      extraFields.addAll(new String[]{ "TRENDFLAGS_TAG", "STATUS_TAG" });
      extraFieldTypes.addAll(new Type[]{ BString.TYPE, BString.TYPE });
      extraFacets.addAll(new BFacets[]{ BFacets.NULL, BFacets.NULL });
    }

    /*Commented out: UTC_OFFSET column not being used
    //add UTC_OFFSET column if the database supports SQL TIMESTAMPS
    if(dialect.supportsTimestamp())
    {
      extraFields.add("UTC_OFFSET");
      extraFieldTypes.add(BString.TYPE);
      extraFacets.add(BFacets.NULL);
    }*/


    // make the table
    String sql = dialect.makeCreateTableSql(
      tableName,
      propTemplate,
      extraFields.trim(),
      extraFieldTypes.trim(),
      extraFacets.trim());

    // If 'dmlConnection' is privileged, avoid the creation of another 'privileged' connection.
    Connection privilegedConnection;
    try
    {
      privilegedConnection = AccessController.doPrivileged((PrivilegedExceptionAction<Connection>) () ->
        dmlConnectionIsPrivileged
          ? dmlConnection
          : BRdbmsHistoryDeviceExt.getConnectionForDescriptor(db, this, /*requestPrivileged*/true, null));
    }
    catch (PrivilegedActionException e)
    {
      Throwable cause = e.getCause();
      if (cause instanceof SQLException)
      {
        throw (SQLException) cause;
      }
      throw new SQLException(cause);
    }

    try (Statement privilegedConnectionStatement = privilegedConnection.createStatement())
    {
      privilegedConnectionStatement.executeUpdate(sql);

      // make the sequence if need be
      if (dialect.hasSequences())
      {
        privilegedConnectionStatement.executeUpdate("CREATE SEQUENCE " + tableName + "_Q");
      }

      if (!devicelet.getUseLastTimestamp() || devicelet.getAlwaysCreateIndexForNewTables())
      {
        Pair<String, String> indexNameAndColumns = dialect.getNewIndexNameAndColumns(String.valueOf(tableId));
        RdbmsHistoryUtil.createIndex(privilegedConnection, dialect, tableName, indexNameAndColumns);
      }
    }
    finally
    {
      if (privilegedConnection != dmlConnection)
      {
        // if 'privilegedConnection' is not 'dmlConnection', it means it was
        // created for use by this method, and so it needs to be released/closed.
        privilegedConnection.close();
      }
    }
    if (LOG.isLoggable(Level.FINE))
    {
      LOG.fine("HistoryExport " + getHistoryId() + ": created table '" + tableName + "'.");
    }
  }

  /**
   * Adds the 'HISTORY_ID' column to the given table, and creates an index using the 'HISTORY_ID' and Timestamp columns.
   *
   * @param tableName The name of the table to which the HISTORY_ID column will be added.
   * @param tableId   The ID value of the entry in the HISTORY_TYPE_MAP for the table to which the HISTORY_ID column will be added.
   * @throws SQLException If an error occurs creating the table or index.
   * @since Niagara 4.14
   */
  private void addHistoryIdColumnAndIndexToTable(String tableName, Number tableId)
    throws SQLException
  {
    if (LOG.isLoggable(Level.FINE))
    {
      LOG.fine("HistoryExport " + getHistoryId() + ": Adding HISTORY_ID column to table '" + tableName + "'");
    }

    String addColumnStatement = dialect.makeAddColumnSql(tableName, HISTORY_ID, BSqlType.sqlVarchar, 500, "");

    // If 'dmlConnection' is privileged, avoid the creation of another 'privileged' connection.
    Connection privilegedConnection;
    try
    {
      privilegedConnection = AccessController.doPrivileged((PrivilegedExceptionAction<Connection>) () ->
        dmlConnectionIsPrivileged
          ? dmlConnection
          : BRdbmsHistoryDeviceExt.getConnectionForDescriptor(db, this, /*requestPrivileged*/true, null));
    }
    catch (PrivilegedActionException e)
    {
      Throwable cause = e.getCause();
      if (cause instanceof SQLException)
      {
        throw (SQLException) cause;
      }
      throw new SQLException(cause);
    }

    try (Statement privilegedConnectionStatement = privilegedConnection.createStatement())
    {
      privilegedConnectionStatement.executeUpdate(addColumnStatement);

      if (!devicelet.getUseLastTimestamp() || devicelet.getAlwaysCreateIndexForNewTables())
      {
        // Add a composite index for HISTORY_ID and Timestamp
        Pair<String, String> indexNameAndColumns = dialect.getNewIndexNameAndColumns(String.valueOf(tableId));
        RdbmsHistoryUtil.createIndex(privilegedConnection, dialect, tableName, indexNameAndColumns);
      }
    }
    finally
    {
      // if 'privilegedConnection' is not 'dmlConnection', it means it was
      // created for use by this method, and so it needs to be released/closed.
      if (privilegedConnection != dmlConnection)
      {
        privilegedConnection.close();
      }
    }
  }

  private boolean columnExists(String tableName, String columnName) throws SQLException
  {
    DatabaseMetaData metadata = dmlConnection.getMetaData();
    try (ResultSet rs = metadata.getColumns(null, null, tableName, columnName))
    {
      return rs.next();
    }
  }

  /**
   * insertRecords
   */
  private BAbsTime insertRecords(
    BAbsTime lastExportTime,
    String tableName,
    boolean fallbackToLastTimestampPropertyValueIfNoMaxTimestampInDatabase
  )
    throws SQLException
  {
    if (exportRangeStartTime == null && fallbackToLastTimestampPropertyValueIfNoMaxTimestampInDatabase)
    {
      // If this block is reached, it means that the HISTORY_ID column was added to an existing table,
      // which may have already contained history records (including records still in the local history
      // database). In that case, these records will have the column's default value (an empty string),
      // and would have been excluded from the record set used to determine the latest timestamp of
      // the History records in the remote database (as that query uses the History's HistoryId in the WHERE clause).
      // To avoid duplicate History records in the remote database, the value of the 'lastTimestamp' property will
      // be used when querying the local database for the next records to export, instead of exporting its entire
      // contents (which may be _one_ cause of duplicate records ending up in the remote database).
      exportRangeStartTime = BAbsTime.make(lastExportTime.getMillis() + dialect.getTimestampAccuracy());
    }

    BHistoryDatabase localDb =
      ((BHistoryService)Sys.getService(BHistoryService.TYPE)).getDatabase();
    // For Rdbms History Exports, we are always excluding archive history data
    // and only considering local history data via the Context used to retrieve
    // the HistorySpaceConnection
    try (HistorySpaceConnection historyConn =
           localDb.getConnection(HistoryQuery.makeExcludeArchiveDataContext(null));
         Cursor<BHistoryRecord> records = historyConn.timeQuery(history, exportRangeStartTime, null).cursor())
    {
      // set up extra values
      Array<String> extraFields = new Array<String>(String.class);
      BObject[] extraValues;
      BFacets[] extraFacets;

      if (db.getExportMode().getOrdinal() == BRdbmsHistoryExportMode.BY_HISTORY_TYPE)
      {
        extraFields.add(HISTORY_ID);
      }
      if (template instanceof BTrendRecord)
      {
        extraFields.addAll(new String[]{ "TRENDFLAGS_TAG", "STATUS_TAG" });
      }

      //check if the UTC_OFFSET column exists (to maintain backwards compatibility with old databases)
      //Note: disable adding the UTC_OFFSET column for now
      //because the presence of this extra column may complicate
      //sql scheme queries involving timestamps
      // TODO: Need to confirm if the "UTC_OFFSET" column addition from
      //  NCCB-2334 ever got released to customers in 3.8 before it was reverted.
      //  If it was never released, this column existence check is probably not
      //  necessary and might make exports run just a little faster.

      boolean addUtcOffsetEntry = false && dialect.supportsTimestamp() && columnExists(tableName, "UTC_OFFSET");

      //check if the UTC_OFFSET column exists and add a field for that if it does
      if(addUtcOffsetEntry)
      {
        extraFields.add("UTC_OFFSET");
      }

      int extraFieldsSize = extraFields.size();
      extraValues = new BObject[extraFieldsSize];
      extraFacets = new BFacets[extraFieldsSize];

      Property[] propTemplate = template.getPropertiesArray();
      // insert
      String sql = dialect.makeInsertSql(tableName, propTemplate, extraFields.trim());
      int totalCount = 0;
      int count = 0;

      try (PreparedStatement ps = dmlConnection.prepareStatement(sql))
      {
        while (records.next())
        {
          BHistoryRecord rec = records.get();
          BAbsTime ts = rec.getTimestamp();

          if (lastExportTime.compareTo(ts) < 0)
          {
            lastExportTime = ts;
          }

          switch (db.getExportMode().getOrdinal())
          {
            // byHistory
            case BRdbmsHistoryExportMode.BY_HISTORY_ID:
              if (template instanceof BTrendRecord)
              {
                BTrendRecord t = (BTrendRecord) rec;
                extraValues[0] = BString.make(t.getTrendFlags().toString());
                extraValues[1] = BString.make(t.getStatus().toString());
              }
              break;

            // byType
            case BRdbmsHistoryExportMode.BY_HISTORY_TYPE:
              if (extraValues.length > 0)
              {
                extraValues[0] = BString.make(history.getId().toString());
              }

              if (template instanceof BTrendRecord)
              {
                BTrendRecord t = (BTrendRecord) rec;
                extraValues[1] = BString.make(t.getTrendFlags().toString());
                extraValues[2] = BString.make(t.getStatus().toString());
              }
              break;

            // oops
            default:
              throw new IllegalStateException();
          }

          //Add UTC millis offset for all records (if the column exists)
          if (addUtcOffsetEntry)
          {
            //calculate the long UTC offset
            RdbmsDialect context = (RdbmsDialect) db.getRdbmsContext();
            BTimeZone timezone = context
              .useUtcTimestamps() ? BTimeZone.UTC : devicelet
              .getUseHistoryConfigTimeZone() ? config.getTimeZone() : BTimeZone
              .getLocal();
            BLong utcOffset = BLong
              .make(timezone.getJavaTimeZone().getOffset(ts.getMillis()));
            extraValues[extraValues.length - 1] = utcOffset; //this assumes the UTC_OFFSET column is the last column in the extra fields
          }
          if (getExportInvalidValues() || !hasInvalidValues(rec, propTemplate, extraValues))
          {
            dialect.insertRecord(ps, rec, propTemplate, extraValues, extraFacets);
            count++;
            totalCount++;
          }

          if (count >= EXPORT_BATCH_SIZE)
          {
            if (LOG.isLoggable(Level.FINE))
            {
              LOG.fine("HistoryExport " + getHistoryId() + ": sending batch at " + totalCount + " records");
            }
            ps.executeBatch();
            ps.clearBatch();
            count = 0;
          }
        }

        if (count > 0)
        {
          ps.executeBatch();
        }
      }
      if (LOG.isLoggable(Level.FINE))
      {
        LOG.fine("HistoryExport " + getHistoryId() + ": inserted " + totalCount + " records");
      }
    }

    return lastExportTime;
  }

  private boolean hasInvalidValues(BHistoryRecord rec, Property[] propTemplate, BObject[] extraValues)
  {

    for(Property p: propTemplate)
    {
      if (rec.get(p).getType().is(BDouble.TYPE) &&
        !Double.isFinite(((BDouble)rec.get(p)).getDouble()))
      {
        return true;
      }
      if (rec.get(p).getType().is(BFloat.TYPE) &&
        !Float.isFinite(((BFloat)rec.get(p)).getFloat()))
      {
        return true;
      }
    }
    for (BObject b: extraValues)
    {
      if (b.getType().is(BDouble.TYPE) &&
        !Double.isFinite(((BDouble)b.asValue()).getDouble()))
      {
        return true;
      }
      if (b.getType().is(BFloat.TYPE) &&
        !Float.isFinite(((BFloat)b.asValue()).getFloat()))
      {
        return true;
      }
    }
    return false;
  }

  /**
   * we can assume there will always be a timestamp field.
   */
  private BAbsTime lookupMaxTimestamp(String tableName)
    throws SQLException
  {
    // use lastExportTime
    if (devicelet.getUseLastTimestamp() /* && !db.getUseUtcTimestamps()*/)
    {
      BAbsTime time = getLastTimestamp();

      // if the timestamp is already set, then return it
      if (!time.equals(BAbsTime.NULL))
      {
        return time;
      }
      // everything is fine -- this must be the first export
    }

    // query the database
    else
    {
      String sql = "SELECT MAX(TIMESTAMP) AS MAX_TIMESTAMP FROM " + tableName;

      switch (db.getExportMode().getOrdinal())
      {
        // byHistory
        case BRdbmsHistoryExportMode.BY_HISTORY_ID:
          break;

        // byType
        case BRdbmsHistoryExportMode.BY_HISTORY_TYPE:
          sql += " WHERE HISTORY_ID = '" + fixQuotes(history.getId().toString()) + '\'';
          break;

        // oops
        default: throw new IllegalStateException();
      }

      /*BTimeZone dbtimezone = getExportTimeZone(); //retrieve timezone stored in DB_TIMEZONE column in meta table
      System.out.println("the db timezone = " + dbtimezone);*/
      Timestamp ts = null;

      try (ResultSet rs = dmlStatement.executeQuery(sql))
      {
        if (rs.next())
        {
          if (dialect.supportsTimestamp())
          {
            Calendar cal = db.getTimestampStorage().equals(BRdbmsTimestampStorage.utcTimestamp) ? Calendar.getInstance(BTimeZone.UTC.getJavaTimeZone()) :
              devicelet.getUseHistoryConfigTimeZone() && devicelet.getUseLastTimestamp() ? Calendar.getInstance(config.getTimeZone().getJavaTimeZone()) : Calendar.getInstance();

            ts = rs.getTimestamp("MAX_TIMESTAMP", cal);
            if (ts == null)
            {
              return null;
            }
          }

          else
          {
            String str = rs.getString("MAX_TIMESTAMP");
            if (str == null)
            {
              return null;
            }

            ts = new Timestamp(Long.parseLong(str));
          }

          long millis = ts.getTime(); //could be UTC millis, station time millis or history config millis
          //based on what timezone was used to export the record to the database
          return BAbsTime.make(millis);
        }
      }
    }
    return null;
  }

  private BTimeZone getExportTimeZone()
    throws SQLException
  {
    Optional<Map<String, Object>> metaRecord = getMetaRecord();
    BTimeZone timezone = BTimeZone.NULL;

    //if we're using the new schema then the stored timezone information is available
    //from the DB_TIMEZONE column for the meta record
    if (metaRecord.isPresent() && isNewSchema)
    {
      Map<String, Object> meta = metaRecord.get();
      try
      {
        String timeZoneId = (String)meta.get("DB_TIMEZONE");
        timeZoneId = timeZoneId.substring(0, timeZoneId.indexOf("(")).trim();
        timezone = BTimeZone.getTimeZone(timeZoneId);
      }

      catch(Exception ex)
      {
        if (LOG.isLoggable(Level.SEVERE))
        {
          LOG.log(Level.SEVERE, "Failed to retrieve timezone for export of " + toPathString(), LOG.isLoggable(Level.FINE) ? ex : null);
        }
      }
    }
    else if (devicelet.getUseHistoryConfigTimeZone() && devicelet.getUseLastTimestamp())
    {
      //else check if the records were exported with the 'use history config timezone' option
      //enabled. Use that timezone if this is the case
      timezone = config.getTimeZone();
    }
    else
    {
      //else default to the local station timezone
      timezone = BTimeZone.getLocal();
    }

    return timezone;
  }

////////////////////////////////////////////////////////////////
// private - meta data
////////////////////////////////////////////////////////////////

  /**
   * getMetaTableName
   */
  private String getMetaTableName()
  {
    switch (db.getExportMode().getOrdinal())
    {
      case BRdbmsHistoryExportMode.BY_HISTORY_ID:
        return "HISTORY_CONFIG";
      case BRdbmsHistoryExportMode.BY_HISTORY_TYPE:
        return "HISTORY_TYPE_MAP";
      default: throw new IllegalStateException();
    }
  }

  /**
   * createMetaTable
   */
  private void createMetaTable()
    throws SQLException
  {
    String name = getMetaTableName();

    Property[] metaTemplate = makeMetaTemplate();

    String sql = dialect.makeCreateTableSql(
      name,
      metaTemplate,
      new String[]{ "TABLE_NAME", "DB_TIMEZONE" },
      new Type[]{ BString.TYPE, BString.TYPE },
      new BFacets[]{ BFacets.NULL, BFacets.NULL }
    );

    // If 'dmlConnection' is privileged, avoid the creation of another 'privileged' connection.
    Connection privilegedConnection;
    try
    {
      privilegedConnection = AccessController.doPrivileged((PrivilegedExceptionAction<Connection>) () ->
        dmlConnectionIsPrivileged
          ? dmlConnection
          : BRdbmsHistoryDeviceExt.getConnectionForDescriptor(db, this, /*requestPrivileged*/true, null));
    }
    catch (PrivilegedActionException e)
    {
      Throwable cause = e.getCause();
      if (cause instanceof SQLException)
      {
        throw (SQLException) cause;
      }
      throw new SQLException(cause);
    }

    try (Statement privilegedConnectionStatement = privilegedConnection.createStatement())
    {
      privilegedConnectionStatement.executeUpdate(sql);

      // make the sequence if need be
      if (dialect.hasSequences())
      {
        privilegedConnectionStatement.executeUpdate("CREATE SEQUENCE " + name + "_Q");
      }
    }
    finally
    {
      if (privilegedConnection != dmlConnection)
      {
        // if 'privilegedConnection' is not 'dmlConnection', it means it was
        // created for use by this method, and so it needs to be released/closed.
        privilegedConnection.close();
      }
    }
    if (LOG.isLoggable(Level.FINE))
    {
      LOG.fine("HistoryExport " + getHistoryId() + ": created table '" + name + "'.");
    }
  }

  /**
   * insertMetaRecord
   */
  private void insertMetaRecord()
    throws SQLException
  {

    try
    {
      Property[] metaTemplate = makeMetaTemplate();

      try (
        PreparedStatement ps = dmlConnection.prepareStatement(
          dialect.makeInsertSql(
            getMetaTableName(),
            metaTemplate,
            isNewSchema ? new String[]{ "TABLE_NAME", "DB_TIMEZONE" } :
              new String[]{ "TABLE_NAME" }
          )
        )
      )
      {
        // invent a table name
        String tableName = inventTableName();
        BString tableNameStr = BString.make(tableName);


        //initialize DB_TIMEZONE
        RdbmsDialect context = (RdbmsDialect)db.getRdbmsContext();
        BTimeZone dbTimeZone = context.useUtcTimestamps() ? BTimeZone.UTC :
          devicelet.getUseHistoryConfigTimeZone() ?
            config.getTimeZone() : BTimeZone.getLocal();

        BString timeZoneStr  = BString.make(dbTimeZone.toString());

        BObject[] tablePlusTz = {tableNameStr, timeZoneStr};
        BObject[] table = {tableNameStr};

        // insert the record
        dialect.insertRecord(
          ps,
          config,
          metaTemplate,
          isNewSchema ? tablePlusTz : table,
          isNewSchema ? new BFacets[] { BFacets.NULL, BFacets.NULL } : new BFacets[] { BFacets.NULL });

        ps.executeBatch();
      }
    }
    catch (Exception e)
    {
      throw new BajaRuntimeException(e);
    }
  }

  /**
   * Check the existence of the meta record by history id from the relevant meta table.
   */
  private Optional<Map<String, Object>> getMetaRecord()
    throws SQLException
  {
    // If we've already got the meta-record, return it.
    if (historyMetaRecord != null)
    {
      return Optional.of(historyMetaRecord);
    }
    else
    {
      String sql;
      String metaTableName = getMetaTableName();
      boolean exportingById = db.getExportMode().equals(BRdbmsHistoryExportMode.byHistoryId);

      //select on the ID_ column if this is the HISTORY_CONFIG table or if the table
      //is using the new schema
      if (exportingById || (isNewSchema != null && isNewSchema))
      {
        sql = "SELECT * FROM " + metaTableName +
          " WHERE ID_ = '" + fixQuotes(config.getId().toString()) + "'";
      }
      else
      {
        //no schema upgrade has been performed and this isn't the HISTORY_CONFIG table
        //so this must be the HISTORY_TYPE_MAP table with the old schema which only
        //contains information about the history record type
        sql = "SELECT * FROM " + metaTableName +
          " WHERE RECORDTYPE = '" + fixQuotes(config.getRecordType().toString()) + "'";
      }

      Optional<Map<String, Object>> metaRecord = Optional.empty();
      try (ResultSet resultSet = dmlStatement.executeQuery(sql))
      {
        if (resultSet.next())
        {
          Map<String, Object> result = new HashMap<>();
          for (int i = 1; i < resultSet.getMetaData().getColumnCount(); i++)
          {
            String columnName = resultSet.getMetaData().getColumnName(i);
            Object columnValue = resultSet.getObject(i);
            result.put(columnName, columnValue);
          }
          metaRecord = Optional.of(result);
        }
      }
      // If we've already determined that the schema we're operating against is the
      // 'new' schema, and we've found the _right_ meta-record for the
      // table corresponding to the History, we can safely cache the
      // meta-record. We avoid doing this if we're unsure about the version of the schema
      // because there's no guarantee that the query used to find the meta-record
      // returned a single record (since exporting in byHistoryType mode can result
      // in multiple records in the HISTORY_TYPE_MAP using the same record type/table name).
      if (isNewSchema != null && metaRecord.isPresent())
      {
        historyMetaRecord = metaRecord.get();
      }
      return metaRecord;
    }
  }

  /**
   * inventTableName
   */
  private String inventTableName()
    throws SQLException
  {
    int max = dialect.getMaxTableName();

    // kludge: if we have sequences, then the table name can be at
    // most two less than the max sequence name, because we create
    // the sequence name from the tableName plus "_Q".
    if (dialect.hasSequences())
    {
      int seq = dialect.getMaxSequenceName();
      if (max > (seq - 2)) max = seq-2;
    }

    // make up a name
    String tableName;
    switch (db.getExportMode().getOrdinal())
    {
      // byHistory
      case BRdbmsHistoryExportMode.BY_HISTORY_ID:

        //NCCB-4553 - Unescape names before calls to toUpperCase()
        String devName  = SlotPath.unescape(config.getId().getDeviceName()).toUpperCase();
        String histName = SlotPath.unescape(config.getId().getHistoryName()).toUpperCase();

        tableName = TextUtil.truncate(
          dialect.mangleIdentifier(devName) + "_" + dialect.mangleIdentifier(histName),
          max
        );
        break;

      // byType
      case BRdbmsHistoryExportMode.BY_HISTORY_TYPE:
        tableName = TextUtil.truncate(
          dialect.mangleIdentifier(config.getRecordType().toString()),
          max
        );
        break;

      // oops
      default: throw new IllegalStateException();
    }

    // make sure its really unique (but only for HISTORY_CONFIG table)
    if (db.getExportMode().equals(BRdbmsHistoryExportMode.byHistoryId))
    {
      /*
       * In case of export by History_Id, tables are created with the generated table name here.
       * If the generated table name is longer than the object character limit of the database in use
       * the table name is trimmed to fit within the max table name limit. This may result in duplicate names.
       * To avoid duplicate names, a counter is used here which is appended to the trimmed table name. In case the newly
       * generated name is already present, increment the counter and check again.
       */
      if(metaRecordExists(tableName))
      {
        String fullName = tableName;

        int n = 1;
        do {
          String nn = Integer.toString(n++);
          tableName = TextUtil.truncate(fullName, max - nn.length()) + nn;
        } while (metaRecordExists(tableName));
      }
    }

    return tableName;
  }

  /**
   * metaRecordExists
   */
  private boolean metaRecordExists(String tableName)
    throws SQLException
  {
    boolean metaRecordExists;
    String selectMetaTableName =
      "SELECT * FROM " + getMetaTableName() + " WHERE " + "TABLE_NAME = '" + tableName + "'";
    try (ResultSet rs = dmlStatement.executeQuery(selectMetaTableName))
    {
      metaRecordExists = rs.next();
    }

    return metaRecordExists;
  }

  /**
   * makeMetaTemplate
   *
   * @throws SQLException
   */
  private Property[] makeMetaTemplate() throws SQLException
  {
    //add a valueFacets slot to the history config if it doesn't contain one
    //this will usually apply to the log and audit histories, trend records already have the property
    if (config.getProperty("valueFacets") == null)
    {
      config.add("valueFacets", BFacets.NULL);
    }

    Array<Property> arr = new Array<>(Property.class);
    SlotCursor<Property> historyConfigProperties = config.getProperties();

    boolean exportingByType = db.getExportMode().equals(BRdbmsHistoryExportMode.byHistoryType);

    while (historyConfigProperties.next())
    {
      Property property = historyConfigProperties.property();
      String name = property.getName();

      if (exportingByType)
      {
        //properties for export by type HISTORY_TYPE_MAP table
        if (isNewSchema)
        {
          if (name.equals(BHistoryConfig.id.getName()) ||
            name.equals(BHistoryConfig.timeZone.getName()) ||
            name.equals(BHistoryConfig.recordType.getName()))
          {
            arr.add(property);
            continue;
          }
        }
        else if (name.equals(BHistoryConfig.recordType.getName()))
        {
          arr.add(property);
          continue;
        }
      }
      else //exporting by id
      {
        //properties for export by id HISTORY_CONFIG table
        if (name.equals(BHistoryConfig.id.getName()) ||
          name.equals(BHistoryConfig.historyName.getName()) ||
          name.equals(BHistoryConfig.source.getName()) ||
          name.equals(BHistoryConfig.sourceHandle.getName()) ||
          name.equals(BHistoryConfig.timeZone.getName()) ||
          name.equals(BHistoryConfig.interval.getName()) ||
          name.equals(BHistoryConfig.systemTags.getName()))
        {
          arr.add(property);
          continue;
        }
      }

      //add the 'valueFacets' dynamic property
      if (property.isDynamic())
      {
        if (!name.equals("valueFacets"))
        {
          continue; //skip all dynamic properties except 'valueFacets'
        }

        String metaTableName = getMetaTableName();
        String facetsColumn = dialect.mangleIdentifier(name);

        //if the meta table doesn't exist add the property to the list of columns to create (during createMetaTable())
        //or if the meta table exists and the facets column is present add the property to the list of columns to populate during an insert (during insertMetaRecord())
        if (!dialect.tableExists(db, dmlConnection, metaTableName) || columnExists(metaTableName, facetsColumn))
        {
          arr.add(property);
        }
      }
    }

    return arr.trim();
  }

////////////////////////////////////////////////////////////////
// attributes
////////////////////////////////////////////////////////////////

  private BRdbmsHistoryDeviceExt devicelet;
  private BRdbms db;
  private BRdbmsDeprecatedDialect dialect;
  // JDBC Connection for DML operations (including querying Database Metadata).
  private Connection dmlConnection;
  private boolean dmlConnectionIsPrivileged;
  private Statement dmlStatement;

  private BIHistory history;
  private BHistoryConfig config;
  private BHistoryRecord template;

  // Intentionally use the ternary Boolean, where null
  // should be treated as 'not yet determined'.
  private Boolean isNewSchema = null;
  private boolean metadataTableExists = false;
  private Map<String, Object> historyMetaRecord = null;
  private BAbsTime exportRangeStartTime = null;

  private static final Logger LOG = Logger.getLogger("rdb");

  private static final BFacets SHOW_MILLIS = BFacets.make(
    BFacets.SHOW_MILLISECONDS, true);

  /*
   * Initialize batch size for history export from the System property if set.
   * If invalid/no property is provided 1000 will be used as a default value.
   */
  private static final int DEFAULT_EXPORT_BATCH_SIZE = 1000;
  private static final Integer EXPORT_BATCH_SIZE =
    AccessController.doPrivileged(
      (PrivilegedAction<Integer>) () -> Integer
        .getInteger("niagara.rdb.historyExport.batchSize",
          DEFAULT_EXPORT_BATCH_SIZE));

  private static final String HISTORY_ID = "HISTORY_ID";

  private static final String METARECORD_TABLE_NAME = "TABLE_NAME";
  private static final String METARECORD_ID = "ID";

}
