/*
 * Copyright 2001 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.rdb.history;

import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import javax.baja.driver.history.BHistoryDeviceExt;
import javax.baja.history.BHistoryId;
import javax.baja.naming.BOrd;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraAction;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.rdb.BRdbms;
import javax.baja.security.BPassword;
import javax.baja.sys.Action;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.rdb.BRdbmsDiscoverTablesJob;
import com.tridium.rdb.history.BRdbmsMigrateIndexesJob;

/**
 * BRdbmsHistoryDeviceExt maps historical data into a relational database.
 *
 * @author Mike Jarmy
 * @version $Revision: 13$ $Date: 1/12/11 9:20:53 AM EST$
 * @creation 24 Jul 03
 * @since Baja 1.0
 */
@NiagaraType
/*
 Whether to make use of the value of the maxTimestamp property on each
 BRdbmsHistoryExport object.
 If useLastTimestamp is false, then queries will be issued on the database
 to find the last export time.  In addition, database timestamp indexes will
 be created at the time that the database tables are created.
 If useLastTimestamp is true, then the value of the maxTimestamp property on each
 BRdbmsHistoryExport object will be used during the history export process
 to find the last export time.  In addition, database indexes will <i><b>NOT</b></i>
 be created at the time that the database tables are created.
 */
@NiagaraProperty(
  name = "useLastTimestamp",
  type = "boolean",
  defaultValue = "false"
)
/*
 Whether to use the timezone property of a history's historyConfig
 when exported records.
 If useHistoryConfigTimeZone is false, then the timezone of the supervisor
 that is actually doing the export will be used when timestamps are exported.
 If useHistoryConfigTimeZone is true, then the timezone property of the
 historyConfig object will be used when timestamps are exported.
 NOTE: if the parent database, namely MySQL, does not support timestamps
 natively, then the value of this property will be ignored.
 */
@NiagaraProperty(
  name = "useHistoryConfigTimeZone",
  type = "boolean",
  defaultValue = "false"
)
/*
 Create optimized indexes for the new history export record tables.
 @since Niagara 4.11
 */
@NiagaraProperty(
  name = "alwaysCreateIndexForNewTables",
  type = "boolean",
  defaultValue = "false"
)
/*
 This action is called programmatically to invoke the job
 for discovering the rdb tables available for import.
 */
@NiagaraAction(
  name = "submitRdbTableDiscoveryJob",
  returnType = "BOrd",
  flags = Flags.HIDDEN
)
/*
 Updates the lastTimestamp field on all the BRdbmsHistoryExport objects,
 by querying the database.
 */
@NiagaraAction(
  name = "updateLastTimestamp",
  flags = Flags.HIDDEN | Flags.CONFIRM_REQUIRED
)
/*
 Set the lastTimestamp field on all the BRdbmsHistoryExport objects
 to NULL.
 */
@NiagaraAction(
  name = "clearLastTimestamp",
  flags = Flags.HIDDEN | Flags.CONFIRM_REQUIRED
)
/*
 Create optimized index on the history exports under the device depending on the export mode.
 In case the export mode is by HistoryId, an index is created on the Timestamp column.
 In case the export mode is by HistoryType, a composite index is created on the
 HistoryId and the Timestamp column which makes it quicker to do searches over the table.
 It drops the existing index (based on just TIMESTAMP OR HISTORY_ID) if any
 before creating the new index on the table.
 @since Niagara 4.11
 */
@NiagaraAction(
  name = "migrateToOptimizedTableIndexes",
  returnType = "BOrd",
  flags = Flags.CONFIRM_REQUIRED
)
public abstract class BRdbmsHistoryDeviceExt
  extends BHistoryDeviceExt
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.rdb.history.BRdbmsHistoryDeviceExt(4149856925)1.0$ @*/
/* Generated Mon Oct 07 10:30:16 EDT 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Property "useLastTimestamp"

  /**
   * Slot for the {@code useLastTimestamp} property.
   * Whether to make use of the value of the maxTimestamp property on each
   * BRdbmsHistoryExport object.
   * If useLastTimestamp is false, then queries will be issued on the database
   * to find the last export time.  In addition, database timestamp indexes will
   * be created at the time that the database tables are created.
   * If useLastTimestamp is true, then the value of the maxTimestamp property on each
   * BRdbmsHistoryExport object will be used during the history export process
   * to find the last export time.  In addition, database indexes will <i><b>NOT</b></i>
   * be created at the time that the database tables are created.
   * @see #getUseLastTimestamp
   * @see #setUseLastTimestamp
   */
  @Generated
  public static final Property useLastTimestamp = newProperty(0, false, null);

  /**
   * Get the {@code useLastTimestamp} property.
   * Whether to make use of the value of the maxTimestamp property on each
   * BRdbmsHistoryExport object.
   * If useLastTimestamp is false, then queries will be issued on the database
   * to find the last export time.  In addition, database timestamp indexes will
   * be created at the time that the database tables are created.
   * If useLastTimestamp is true, then the value of the maxTimestamp property on each
   * BRdbmsHistoryExport object will be used during the history export process
   * to find the last export time.  In addition, database indexes will <i><b>NOT</b></i>
   * be created at the time that the database tables are created.
   * @see #useLastTimestamp
   */
  @Generated
  public boolean getUseLastTimestamp() { return getBoolean(useLastTimestamp); }

  /**
   * Set the {@code useLastTimestamp} property.
   * Whether to make use of the value of the maxTimestamp property on each
   * BRdbmsHistoryExport object.
   * If useLastTimestamp is false, then queries will be issued on the database
   * to find the last export time.  In addition, database timestamp indexes will
   * be created at the time that the database tables are created.
   * If useLastTimestamp is true, then the value of the maxTimestamp property on each
   * BRdbmsHistoryExport object will be used during the history export process
   * to find the last export time.  In addition, database indexes will <i><b>NOT</b></i>
   * be created at the time that the database tables are created.
   * @see #useLastTimestamp
   */
  @Generated
  public void setUseLastTimestamp(boolean v) { setBoolean(useLastTimestamp, v, null); }

  //endregion Property "useLastTimestamp"

  //region Property "useHistoryConfigTimeZone"

  /**
   * Slot for the {@code useHistoryConfigTimeZone} property.
   * Whether to use the timezone property of a history's historyConfig
   * when exported records.
   * If useHistoryConfigTimeZone is false, then the timezone of the supervisor
   * that is actually doing the export will be used when timestamps are exported.
   * If useHistoryConfigTimeZone is true, then the timezone property of the
   * historyConfig object will be used when timestamps are exported.
   * NOTE: if the parent database, namely MySQL, does not support timestamps
   * natively, then the value of this property will be ignored.
   * @see #getUseHistoryConfigTimeZone
   * @see #setUseHistoryConfigTimeZone
   */
  @Generated
  public static final Property useHistoryConfigTimeZone = newProperty(0, false, null);

  /**
   * Get the {@code useHistoryConfigTimeZone} property.
   * Whether to use the timezone property of a history's historyConfig
   * when exported records.
   * If useHistoryConfigTimeZone is false, then the timezone of the supervisor
   * that is actually doing the export will be used when timestamps are exported.
   * If useHistoryConfigTimeZone is true, then the timezone property of the
   * historyConfig object will be used when timestamps are exported.
   * NOTE: if the parent database, namely MySQL, does not support timestamps
   * natively, then the value of this property will be ignored.
   * @see #useHistoryConfigTimeZone
   */
  @Generated
  public boolean getUseHistoryConfigTimeZone() { return getBoolean(useHistoryConfigTimeZone); }

  /**
   * Set the {@code useHistoryConfigTimeZone} property.
   * Whether to use the timezone property of a history's historyConfig
   * when exported records.
   * If useHistoryConfigTimeZone is false, then the timezone of the supervisor
   * that is actually doing the export will be used when timestamps are exported.
   * If useHistoryConfigTimeZone is true, then the timezone property of the
   * historyConfig object will be used when timestamps are exported.
   * NOTE: if the parent database, namely MySQL, does not support timestamps
   * natively, then the value of this property will be ignored.
   * @see #useHistoryConfigTimeZone
   */
  @Generated
  public void setUseHistoryConfigTimeZone(boolean v) { setBoolean(useHistoryConfigTimeZone, v, null); }

  //endregion Property "useHistoryConfigTimeZone"

  //region Property "alwaysCreateIndexForNewTables"

  /**
   * Slot for the {@code alwaysCreateIndexForNewTables} property.
   * Create optimized indexes for the new history export record tables.
   * @since Niagara 4.11
   * @see #getAlwaysCreateIndexForNewTables
   * @see #setAlwaysCreateIndexForNewTables
   */
  @Generated
  public static final Property alwaysCreateIndexForNewTables = newProperty(0, false, null);

  /**
   * Get the {@code alwaysCreateIndexForNewTables} property.
   * Create optimized indexes for the new history export record tables.
   * @since Niagara 4.11
   * @see #alwaysCreateIndexForNewTables
   */
  @Generated
  public boolean getAlwaysCreateIndexForNewTables() { return getBoolean(alwaysCreateIndexForNewTables); }

  /**
   * Set the {@code alwaysCreateIndexForNewTables} property.
   * Create optimized indexes for the new history export record tables.
   * @since Niagara 4.11
   * @see #alwaysCreateIndexForNewTables
   */
  @Generated
  public void setAlwaysCreateIndexForNewTables(boolean v) { setBoolean(alwaysCreateIndexForNewTables, v, null); }

  //endregion Property "alwaysCreateIndexForNewTables"

  //region Action "submitRdbTableDiscoveryJob"

  /**
   * Slot for the {@code submitRdbTableDiscoveryJob} action.
   * This action is called programmatically to invoke the job
   * for discovering the rdb tables available for import.
   * @see #submitRdbTableDiscoveryJob()
   */
  @Generated
  public static final Action submitRdbTableDiscoveryJob = newAction(Flags.HIDDEN, null);

  /**
   * Invoke the {@code submitRdbTableDiscoveryJob} action.
   * This action is called programmatically to invoke the job
   * for discovering the rdb tables available for import.
   * @see #submitRdbTableDiscoveryJob
   */
  @Generated
  public BOrd submitRdbTableDiscoveryJob() { return (BOrd)invoke(submitRdbTableDiscoveryJob, null, null); }

  //endregion Action "submitRdbTableDiscoveryJob"

  //region Action "updateLastTimestamp"

  /**
   * Slot for the {@code updateLastTimestamp} action.
   * Updates the lastTimestamp field on all the BRdbmsHistoryExport objects,
   * by querying the database.
   * @see #updateLastTimestamp()
   */
  @Generated
  public static final Action updateLastTimestamp = newAction(Flags.HIDDEN | Flags.CONFIRM_REQUIRED, null);

  /**
   * Invoke the {@code updateLastTimestamp} action.
   * Updates the lastTimestamp field on all the BRdbmsHistoryExport objects,
   * by querying the database.
   * @see #updateLastTimestamp
   */
  @Generated
  public void updateLastTimestamp() { invoke(updateLastTimestamp, null, null); }

  //endregion Action "updateLastTimestamp"

  //region Action "clearLastTimestamp"

  /**
   * Slot for the {@code clearLastTimestamp} action.
   * Set the lastTimestamp field on all the BRdbmsHistoryExport objects
   * to NULL.
   * @see #clearLastTimestamp()
   */
  @Generated
  public static final Action clearLastTimestamp = newAction(Flags.HIDDEN | Flags.CONFIRM_REQUIRED, null);

  /**
   * Invoke the {@code clearLastTimestamp} action.
   * Set the lastTimestamp field on all the BRdbmsHistoryExport objects
   * to NULL.
   * @see #clearLastTimestamp
   */
  @Generated
  public void clearLastTimestamp() { invoke(clearLastTimestamp, null, null); }

  //endregion Action "clearLastTimestamp"

  //region Action "migrateToOptimizedTableIndexes"

  /**
   * Slot for the {@code migrateToOptimizedTableIndexes} action.
   * Create optimized index on the history exports under the device depending on the export mode.
   * In case the export mode is by HistoryId, an index is created on the Timestamp column.
   * In case the export mode is by HistoryType, a composite index is created on the
   * HistoryId and the Timestamp column which makes it quicker to do searches over the table.
   * It drops the existing index (based on just TIMESTAMP OR HISTORY_ID) if any
   * before creating the new index on the table.
   * @since Niagara 4.11
   * @see #migrateToOptimizedTableIndexes()
   */
  @Generated
  public static final Action migrateToOptimizedTableIndexes = newAction(Flags.CONFIRM_REQUIRED, null);

  /**
   * Invoke the {@code migrateToOptimizedTableIndexes} action.
   * Create optimized index on the history exports under the device depending on the export mode.
   * In case the export mode is by HistoryId, an index is created on the Timestamp column.
   * In case the export mode is by HistoryType, a composite index is created on the
   * HistoryId and the Timestamp column which makes it quicker to do searches over the table.
   * It drops the existing index (based on just TIMESTAMP OR HISTORY_ID) if any
   * before creating the new index on the table.
   * @since Niagara 4.11
   * @see #migrateToOptimizedTableIndexes
   */
  @Generated
  public BOrd migrateToOptimizedTableIndexes() { return (BOrd)invoke(migrateToOptimizedTableIndexes, null, null); }

  //endregion Action "migrateToOptimizedTableIndexes"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BRdbmsHistoryDeviceExt.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// Callbacks
////////////////////////////////////////////////////////////////

  /**
   * Get the Type for import descriptors managed by this devicelet.
   * If null, then the devicelet does not support imports.
   *
   * @return Returns the protocol specific import descriptor type
   *   or null if this devicelet does not support history imports.
   */
  @Override
  public Type getImportDescriptorType()
  {
    return BRdbmsHistoryImport.TYPE;
  }

  /**
   * The BRdbmsHistoryDeviceExt returns true for this method by default,
   * indicating it supports the generic BArchiveFolder, and its agent
   * views can be safely applied to the generic BArchiveFolder.  If this
   * is not true for a subclass, this method should be overridden to false.
   *
   * @since Niagara 3.5
   */
  @Override
  public boolean supportsGenericArchiveFolder()
  {
    return true;
  }


////////////////////////////////////////////////////////////////
// Actions
////////////////////////////////////////////////////////////////

  /**
   * Submits the job to discover the rdb tables available for
   * import.
   *
   * @return Returns the Ord to the RdbmsDiscoverTablesJob instance.
   */
  public BOrd doSubmitRdbTableDiscoveryJob(Context cx)
  {
    if ((getDevice().isFatalFault()) ||
        (getDevice().isDown()) ||
        (getDevice().isDisabled()))
      return null;

    return new BRdbmsDiscoverTablesJob(this).submit(cx);
  }

  /**
   * Updates the lastTimestamp field on all the BRdbmsHistoryExport objects,
   * by querying the database.
   * Note that this method always uses the username and password on the rdbms,
   * rather than the ones on each descriptor.
   */
  public void doUpdateLastTimestamp()
  {
    LOG.fine("Beginning timestamp update...");
    long t0 = System.currentTimeMillis();

    BRdbms db = (BRdbms) getDevice();

    if (db.getExportMode().getOrdinal() == BRdbmsHistoryExportMode.BY_HISTORY_ID)
    {
      throw new BajaRuntimeException(
        "updateLastTimestamp does not work when the " +
        "export mode is set to BY_HISTORY_ID.");
    }

    ///////////////////////////////////////////////////////////

    Map<String, BAbsTime> timestampMap = makeTimestampMap(db);

    BRdbmsHistoryExport[] exports = getChildren(BRdbmsHistoryExport.class);

    for (BRdbmsHistoryExport export : exports)
    {
      BHistoryId id = export.getHistoryId();
      BAbsTime timestamp = timestampMap.get(id.toString());
      export.setLastTimestamp((timestamp == null) ? BAbsTime.NULL : timestamp);
    }

    long ms = System.currentTimeMillis() - t0;
    LOG.fine("Updated " + exports.length + " timestamps (" + ms + "ms)");
  }

  /**
   * makeTimestampMap
   */
  private Map<String, BAbsTime> makeTimestampMap(BRdbms db)
  {
    try
    {
      Map<String, BAbsTime> map = new HashMap<>();
      try (
        Connection nonPrivilegedConnection = AccessController.doPrivileged((PrivilegedExceptionAction<Connection>) () -> db.getNonPrivilegedConnection(null));
        Statement tablesStatement = nonPrivilegedConnection.createStatement()
      )
      {
        ResultSet tables = tablesStatement.executeQuery("SELECT DISTINCT TABLE_NAME FROM HISTORY_TYPE_MAP");
        while (tables.next())
        {
          try (Statement maxTimestampStatement = nonPrivilegedConnection.createStatement())
          {
            ResultSet rs = maxTimestampStatement.executeQuery(
              "SELECT MAX(TIMESTAMP) AS MAX_TIMESTAMP, HISTORY_ID " +
                "FROM " + tables.getString("TABLE_NAME") + " GROUP BY HISTORY_ID");
            while (rs.next())
            {
              // TODO -- this won't work in MySQL!!!
              BAbsTime timestamp = BAbsTime.make(
                rs.getTimestamp("MAX_TIMESTAMP").getTime());

              map.put(rs.getString("HISTORY_ID"), timestamp);
            }
          }
        }
      }
      return map;
    }
    catch (Exception e)
    {
      Throwable cause = e;
      if (e instanceof PrivilegedActionException && e.getCause() != null)
      {
        cause = e.getCause();
      }

      throw new BajaRuntimeException(cause);
    }
  }

  /**
   * Set the lastTimestamp field on all the BRdbmsHistoryExport objects
   * to NULL.
   */
  public void doClearLastTimestamp()
  {
    BRdbmsHistoryExport[] exports = getChildren(BRdbmsHistoryExport.class);
    for (BRdbmsHistoryExport export : exports)
    {
      export.setLastTimestamp(BAbsTime.NULL);
    }
    LOG.fine("Cleared " + exports.length + " timestamps.");
  }

  /**
   * Invoke the job to create the composite indexes on the history export tables
   * @param cx the context for this job submission.
   * @since Niagara 4.11
   */
  public BOrd doMigrateToOptimizedTableIndexes(Context cx)
    throws Exception
  {
    return new BRdbmsMigrateIndexesJob(this).submit(cx);
  }

////////////////////////////////////////////////////////////////
// db utilities
////////////////////////////////////////////////////////////////

  /**
   * Deprecated. Use the
   * {@link #getConnectionForDescriptor(BRdbms, BRdbmsHistoryExport, boolean, Context)} method
   * instead.
   *
   * @deprecated since Niagara 4.15 - use {@link #getConnectionForDescriptor(BRdbms, BRdbmsHistoryExport, boolean, Context)}
   */
  @Deprecated
  public String getUserName(BRdbms database, BRdbmsHistoryExport descriptor)
  {
    String str = descriptor.getUserName();
    if ((str != null) && (!str.isEmpty()))
    {
      return str;
    }

    str = database.getUserName();
    if ((str != null) && (!str.isEmpty()))
    {
      return str;
    }

    return "";
  }

  /**
   * Deprecated. Use the
   * {@link #getConnectionForDescriptor(BRdbms, BRdbmsHistoryExport, boolean, Context)} method
   * instead.
   *
   * @deprecated since Niagara 4.15 - use {@link #getConnectionForDescriptor(BRdbms, BRdbmsHistoryExport, boolean, Context)}
   */
  @Deprecated
  public BPassword getPassword(BRdbms database, BRdbmsHistoryExport descriptor)
  {
    BPassword password = descriptor.getPassword();
    if ((password != null) && (!password.equals(BPassword.DEFAULT)))
    {
      return password;
    }

    password = database.getPassword();
    if ((password != null) && (!password.equals(BPassword.DEFAULT)))
      return password;

    return BPassword.DEFAULT;
  }

  /**
   * Get a JDBC {@link Connection} to use for processing the given {@link BRdbmsHistoryExport}
   * descriptor, which should live within the given {@link BRdbms} database. If the given
   * descriptor has valid credentials specified in its {@link BRdbmsHistoryExport#userName}
   * and {@link BRdbmsHistoryExport#password} properties, then those credentials will be used to
   * create the returned Connection (via the given database's
   * {@link BRdbms#getConnection(String, BPassword)} method). If the given descriptor does not
   * have valid credentials specified, then the given requestPrivileged argument will decide whether
   * the returned Connection from the given database will be obtained from
   * {@link BRdbms#getPrivilegedConnection(Context)} (for DDL operations) or
   * {@link BRdbms#getNonPrivilegedConnection(Context)} (for DML operations).
   * <p>
   * Callers of this method that use the Connection internally may want to wrap the call in a
   * doPrivileged() block (and add the entry below), otherwise any calling module in the call stack
   * will need an entry like the following in the module's module-permissions.xml file:
   * <p>
   * <pre>{@code
   * <req-permission>
   *   <name>RDB</name>
   *   <purposeKey>This module needs to access JDBC Connections from RDB drivers</purposeKey>
   * </req-permission>
   * }</pre>
   * <p>
   *
   * @param database A running database that contains the given descriptor, which will be utilized
   *                 to compute the returned JDBC Connection.
   * @param descriptor The descriptor for which the returned JDBC Connection will be used for
   *                   processing
   * @param requestPrivileged Only used if the given descriptor does not specify its own valid
   *                          Connection credentials, when true, informs the database to return
   *                          a privileged Connection for DDL operations. When false, informs the
   *                          database to return a non-privileged Connection for DML operations.
   * @param cx The Context associated with this connection request.
   * @return An active {@link Connection} to use for processing operations against the given
   * descriptor. This method may return null if there is a misconfiguration of the connection
   * settings.
   * @throws SQLException if an exception occurs while attempting to establish a JDBC Connection.
   *
   * @since Niagara 4.15
   */
  public static Connection getConnectionForDescriptor(BRdbms database, BRdbmsHistoryExport descriptor,
                                                      boolean requestPrivileged, Context cx)
    throws SQLException
  {
    // If the given descriptor has a valid userName and password entered for it, then use those
    // credentials for the returned Connection
    String descriptorUserName = descriptor.getUserName();
    if (descriptorUserName != null && !descriptorUserName.isEmpty())
    {
      BPassword descriptorPassword = descriptor.getPassword();
      if (descriptorPassword != null && !descriptorPassword.equals(BPassword.DEFAULT))
      {
        return database.getConnection(descriptorUserName, descriptorPassword);
      }
    }

    // If the given descriptor does not specify valid alternate credentials, then revert to
    // the credentials stored on the database to return the Connection to use for the given
    // descriptor.
    if (requestPrivileged)
    {
      return database.getPrivilegedConnection(cx);
    }

    return database.getNonPrivilegedConnection(cx);
  }

////////////////////////////////////////////////////////////////
// attributes
////////////////////////////////////////////////////////////////

  private static final Logger LOG = Logger.getLogger("rdb");
}
