/*
 * Copyright 2024 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.kitControl.math;

import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.status.BStatus;
import javax.baja.status.BStatusNumeric;
import javax.baja.status.BStatusValue;
import javax.baja.sys.BIcon;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Slot;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.BFormat;
import javax.baja.util.BNameMap;
import javax.baja.util.Lexicon;

/**
 * The {@code BLine} class takes 5 inputs:
 *
 * <ul>
 *   <li>{@code in} - The input signal</li>
 *   <li>{@code x1} - Support point x1, with x1 < x2</li>
 *   <li>{@code f1} - Support point f(x1)</li>
 *   <li>{@code x2} - Support point x2, with x2 > x1</li>
 *   <li>{@code f2} - Support point f(x2)</li>
 * </ul>
 * <p>
 * The line equation is:
 * <p>
 *   {@code y = (m * in) + b}
 * </p>
 * where {@code m} and {@code b} are calculated such that the line intercepts the points
 * ({@code x1}, {@code f1}) and ({@code x2}, {@code f2}).
 * <p>
 * The parameters {@code limitBelow} and {@code limitAbove} determine whether {@code x1} and {@code x2}
 * are also used to limit the input {@code in}. If the limits are used, then this block requires {@code x1 < x2}.
 *
 * <p>
 * Parameters:
 * <ul>
 *   <li>{@code limitBelow} (boolean) default true - If true, limit input {@code in} to be no smaller than {@code x1}</li>
 *   <li>{@code limitAbove} (boolean) default true - If true, limit input {@code in} to be no larger than {@code x2}</li>
 * </ul>
 * </p>
 * <p>
 * Output:
 * <ul>
 *   <li>{@code out} - The output signal</li>
 * </ul>
 *
 * @author Rayan Bouhal on 29 Jul 2024
 * @since Niagara 4.15
 */

@NiagaraType
/*
 inC in BLine is represented by display name x2
 */
@NiagaraProperty(
  name = "inC",
  type = "BStatusNumeric",
  defaultValue = "new BStatusNumeric(0, BStatus.nullStatus)",
  flags = Flags.SUMMARY,
  override = true
)
/*
 inD in BLine is represented by display name y2
 */
@NiagaraProperty(
  name = "inD",
  type = "BStatusNumeric",
  defaultValue = "new BStatusNumeric(0, BStatus.nullStatus)",
  flags = Flags.SUMMARY,
  override = true
)
/*
 in - input signal
 */
@NiagaraProperty(
  name = "in",
  type = "BStatusNumeric",
  defaultValue = "new BStatusNumeric(0, BStatus.nullStatus)",
  flags = Flags.SUMMARY
)
/*
 limitBelow - limiter for x1
 */
@NiagaraProperty(
  name = "limitBelow",
  type = "boolean",
  defaultValue = "true"
)
/*
 limitAbove - limiter for x2
 */
@NiagaraProperty(
  name = "limitAbove",
  type = "boolean",
  defaultValue = "true"
)
/*
 faultCause - used to indicate incorrect limit values
 */
@NiagaraProperty(
  name = "faultCause",
  type = "String",
  defaultValue = "",
  flags = Flags.READONLY | Flags.TRANSIENT
)
public final class BLine
  extends BQuadMath
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $com.tridium.kitControl.math.BLine(528730563)1.0$ @*/
/* Generated Fri Aug 02 14:39:47 EDT 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Property "inC"

  /**
   * Slot for the {@code inC} property.
   * inC in BLine is represented by display name x2
   * @see #getInC
   * @see #setInC
   */
  @Generated
  public static final Property inC = newProperty(Flags.SUMMARY, new BStatusNumeric(0, BStatus.nullStatus), null);

  //endregion Property "inC"

  //region Property "inD"

  /**
   * Slot for the {@code inD} property.
   * inD in BLine is represented by display name y2
   * @see #getInD
   * @see #setInD
   */
  @Generated
  public static final Property inD = newProperty(Flags.SUMMARY, new BStatusNumeric(0, BStatus.nullStatus), null);

  //endregion Property "inD"

  //region Property "in"

  /**
   * Slot for the {@code in} property.
   * in - input signal
   * @see #getIn
   * @see #setIn
   */
  @Generated
  public static final Property in = newProperty(Flags.SUMMARY, new BStatusNumeric(0, BStatus.nullStatus), null);

  /**
   * Get the {@code in} property.
   * in - input signal
   * @see #in
   */
  @Generated
  public BStatusNumeric getIn() { return (BStatusNumeric)get(in); }

  /**
   * Set the {@code in} property.
   * in - input signal
   * @see #in
   */
  @Generated
  public void setIn(BStatusNumeric v) { set(in, v, null); }

  //endregion Property "in"

  //region Property "limitBelow"

  /**
   * Slot for the {@code limitBelow} property.
   * limitBelow - limiter for x1
   * @see #getLimitBelow
   * @see #setLimitBelow
   */
  @Generated
  public static final Property limitBelow = newProperty(0, true, null);

  /**
   * Get the {@code limitBelow} property.
   * limitBelow - limiter for x1
   * @see #limitBelow
   */
  @Generated
  public boolean getLimitBelow() { return getBoolean(limitBelow); }

  /**
   * Set the {@code limitBelow} property.
   * limitBelow - limiter for x1
   * @see #limitBelow
   */
  @Generated
  public void setLimitBelow(boolean v) { setBoolean(limitBelow, v, null); }

  //endregion Property "limitBelow"

  //region Property "limitAbove"

  /**
   * Slot for the {@code limitAbove} property.
   * limitAbove - limiter for x2
   * @see #getLimitAbove
   * @see #setLimitAbove
   */
  @Generated
  public static final Property limitAbove = newProperty(0, true, null);

  /**
   * Get the {@code limitAbove} property.
   * limitAbove - limiter for x2
   * @see #limitAbove
   */
  @Generated
  public boolean getLimitAbove() { return getBoolean(limitAbove); }

  /**
   * Set the {@code limitAbove} property.
   * limitAbove - limiter for x2
   * @see #limitAbove
   */
  @Generated
  public void setLimitAbove(boolean v) { setBoolean(limitAbove, v, null); }

  //endregion Property "limitAbove"

  //region Property "faultCause"

  /**
   * Slot for the {@code faultCause} property.
   * faultCause - used to indicate incorrect limit values
   * @see #getFaultCause
   * @see #setFaultCause
   */
  @Generated
  public static final Property faultCause = newProperty(Flags.READONLY | Flags.TRANSIENT, "", null);

  /**
   * Get the {@code faultCause} property.
   * faultCause - used to indicate incorrect limit values
   * @see #faultCause
   */
  @Generated
  public String getFaultCause() { return getString(faultCause); }

  /**
   * Set the {@code faultCause} property.
   * faultCause - used to indicate incorrect limit values
   * @see #faultCause
   */
  @Generated
  public void setFaultCause(String v) { setString(faultCause, v, null); }

  //endregion Property "faultCause"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BLine.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  @Override
  protected double calculate(BStatusNumeric in1, BStatusNumeric f1, BStatusNumeric in2, BStatusNumeric f2)
  {
    double x = getIn().getValue();
    double x1 = in1.getValue();
    double x2 = in2.getValue();
    boolean limBelow = getLimitBelow();
    boolean limAbove = getLimitAbove();

    if ((limBelow || limAbove) && x1 >= x2)
    {
      return Double.NaN;
    }

    double y1 = f1.getValue();
    double y2 = f2.getValue();
    double m = (y2 - y1) / (x2 - x1);
    double b = y2 - m * x2;
    double xLim = limBelow ? Math.max(x1, x) : x;
    xLim = limAbove ? Math.min(x2, xLim) : xLim;

    return m * xLim + b;
  }

  @Override
  public void onExecute(BStatusValue o, Context cx)
  {
    super.onExecute(o, cx);
    boolean invalidPoints = (getLimitBelow() || getLimitAbove()) && getInA().getValue() >= getInC().getValue();
    setFaultCause(invalidPoints ? lex.getText("line.invalidPoints") : "");

    if (invalidPoints)
    {
      o.setStatus(o.getStatus().getBits() | BStatus.FAULT);
    }
    else
    {
      o.setStatus(o.getStatus().getBits() & ~BStatus.FAULT);
    }
  }

  @Override
  public int minInputs()
  {
    return 4;
  }

  @Override
  public String getDisplayName(Slot slot, Context cx)
  {
    // check displayNames
    Property p = getProperty("displayNames");

    if (p != null && p.getType().is(BNameMap.TYPE))
    {
      BNameMap nameMap = (BNameMap)get(p);
      BFormat format = nameMap.get(slot.getName());
      if (format != null)
      {
        return format.format((Object)null, cx);
      }
    }
    String text = lex.get("line." + slot.getName());
    if (text != null)
    {
      return text;
    }
    // use default
    return slot.getDefaultDisplayName(cx);
  }

  /**
   * Get the icon.
   */
  public BIcon getIcon()
  {
    return ICON;
  }

  private static final BIcon ICON = BIcon.std("control/util/xy.png");
  private static final Lexicon lex = Lexicon.make(BLine.class);
}
