/*
 * Copyright 2024 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.kitControl.math;

import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.status.BStatus;
import javax.baja.status.BStatusNumeric;
import javax.baja.status.BStatusValue;
import javax.baja.sys.BDouble;
import javax.baja.sys.BIcon;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.Lexicon;

/**
 * The {@code BLimiter} class limits the range of an input signal using two limits:
 * <ul>
 *   <li>{@code inMax} - The upper limit of input</li>
 *   <li>{@code inMin} - The lower limit of input</li>
 * </ul>
 * Output Equation:
 * <p>
 *   y = min(inMax, max(inMin, in))
 * </p>
 * <p>
 *   where 'in' is the input signal, 'inMax' is the upper limit, and 'inMin' is the lower limit.
 * </p>
 *
 * @author Rayan Bouhal on 16 Jul 2024
 * @since Niagara 4.15
 */
@NiagaraType
/*
 inMin - Lower limit of input
 */
@NiagaraProperty(
  name = "inMin",
  type = "BDouble",
  defaultValue = "BDouble.make(0)"
)
/*
 inMax - Upper limit of input
 */
@NiagaraProperty(
  name = "inMax",
  type = "BDouble",
  defaultValue = "BDouble.make(0)"
)
/*
 faultCause - used to indicate incorrect limit values
 */
@NiagaraProperty(
  name = "faultCause",
  type = "String",
  defaultValue = "",
  flags = Flags.READONLY | Flags.TRANSIENT
)

public class BLimiter
  extends BUnaryMath
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $com.tridium.kitControl.math.BLimiter(2376724833)1.0$ @*/
/* Generated Tue Jul 16 11:28:30 EDT 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Property "inMin"

  /**
   * Slot for the {@code inMin} property.
   * inMin - Lower limit of input
   * @see #getInMin
   * @see #setInMin
   */
  @Generated
  public static final Property inMin = newProperty(0, BDouble.make(0).as(BDouble.class).getDouble(), null);

  /**
   * Get the {@code inMin} property.
   * inMin - Lower limit of input
   * @see #inMin
   */
  @Generated
  public double getInMin() { return getDouble(inMin); }

  /**
   * Set the {@code inMin} property.
   * inMin - Lower limit of input
   * @see #inMin
   */
  @Generated
  public void setInMin(double v) { setDouble(inMin, v, null); }

  //endregion Property "inMin"

  //region Property "inMax"

  /**
   * Slot for the {@code inMax} property.
   * inMax - Upper limit of input
   * @see #getInMax
   * @see #setInMax
   */
  @Generated
  public static final Property inMax = newProperty(0, BDouble.make(0).as(BDouble.class).getDouble(), null);

  /**
   * Get the {@code inMax} property.
   * inMax - Upper limit of input
   * @see #inMax
   */
  @Generated
  public double getInMax() { return getDouble(inMax); }

  /**
   * Set the {@code inMax} property.
   * inMax - Upper limit of input
   * @see #inMax
   */
  @Generated
  public void setInMax(double v) { setDouble(inMax, v, null); }

  //endregion Property "inMax"

  //region Property "faultCause"

  /**
   * Slot for the {@code faultCause} property.
   * faultCause - used to indicate incorrect limit values
   * @see #getFaultCause
   * @see #setFaultCause
   */
  @Generated
  public static final Property faultCause = newProperty(Flags.READONLY | Flags.TRANSIENT, "", null);

  /**
   * Get the {@code faultCause} property.
   * faultCause - used to indicate incorrect limit values
   * @see #faultCause
   */
  @Generated
  public String getFaultCause() { return getString(faultCause); }

  /**
   * Set the {@code faultCause} property.
   * faultCause - used to indicate incorrect limit values
   * @see #faultCause
   */
  @Generated
  public void setFaultCause(String v) { setString(faultCause, v, null); }

  //endregion Property "faultCause"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BLimiter.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  /**
   * Executes limiting on the input signal to confine it within a specific range.
   *
   * @param o  the changed status value to update.
   * @param cx the context in which the method is executed.
   */
  @Override
  public void onExecute(BStatusValue o, Context cx)
  {
    BStatusNumeric inA = getInA();
    BStatus sa = inA.getStatus();
    BStatusNumeric out = (BStatusNumeric)o;

    //  Update output
    if (sa.isNull())
    {
      out.setValue(Double.NaN);
      out.setStatus(BStatus.nullStatus);
    }
    else if(getInMax() < getInMin())
    {
      out.setValue(Double.NaN);
      setFaultCause(INVALID_LIMITS);
      out.setStatus(BStatus.fault);
    }
    else
    {
      out.setValue(calculate(inA.getValue()));
      out.setStatus(propagate(sa));
      setFaultCause("");
    }
  }

  /**
   * Calculates the Double output based on the input value and limiter parameters.
   *
   * @param a the input value to evaluate.
   * @return calculated double output, or a faultCause if inMax < inMin
   */
  @Override
  protected double calculate(double a)
  {
    return Math.min(getInMax(), Math.max(getInMin(), a));
  }

  /**
   * Get the icon.
   */
  public BIcon getIcon()
  {
    return ICON;
  }

  private static final BIcon ICON = BIcon.std("control/logic/limiter.png");
  public static final String INVALID_LIMITS = Lexicon.make(BLimiter.class).getText("limiter.invalidLimits");
}
