/*
 * Copyright 2024 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.kitControl.math;


import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.sys.BIcon;
import javax.baja.sys.Clock;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

/**
 * BDerivative approximates the transfer function between the input (in) and
 * the output (out) by calculating the instantaneous rate of change:
 * <pre> out = k * (in(t) - in(t-deltaT))/deltaT </pre>
 * <p>
 * If k=0, the block reduces to y=0.
 * <p>
 * <p>
 * Where:
 * <br>
 * out = Approximation of the derivative d(in)/dt
 * <br>
 * k = Connector for the gain (constant)
 * <br>
 * t = Time constant (T>0 required; T=0 is ideal derivative block)
 * <br>
 * in = Connector of input signals to be differentiated
 *
 * @author Rayan Bouhal on 24 Jun 2024
 * @since Niagara 4.15
 */

@NiagaraType
/*
 The constant value of the gain (k).
 */
@NiagaraProperty(
  name = "k",
  type = "double",
  defaultValue = "1.0",
  flags = Flags.SUMMARY
)
public class BDerivative
  extends BUnaryMath
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $com.tridium.kitControl.math.BDerivative(2568517624)1.0$ @*/
/* Generated Mon Jun 24 11:56:24 EDT 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Property "k"

  /**
   * Slot for the {@code k} property.
   * The constant value of the gain (k).
   * @see #getK
   * @see #setK
   */
  @Generated
  public static final Property k = newProperty(Flags.SUMMARY, 1.0, null);

  /**
   * Get the {@code k} property.
   * The constant value of the gain (k).
   * @see #k
   */
  @Generated
  public double getK() { return getDouble(k); }

  /**
   * Set the {@code k} property.
   * The constant value of the gain (k).
   * @see #k
   */
  @Generated
  public void setK(double v) { setDouble(k, v, null); }

  //endregion Property "k"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BDerivative.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  /**
   * Calculation method for the approximate derivative.
   * The first time calculate() is run, 0.0 is returned as
   * the function needs two points in order to return the
   * rate of change.
   * <p>
   * out = k * (in(t) - in(t-deltaT))/deltaT
   */
  @Override
  protected double calculate(double a)
  {
    if (initialTime == 0)
    {
      initialTime = Clock.nanoTicks();
      initialVal = a;
      return 0.0; // Derivative undefined for first call
    }
    else
    {
      long now = Clock.nanoTicks();
      // Convert nanoseconds to seconds
      double deltaT = (now - initialTime) / 1e9d;
      double derivative = getK() * (a - initialVal) / deltaT;
      initialTime = now;
      initialVal = a;
      return derivative;
    }
  }


  /**
   * Get the icon.
   */
  public BIcon getIcon()
  {
    return ICON;
  }


  private static final BIcon ICON = BIcon.std("control/math/derivative.png");
  private long initialTime;
  private double initialVal;
}
