/*
 * Copyright 2024 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.kitControl.logic;

import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.status.BStatus;
import javax.baja.status.BStatusBoolean;
import javax.baja.status.BStatusNumeric;
import javax.baja.status.BStatusValue;
import javax.baja.sys.BDouble;
import javax.baja.sys.BIcon;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.Lexicon;


/**
 * The {@code BHysteresis} class transforms a real input signal into a Boolean output signal based on hysteresis.
 * <p>This class implements a simple hysteresis function with two thresholds: {@code inHigh} and {@code inLow}.
 * The output behaves as follows:
 * <ul>
 *   <li>If the output is currently {@code false} and the input becomes greater than {@code inHigh}, the output
 *       switches to {@code true}.</li>
 *   <li>If the output is currently {@code true} and the input becomes less than {@code inLow}, the output
 *       switches to {@code false}.</li>
 * </ul>
 *
 * @author Rayan Bouhal on 11 Jul 2024
 * @since Niagara 4.15
 */
@NiagaraType
/*
 input val
 */
@NiagaraProperty(
  name = "in",
  type = "BStatusNumeric",
  defaultValue = "new BStatusNumeric(0, BStatus.nullStatus)",
  flags = Flags.SUMMARY | Flags.TRANSIENT
)
/*
 inLow - low threshold value
 */
@NiagaraProperty(
  name = "inLow",
  type = "BDouble",
  defaultValue = "BDouble.make(0)"
)
/*
 inHigh - high threshold value
 */
@NiagaraProperty(
  name = "inHigh",
  type = "BDouble",
  defaultValue = "BDouble.make(0)"
)
/*
 faultCause - used to indicate incorrect threshold values
 */
@NiagaraProperty(
  name = "faultCause",
  type = "String",
  defaultValue = "",
  flags = Flags.READONLY | Flags.TRANSIENT
)


public class BHysteresis
  extends BLogic
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $com.tridium.kitControl.logic.BHysteresis(323650876)1.0$ @*/
/* Generated Tue Jul 16 12:40:17 EDT 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Property "in"

  /**
   * Slot for the {@code in} property.
   * input val
   * @see #getIn
   * @see #setIn
   */
  @Generated
  public static final Property in = newProperty(Flags.SUMMARY | Flags.TRANSIENT, new BStatusNumeric(0, BStatus.nullStatus), null);

  /**
   * Get the {@code in} property.
   * input val
   * @see #in
   */
  @Generated
  public BStatusNumeric getIn() { return (BStatusNumeric)get(in); }

  /**
   * Set the {@code in} property.
   * input val
   * @see #in
   */
  @Generated
  public void setIn(BStatusNumeric v) { set(in, v, null); }

  //endregion Property "in"

  //region Property "inLow"

  /**
   * Slot for the {@code inLow} property.
   * inLow - low threshold value
   * @see #getInLow
   * @see #setInLow
   */
  @Generated
  public static final Property inLow = newProperty(0, BDouble.make(0).as(BDouble.class).getDouble(), null);

  /**
   * Get the {@code inLow} property.
   * inLow - low threshold value
   * @see #inLow
   */
  @Generated
  public double getInLow() { return getDouble(inLow); }

  /**
   * Set the {@code inLow} property.
   * inLow - low threshold value
   * @see #inLow
   */
  @Generated
  public void setInLow(double v) { setDouble(inLow, v, null); }

  //endregion Property "inLow"

  //region Property "inHigh"

  /**
   * Slot for the {@code inHigh} property.
   * inHigh - high threshold value
   * @see #getInHigh
   * @see #setInHigh
   */
  @Generated
  public static final Property inHigh = newProperty(0, BDouble.make(0).as(BDouble.class).getDouble(), null);

  /**
   * Get the {@code inHigh} property.
   * inHigh - high threshold value
   * @see #inHigh
   */
  @Generated
  public double getInHigh() { return getDouble(inHigh); }

  /**
   * Set the {@code inHigh} property.
   * inHigh - high threshold value
   * @see #inHigh
   */
  @Generated
  public void setInHigh(double v) { setDouble(inHigh, v, null); }

  //endregion Property "inHigh"

  //region Property "faultCause"

  /**
   * Slot for the {@code faultCause} property.
   * faultCause - used to indicate incorrect threshold values
   * @see #getFaultCause
   * @see #setFaultCause
   */
  @Generated
  public static final Property faultCause = newProperty(Flags.READONLY | Flags.TRANSIENT, "", null);

  /**
   * Get the {@code faultCause} property.
   * faultCause - used to indicate incorrect threshold values
   * @see #faultCause
   */
  @Generated
  public String getFaultCause() { return getString(faultCause); }

  /**
   * Set the {@code faultCause} property.
   * faultCause - used to indicate incorrect threshold values
   * @see #faultCause
   */
  @Generated
  public void setFaultCause(String v) { setString(faultCause, v, null); }

  //endregion Property "faultCause"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BHysteresis.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/



  /**
   * Executes the transformation from Real to Boolean signal with hysteresis.
   *
   * @param o  the changed status value to update.
   * @param cx the context in which the method is executed.
   */
  @Override
  public void onExecute(BStatusValue o, Context cx)
  {
    BStatusNumeric in = getIn();
    BStatus aStatus = in.getStatus();
    BStatusBoolean out = (BStatusBoolean)o;

    if (aStatus.isNull())
    {
      out.setValue(false);
      out.setStatus(BStatus.nullStatus);
      setFaultCause("");
      return;
    }

    if (getInHigh() < getInLow())
    {
      out.setValue(false);
      out.setStatus(BStatus.fault);
      setFaultCause(INVALID_THRESHOLDS);
      return;
    }

    out.setValue(calculate(in.getValue()));
    out.setStatus(propagate(BStatus.make(aStatus.getBits())));
    setFaultCause("");

    if (getNullOnInactive() && !out.getValue())
    {
      out.setStatusNull(true);
    }
  }


  /**
   * Calculates the Boolean output based on the input value and hysteresis parameters.
   *
   * @param input the input value to evaluate.
   * @return calculated Boolean output.
   */
  protected boolean calculate(double input)
  {
    double inHigh = getInHigh();
    double inLow = getInLow();

    boolean currentOutput;
    if (!previousOutput && input > inHigh)
    {
      currentOutput = true;
    }
    else if (previousOutput && input < inLow)
    {
      currentOutput = false;
    }
    else
    {
      // Maintain previous state
      currentOutput = previousOutput;
    }

    previousOutput = currentOutput;
    return currentOutput;
  }

  /**
   * Get the icon.
   */
  public BIcon getIcon()
  {
    return ICON;
  }

  private static final BIcon ICON = BIcon.std("control/logic/hysteresis.png");
  private boolean previousOutput = false;
  public static final String INVALID_THRESHOLDS = Lexicon.make(BHysteresis.class).getText("hysteresis.invalidThresholds");
}
