/*
 * Copyright (c) 2016. Tridium, Inc. All Rights Reserved.
 */

package javax.baja.security.crypto;

import com.tridium.crypto.core.cert.CertUtils;
import com.tridium.crypto.core.cert.KeyPurpose;
import com.tridium.crypto.core.cert.NCertificateParameters;
import com.tridium.crypto.core.cert.NX509Certificate;
import com.tridium.crypto.core.cert.NX509CertificateEntry;

import javax.baja.nre.security.IKeyPurpose;
import javax.baja.nre.security.IX509Certificate;
import javax.baja.nre.security.IX509CertificateEntry;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.Date;

/**
 * @author Bill Smith on 11/29/2016
 *
 * The X509CertificateFactory can be used for creating basic X509
 * certificates.
 */
public final class X509CertificateFactory
{
  /**
   * Returns an instance of the X509CertificateFactory.
   *
   * @return X509CertificateFactory
   */
  public static X509CertificateFactory getInstance()
  {
    return instance;
  }

  private X509CertificateFactory()
  {
  }

  /**
   * generateSelfSignedCert generates an RSA keypair and a self
   * signed cert that is signed with the generated RSA private key
   *
   * @param alias the alias name used to populate the entry response
   * @param subjectDn the subject distinquished name to put in the cert
   * @param issuerDn the issuer distinguished name to put in the cert
   * @param notBeforeDate the date when the cert becomes valid
   * @param notAfterDate the date when the cert expires
   * @param keyPurpose the general purpose use of the keys and cert
   * @param subAltName an optional subject alternative name
   * @param email an optional email address to embed in the cert
   * @return a certifcate entry generated from provided parameters
   * @throws Exception
   */
  public IX509CertificateEntry generateSelfSignedCert(String alias,
                                                      String subjectDn,
                                                      String issuerDn,
                                                      Date notBeforeDate,
                                                      Date notAfterDate,
                                                      int keySize,
                                                      IKeyPurpose keyPurpose,
                                                      String subAltName,
                                                      String email)
      throws Exception
  {
    NCertificateParameters params = new NCertificateParameters(alias,
                                                               subjectDn,
                                                               issuerDn,
                                                               notBeforeDate,
                                                               notAfterDate,
                                                               keySize,
                                                               (KeyPurpose) keyPurpose,
                                                               subAltName,
                                                               email,
                                                               null);
    return CertUtils.generateSelfSignedCert(params);
  }

  /**
   * generateSelfSignedCert generates a self signed cert that is
   * signed with the provided private key and contains the provided
   * public key
   *
   * @param pair a java KeyPair, the supported types are RSA and ECDSA
   * @param alias the alias name used to populate the entry response
   * @param subjectDn the subject distinquished name to put in the cert
   * @param issuerDn the issuer distinguished name to put in the cert
   * @param notBeforeDate the date when the cert becomes valid
   * @param notAfterDate the date when the cert expires
   * @param keyPurpose the general purpose use of the keys and cert
   * @param subAltName an optional subject alternative name
   * @param email an optional email address to embed in the cert
   * @return a certifcate entry generated from provided parameters
   * @throws Exception
   */
  public IX509CertificateEntry generateSelfSignedCert(KeyPair pair,
                                                      String alias,
                                                      String subjectDn,
                                                      String issuerDn,
                                                      Date notBeforeDate,
                                                      Date notAfterDate,
                                                      IKeyPurpose keyPurpose,
                                                      String subAltName,
                                                      String email)
      throws Exception
  {
    NCertificateParameters params = new NCertificateParameters(alias,
                                                               subjectDn,
                                                               issuerDn,
                                                               notBeforeDate,
                                                               notAfterDate,
                                                               0,
                                                               (KeyPurpose) keyPurpose,
                                                               subAltName,
                                                               email,
                                                               null);
    return CertUtils.generateSelfSignedCert(pair, params);
  }

  /**
   * createCertificate creates an instance of an IX509Certificate
   * from a raw x509 certificate.
   *
   * @param cert a raw x509 certificate
   * @return an IX509Certificate
   */
  public IX509Certificate createCertificate(X509Certificate cert)
  {
    return NX509Certificate.make(cert);
  }

  /**
   * createCertificateEntry creates an instance of an IX509CertificateEntry
   *
   * from a raw certificate chain and private key
   * @param alias the alias name to associate with this entry
   * @param chain the certificate chain to associate with this entry
   * @param key an option private key to associate with this entry
   * @return an IX509CertificateEntry
   */
  public IX509CertificateEntry createCertificateEntry(String alias, X509Certificate[] chain, PrivateKey key)
  {
    return NX509CertificateEntry.make(alias, chain, key);
  }

  private static final X509CertificateFactory instance = new X509CertificateFactory();
}
