/*
 * Copyright 2015 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.nio.charset.StandardCharsets;
import java.util.Objects;
import java.util.Optional;

import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.nre.util.SecurityUtil;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.nre.security.ISecretBytesSupplier;
import com.tridium.nre.security.NiagaraBasicPermission;
import com.tridium.nre.security.SecretBytes;
import com.tridium.nre.security.SecretChars;

/**
 * Base class for password encoders that use reversible encryption algorithms. Note that this does
 * not necessarily mean that that original password can be retrieved. The data under the reversible
 * encryption could be hashed or otherwise irretrievable. To check if the original value can be
 * retrieved, use the isReversible() method.
 *
 * @author Matt Boon
 * @creation February 13, 2015
 * @since Niagara 4.0
 */
@NiagaraType
public abstract class BReversiblePasswordEncoder
  extends BAbstractPasswordEncoder
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.security.BReversiblePasswordEncoder(2979906276)1.0$ @*/
/* Generated Wed Dec 29 19:27:38 CST 2021 by Slot-o-Matic (c) Tridium, Inc. 2012-2021 */

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BReversiblePasswordEncoder.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// Defaults
////////////////////////////////////////////////////////////////

  /**
   * Return the system default reversible encoding type
   */
  public static String getDefaultEncodingType()
  {
    return getDefaultEncodingType(true);
  }

  /**
   * Returns an instance of the system default reversible encoder
   */
  public static BReversiblePasswordEncoder makeDefaultInstance()
    throws Exception
  {
    return (BReversiblePasswordEncoder)makeDefaultInstance(true);
  }

////////////////////////////////////////////////////////////////
// Access
////////////////////////////////////////////////////////////////

  /**
   * Takes secret data and encodes it. The encoding will depend on the
   * particular subclass of BReversiblePasswordEncoder.
   * @throws Exception
   */
  public abstract void encode(SecretBytes secret) throws Exception;

  /**
   * Returns the unencrypted value of the secret data.
   *
   * Callers should take care to ensure that the result is closed when the secret data
   * is no longer needed.  Putting it in a try-with-resources is a good way to do that.
   *
   * Since Niagara 4.6, reversible passwords are encrypted by module specific keys and
   * protected by a permission check. To avoid permissions issues, it is recommended to
   * wrap all calls to getSecretBytes() in a doPrivileged block like this:
   *
   * {@code AccessController.doPrivileged((PrivilegedAction<SecretBytes>)encoder::getSecretBytes)}
   */
  public abstract SecretBytes getSecretBytes() throws Exception;

  /**
   * Validates that the provided secret matches the encoded value.
   *
   * @throws Exception
   */
  public abstract boolean validate(SecretBytes secret) throws Exception;

  /**
   * Transcode a given value (as generated by {@link #getEncodedValue()}). 
   * 
   * @param encodedValue Value to transcode
   * @param key Key to use--if not present, attempt to use the security provider's keyring
   *            instead
   *            
   * @throws Exception
   */
  public abstract void transcode(String encodedValue, Optional<ISecretBytesSupplier> key) throws Exception;

  /**
   * Validates that the provided secret matches the encoded value.
   *
   * @throws Exception
   */
  public final boolean validate(SecretChars password) throws Exception
  {
    try(SecretBytes bytes = password.asSecretBytes(StandardCharsets.UTF_8, false))
    {
      return validate(bytes);
    }
  }

  /**
   * Tell the encoder where it should get the encryption key
   *
   * @param value true if this encoder requires an external encryption key to be supplied for encoding
   * and decoding, or false if the key in the security provider's keyring is to be used.
   */
  public final void setUsesExternalEncryptionKey(boolean value)
  {
    usesExternalEncryptionKey = value;
  }

  /**
   * Return true if this encoder requires an external encryption key to be supplied for encoding
   * and decoding, or return false if the key in the security provider's keyring is to be used.
   */
  public final boolean usesExternalEncryptionKey()
  {
    return usesExternalEncryptionKey;
  }

  /**
   * Return true if this encoder doesn't use an external encryption key, or if it uses one and the
   * parameter's key matches it.
   */
  public final boolean validateExternalEncryptionKey(Optional<ISecretBytesSupplier> key)
  {
    if (key.isPresent())
    {
      return usesExternalEncryptionKey() &&
        getEncryptionKey().isPresent() &&
        SecurityUtil.equals(getEncryptionKey().get(), key.get().get().get());
    }
    else
    {
      return !usesExternalEncryptionKey();
    }
  }

  /**
   * Provide the external encryption key
   */
  public final void setExternalEncryptionKey(SecretBytes value)
  {
    Objects.requireNonNull(value);
    SecurityManager sm = System.getSecurityManager();
    if (sm != null) sm.checkPermission(SET_ENCODING_KEY_PERMISSION);
    this.usesExternalEncryptionKey = true;
    // Not closing the SecretBytes copy here because that would zero-out the byte[]. Calling newCopy
    // ensures value is not already closed. There would be no advantage to closing the SecretBytes
    // copy after cloning the byte[] because the array clone would still be present in memory.
    this.encryptionKey = Optional.of(value.newCopy().get());
  }

////////////////////////////////////////////////////////////////
// BAbstractPasswordEncoder
////////////////////////////////////////////////////////////////

  /**
   * Returns true if the original password can be recovered from the encoded password.
   * Returns false if the password encoding is one-way.
   *
   * Note that a BReversiblePasswordEncoder subclass may still override this method to return false.
   * BReversiblePasswordEncoder simply means that it uses reversible encryption. However, the data
   * under the reversible encryption may not be retrievable (e.g. hashed).
   */
  @Override
  public boolean isReversible()
  {
    return true;
  }

  @Override
  public final void encode(SecretChars password) throws Exception
  {
    try(SecretBytes secretBytes = password.asSecretBytes(StandardCharsets.UTF_8, false))
    {
      encode(secretBytes);
    }
  }

  @Override
  public final SecretChars getSecretChars() throws Exception
  {
    return SecretChars.fromSecretBytes(getSecretBytes(), StandardCharsets.UTF_8, true);
  }

  @Override
  public final String getValue() throws Exception
  {
    try(SecretBytes valueBytes = getSecretBytes())
    {
      return new String(valueBytes.get(), StandardCharsets.UTF_8);
    }
  }

////////////////////////////////////////////////////////////////
// Protected Implementation
////////////////////////////////////////////////////////////////

  final Optional<byte[]> getEncryptionKey()
  {
    return encryptionKey;
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  private Optional<byte[]> encryptionKey = Optional.empty();
  private boolean usesExternalEncryptionKey = false;

  private static final NiagaraBasicPermission SET_ENCODING_KEY_PERMISSION = new NiagaraBasicPermission("SET_ENCODING_KEY");
}
