/*
 * Copyright 2024 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.baja.nre.annotations.AgentOn;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.nre.util.ByteArrayUtil;
import javax.baja.nre.util.TextUtil;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.nre.auth.Pbkdf2;
import com.tridium.nre.security.AesPbkdf2AlgorithmBundle;
import com.tridium.nre.security.EncryptionAlgorithmBundle;
import com.tridium.nre.security.KeyDerivationAlgorithmBundle;
import com.tridium.nre.security.PBEEncodingKey;
import com.tridium.nre.security.SecretBytes;
import com.tridium.nre.security.SecretChars;

/**
 * This password encoder first hashes the password using PBKDF2 with HMAC SHA-256, then
 * encrypts the result using AES-256.
 *
 * @author Melanie Coggan on 2024-07-05
 * @since Niagara 4.15
 */
@NiagaraType(
  agent = @AgentOn(
    types = "baja:Password"
  )
)
public final class BAes256Pbkdf2HmacSha256PasswordEncoder
  extends BAes256PasswordEncoder
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.security.BAes256Pbkdf2HmacSha256PasswordEncoder(2040504243)1.0$ @*/
/* Generated Fri Jul 05 16:25:45 EDT 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BAes256Pbkdf2HmacSha256PasswordEncoder.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

//region BAbstractPasswordEncoder
  @Override
  public String getEncodingType()
  {
    return ENCODING_TYPE;
  }

  @Override
  public boolean isReversible()
  {
    // This encoder extends BReversiblePasswordEncoder - however, BReversiblePasswordEncoder is used
    // as meaning "encrypted" rather than reversible. isReversible() is generally used as meaning
    // actually reversible.
    // This is why we have a BReversiblePasswordEncoder that has isReversible return false(). It is
    // encrypted, and we want to use our existing encryption/decryption functionality, but we can't
    // actually retrieve the password's original value.
    return false;
  }
//endregion BAbstractPasswordEncoder


//region Transcoding

  /**
   * Returns the version of this encoder that corresponds to the provided AesAlgorithmBundle version.
   * This encoder was created when the AesAlgorithmBundles were already at version 2, so version 1 of
   * this encoder corresponds to version 2 of the Aes encoders.
   * @param aesVersion The version of AesAlgorithmBundle to compare to.
   * @return the version of this encoder that corresponds to the provided AesAlgorithmBundle version
   */
  public static String getVersionFromAesEncoderVersion(String aesVersion)
  {
    String aesPbkdf2Version = AES_TO_AES_PBKDF2_VERSION_MAP.get(aesVersion);
    return aesPbkdf2Version == null ?
      ALGORITHM_BUNDLE.getAlgorithmVersion() :
      aesPbkdf2Version;
  }
//endregion Transcoding


//region BAbstractAes256PasswordEncoder
  @Override
  public void encode(SecretBytes passwordBytes)
    throws Exception
  {
    String hashData;
    if (isHashed(passwordBytes))
    {
      // The data is already hashed, so we just need to encrypt it
      super.encode(passwordBytes);
    }
    else
    {
      // The data is not already hashed, so we need to hash it first and then encrypt it
      // Encode PBKDF2 part
      KeyDerivationAlgorithmBundle pbkdf2AlgorithmBundle = ALGORITHM_BUNDLE.getPbkdf2AlgorithmBundle();
      byte[] salt = new byte[SALT_LENGTH];
      secureRandom.nextBytes(salt);
      String salt1 = ByteArrayUtil.toHexString(salt);
      int iterationCount = ITERATION_COUNT;
      try (SecretChars passwordChars = SecretChars.fromSecretBytes(passwordBytes))
      {
        byte[] hashedPassword = Pbkdf2.deriveKey(salt, iterationCount, passwordChars.get(), pbkdf2AlgorithmBundle);
        hashData = pbkdf2AlgorithmBundle.encode(new String[]{ salt1, String.valueOf(iterationCount), TextUtil.bytesToHexString(hashedPassword) });

        // Encode AES part
        try (SecretBytes hashBytes = SecretBytes.fromString(hashData))
        {
          super.encode(hashBytes);
        }
      }
    }
  }

  @Override
  public boolean validate(SecretBytes secret)
    throws Exception
  {
    BPbkdf2HmacSha256PasswordEncoder pbkdf2Encoder = new BPbkdf2HmacSha256PasswordEncoder();
    pbkdf2Encoder.parse(getValue());
    try (SecretChars secretChars = SecretChars.fromSecretBytes(secret))
    {
      return pbkdf2Encoder.validate(secretChars);
    }
  }

  @Override
  protected EncryptionAlgorithmBundle getAlgorithmBundle()
  {
    return ALGORITHM_BUNDLE;
  }
//endregion BAbstractAes256PasswordEncoder


//region Helpers
  private static boolean isHashed(SecretBytes passwordBytes)
  {
    // We're checking to see if the data starts with [pbkdf2-sha256.1], which is the type of hash
    // this encoder supports.
    return ByteArrayUtil.equals(HASHED_START, Arrays.copyOf(passwordBytes.get(), HASHED_START.length));
  }

  /**
   * Generate a fake encoder. This is used during the login process to ensure that logins with valid users
   * and invalid users take the same amount of time.
   * @param userName The username for which to generate a fake encoder
   * @return A fake encoder that will always fail validation.
   */
  public static BAes256Pbkdf2HmacSha256PasswordEncoder makeFake(String userName)
  {
    BPbkdf2HmacSha256PasswordEncoder fakePbkdf2 = BPbkdf2HmacSha256PasswordEncoder.makeFake(userName);
    BAes256Pbkdf2HmacSha256PasswordEncoder fakeAesPbkdf2 = new BAes256Pbkdf2HmacSha256PasswordEncoder();

    byte[] iv = new byte[IV_LENGTH];
    fakeAesPbkdf2.secureRandom.nextBytes(iv);
    fakeAesPbkdf2.iv = TextUtil.bytesToHexString(iv);
    try
    {
      fakeAesPbkdf2.encode(fakePbkdf2.getEncodedValue());
    }
    catch (Exception e)
    {
      // This shouldn't ever happen; we're encrypting the data using the key ring rather than
      // trying to decrypt it, and we should always support AES. But just in case.
      throw new UnsupportedOperationException(e.getLocalizedMessage());
    }

    return fakeAesPbkdf2;
  }

  /**
   * Returns the salt that was used while hashing the password.
   * @return The salt as a hex string
   * @throws Exception If the password cannot be decrypted, or if the decrypted value is malformed
   * and the salt cannot be parsed.
   */
  public String getSalt()
    throws Exception
  {
    BPbkdf2HmacSha256PasswordEncoder pbkdf2Encoder = new BPbkdf2HmacSha256PasswordEncoder();
    pbkdf2Encoder.parse(getValue());
    return pbkdf2Encoder.getSalt();
  }

  /**
   * Returns the iteration count that was used while hashing the password.
   * @return The iteration count
   * @throws Exception If the password cannot be decrypted, or if the decrypted value is malformed
   * and the iteration count cannot be parsed.
   */
  public int getIterationCount()
    throws Exception
  {
    BPbkdf2HmacSha256PasswordEncoder pbkdf2Encoder = new BPbkdf2HmacSha256PasswordEncoder();
    pbkdf2Encoder.parse(getValue());
    return pbkdf2Encoder.getIterationCount();
  }

  /**
   * Returns the hashed password without the salt and iteration count.
   * @return The hashed password as a hex string
   * @throws Exception If the password cannot be decrypted, or if the decrypted value is malformed
   * and the key cannot be parsed.
   */
  public byte[] getKey()
    throws Exception
  {
    BPbkdf2HmacSha256PasswordEncoder pbkdf2Encoder = new BPbkdf2HmacSha256PasswordEncoder();
    pbkdf2Encoder.parse(getValue());
    return pbkdf2Encoder.getKey();
  }
//endregion Helpers


//region Constants
  public static final AesPbkdf2AlgorithmBundle ALGORITHM_BUNDLE = AesPbkdf2AlgorithmBundle.make(256);
  public static final String ENCODING_TYPE = ALGORITHM_BUNDLE.getAlgorithmName();

  private static final int ITERATION_COUNT = PBEEncodingKey.DEFAULT_VALIDATION_ITERATION_COUNT;
  private static final int SALT_LENGTH = 16;
  private static final int IV_LENGTH = 16;

  private static final byte[] HASHED_START = ("[" + BPbkdf2HmacSha256PasswordEncoder.ENCODING_TYPE + "]").getBytes(StandardCharsets.UTF_8);

  private static final Map<String, String> AES_TO_AES_PBKDF2_VERSION_MAP;
  static
  {
    Map<String, String> versionMap = new HashMap<>();
    versionMap.put("2", "1");
    AES_TO_AES_PBKDF2_VERSION_MAP = Collections.unmodifiableMap(versionMap);
  }
//endregion Constants


//region Fields
  private final SecureRandom secureRandom = new SecureRandom();
//endregion Fields
}
