/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import javax.baja.sys.Action;
import javax.baja.sys.BComplex;
import javax.baja.sys.BComponent;
import javax.baja.sys.BObject;
import javax.baja.sys.BValue;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Slot;
import javax.baja.sys.SlotCursor;
import javax.baja.sys.Topic;

/**
 * <p>
 * AccessSlotCursor wraps a SlotCursor to provide access
 * control checking.  It automatically skips hidden slots,
 * properties which the user doesn't have read access on, 
 * and actions which the user doesn't have invoke access on.
 * Slot permissions are check as follows:
 * </p>
 * <table border='1' summary="Slot types and their associated permission checks">
 * <tr><td>Operator Non-BComponent Properties</td><td>operatorRead</td></tr>
 * <tr><td>Admin Non-BComponent Properties</td><td>adminRead</td></tr>
 * <tr><td>Operator BComponent Properties</td><td>operatorRead on component</td></tr>
 * <tr><td>Admin BComponent Properties</td><td>operatorRead on component</td></tr>
 * <tr><td>Operator Actions</td><td>operatorInvoke</td></tr>
 * <tr><td>Admin Actions</td><td>adminInvoke</td></tr>
 * <tr><td>Operator Topics</td><td>operatorRead</td></tr>
 * <tr><td>Admin Topics</td><td>adminRead</td></tr>
 * </table>
 *
 * @author    Brian Frank
 * @creation  1 Apr 02
 * @version   $Revision: 7$ $Date: 6/11/07 12:41:24 PM EDT$
 * @since     Baja 1.0 
 */
public final class AccessSlotCursor implements SlotCursor<Slot>
{

////////////////////////////////////////////////////////////////
// Factory
////////////////////////////////////////////////////////////////

  /**
   * Create an AccessSlotCursor which wraps the specified cursor
   * and uses the permissions of the context's user.
   */
  public static AccessSlotCursor make(SlotCursor<? extends Slot> c, Context context)
  {
    return new AccessSlotCursor(c, context);
  }

////////////////////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////////////////////  

  /**
   * Private constructor.
   */
  private AccessSlotCursor(SlotCursor<? extends Slot> c, Context context)
  {
    this.c = c;
    this.context = context;
  }

////////////////////////////////////////////////////////////////
// Cursor
////////////////////////////////////////////////////////////////

  @Override
  public void close()
  {
    c.close();
  }

  /**
   * Get the context for this cursor.
   */
  @Override
  public Context getContext()
  {
    return context;
  }

  /**
   * Advance the cursor to the next slot the user 
   * has permission to access.
   */
  @Override
  public boolean next()
  {
    while(c.next()) { if (has()) return true; }
    return false;
  }
  
  /**
   * Advance the cursor to the next non-primitive property 
   * the user has permission to access.
   */
  @Override
  public boolean nextObject()
  {
    while(c.nextObject()) { if (has()) return true; }
    return false;
  }

  /**
   * Advance the cursor to the next BComponent property 
   * the user has permission to access.
   */
  @Override
  public boolean nextComponent()
  {
    while(c.nextComponent()) { if (has()) return true; }
    return false;
  }
  
  /**
   * Advance the cursor to the next property of the specified
   * type which the user has permission to access.
   */
  @Override
  public boolean next(Class<?> cls)
  {
    while(c.next(cls)) { if (has()) return true; }
    return false;
  }

  @Override
  public final BObject target() { return c.target(); }
  @Override
  public final Slot slot() { return c.slot(); }
  @Override
  public final Property property() { return c.property(); }
  @Override
  public final int getTypeAccess() { return c.getTypeAccess(); }
  @Override
  public BValue get() { return c.get(); }
  @Override
  public final boolean getBoolean() { return c.getBoolean(); }
  @Override
  public final int getInt() { return c.getInt(); }
  @Override
  public final long getLong() { return c.getLong(); }
  @Override
  public final float getFloat() { return c.getFloat(); }
  @Override
  public final double getDouble() { return c.getDouble(); }
  @Override
  public final String getString() { return c.getString(); }
  @Override
  public final Action action() { return c.action(); }
  @Override
  public final Topic topic() { return c.topic(); }

////////////////////////////////////////////////////////////////
// Access
////////////////////////////////////////////////////////////////  

  /**
   * Return if the user has read permission to current slot.
   */
  private boolean has()
  {
    // get current cursor
    BComplex target = (BComplex)c.target();
    Slot slot = c.slot();
    int flags = target.getFlags(slot);
    
    // automatically skip hidden
    if ((flags & Flags.HIDDEN) != 0) return false;
    
    // if we don't have a user to check, 
    // then give access to everything else
    if (context == null) return true;

    // check for component property in which
    // case we use opRead on component itself
    if (slot.isProperty() && 
        c.getTypeAccess() == Slot.BOBJECT_TYPE &&
        (c.get() instanceof BComponent))
    {
      BComponent component = c.get().as(BComponent.class);
      return permissionsFor(component).has(BPermissions.operatorRead);
    }
    
    // otherwise we check the slot...
    
    // map to component
    BComponent component;
    if (target.isComponent()) 
      component = target.asComponent();
    else
      component = target.asComplex().getParentComponent();
    
    // get permissions user has on component
    BPermissions p = permissionsFor(component);
    
    // if an action must have invoke, otherwise must have read
    boolean op = (flags & Flags.OPERATOR) != 0;
    if (slot.isAction())
    {
      if (op)
        return p.has(BPermissions.OPERATOR_INVOKE);
      else
        return p.has(BPermissions.ADMIN_INVOKE);
    }
    else
    {
      if (op)
        return p.has(BPermissions.OPERATOR_READ);
      else
        return p.has(BPermissions.ADMIN_READ);
    }
  }

  /**
   * Get permission for target.
   */
  private BPermissions permissionsFor(BComponent component)
  {
    // if there is no mounted component ancestor then give all permissions
    if (component == null || context == null) 
      return BPermissions.all;
    
    // return user permissions
    return component.getPermissions(context);
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////  

  SlotCursor<? extends Slot> c;
  Context context;
}
