/*
 * Copyright 2003 Tridium, Inc. All Rights Reserved.
 */

package javax.baja.job;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.concurrent.ForkJoinPool;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.baja.naming.BOrd;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraAction;
import javax.baja.nre.annotations.NiagaraTopic;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.spy.SpyWriter;
import javax.baja.sys.Action;
import javax.baja.sys.BComponent;
import javax.baja.sys.BIService;
import javax.baja.sys.BIcon;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.ServiceNotFoundException;
import javax.baja.sys.Sys;
import javax.baja.sys.Topic;
import javax.baja.sys.Type;
import javax.baja.util.BIRestrictedComponent;
import javax.baja.util.BNotification;

import com.tridium.nre.util.NreForkJoinWorkerThreadFactory;
import com.tridium.sys.service.ServiceManager;
import com.tridium.sys.station.BStationSaveJob;

/**
 * BJobService is used to manage all the {@link BJob} instances in a station VM.
 * Refer to {@link BJob} class header for details.
 *
 * @author Brian Frank on 30 Apr 03
 * @since Baja 1.0
 */
@NiagaraType
@NiagaraAction(
  name = "submitAction",
  parameterType = "BJob",
  defaultValue = "new BStationSaveJob()",
  returnType = "BOrd",
  flags = Flags.HIDDEN
)
@NiagaraTopic(
  name = "notification",
  eventType = "BNotification",
  flags = Flags.HIDDEN
)
public class BJobService
  extends BComponent
  implements BIService, BIJobService, BIRestrictedComponent
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.job.BJobService(2527441570)1.0$ @*/
/* Generated Thu Jun 02 14:30:01 EDT 2022 by Slot-o-Matic (c) Tridium, Inc. 2012-2022 */

  //region Action "submitAction"

  /**
   * Slot for the {@code submitAction} action.
   * @see #submitAction(BJob parameter)
   */
  @Generated
  public static final Action submitAction = newAction(Flags.HIDDEN, new BStationSaveJob(), null);

  /**
   * Invoke the {@code submitAction} action.
   * @see #submitAction
   */
  @Generated
  public BOrd submitAction(BJob parameter) { return (BOrd)invoke(submitAction, parameter, null); }

  //endregion Action "submitAction"

  //region Topic "notification"

  /**
   * Slot for the {@code notification} topic.
   * @see #fireNotification
   */
  @Generated
  public static final Topic notification = newTopic(Flags.HIDDEN, null);

  /**
   * Fire an event for the {@code notification} topic.
   * @see #notification
   */
  @Generated
  public void fireNotification(BNotification event) { fire(notification, event, null); }

  //endregion Topic "notification"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BJobService.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// Factory
////////////////////////////////////////////////////////////////

  /**
   * Get the JobService or throw ServiceNotFoundException.
   */
  public static BIJobService getService()
  {
    try
    {
      return (BIJobService)Sys.getService(TYPE);
    }
    catch (ServiceNotFoundException serviceNotFoundException)
    {
      try
      {
        BOrd ord = BOrd.make("tool:workbench:WbJobService|slot:/");
        return (BIJobService)ord.resolve().get();
      }
      catch (Exception exception)
      {
        throw new ServiceNotFoundException("IJobService", exception);
      }
    }
  }

////////////////////////////////////////////////////////////////
// IJobService
////////////////////////////////////////////////////////////////

  /**
   * Get all the child jobs under this service.
   */
  @Override
  public final BJob[] getJobs()
  {
    return getChildren(BJob.class);
  }

  /**
   * Submit a job and run it.
   *
   * @return {@link BOrd} is the ORD to the BJob that is created
   */
  @Override
  public BOrd submit(BJob job, Context cx)
  {
    return (BOrd)invoke(submitAction, job, cx);
  }

////////////////////////////////////////////////////////////////
// Actions
////////////////////////////////////////////////////////////////

  /**
   * Submit action implementation, do not use directly.
   */
  public BOrd doSubmitAction(BJob job, Context cx)
  {
    return AccessController.doPrivileged((PrivilegedAction<BOrd>)() -> {
      add(job.getType().getTypeName() + '?', job, Flags.TRANSIENT);
      job.doSubmit(cx);
      ServiceManager.houseKeeping(BJobService.this);
      return job.getSlotPathOrd();
    });
  }

////////////////////////////////////////////////////////////////
// IService
////////////////////////////////////////////////////////////////  

  @Override
  public Type[] getServiceTypes()
  {
    return new Type[]{ TYPE };
  }

  @Override
  public void serviceStarted()
  {
    // Default number of threads is twice the number of CPUs in order to handle the blocking I/O
    // paradigm in Niagara.  If jobs used only asynchronous, non-blocking I/O,
    // the number of threads could be reduced back to the number of CPUs to reduce stack
    // requirements.
    int threadsPerCPU = AccessController.doPrivileged((PrivilegedAction<Integer>)() ->
      Integer.getInteger("niagara.job.threadsPerCPU", 2));

    // Allow total override in system.properties for some installations if needed
    int defaultThreads = Runtime.getRuntime().availableProcessors() * threadsPerCPU;
    int threads = AccessController.doPrivileged((PrivilegedAction<Integer>)() ->
      Integer.getInteger("niagara.job.threads", defaultThreads));

    startThreadPool(threads);
  }

  @Override
  public void serviceStopped()
  {
    stopThreadPool();
  }

////////////////////////////////////////////////////////////////
// BIRestrictedComponent
////////////////////////////////////////////////////////////////

  /**
   * Only one allowed to live under the station's BServiceContainer.
   */
  @Override
  public final void checkParentForRestrictedComponent(BComponent parent, Context cx)
  {
    BIRestrictedComponent.checkParentForRestrictedComponent(parent, this);
  }

////////////////////////////////////////////////////////////////
// Presentation
////////////////////////////////////////////////////////////////

  @Override
  public BIcon getIcon()
  {
    return ICON;
  }

////////////////////////////////////////////////////////////////
// Spy
////////////////////////////////////////////////////////////////  

  @Override
  public void spy(SpyWriter out)
    throws Exception
  {
    BJob[] jobs = getJobs();
    out.startTable(true);
    out.trTitle("Jobs", 6);
    out.w("<tr>").th("Name").th("State").th("Progress")
      .th("Start").th("Heartbeat").th("End").w("</tr>").nl();
    for (BJob j : jobs)
    {
      out.tr(j.getName(), j.getJobState(), String.valueOf(j.getProgress()),
        j.getStartTime(), j.getHeartbeatTime(), j.getEndTime()
      );
    }
    out.endTable();

    if (getExecutor() != null)
    {
      out.w("<p>");
      out.startTable(true);
      out.trTitle("Thread Pool", 7);
      out.w("<tr>").th("Current Pool Size").th("Max Pool Size").th("Active").th("Running")
        .th("Submitted").th("Queued").th("Steals")
        .w("</tr>").nl();
      out.tr(
        getExecutor().getPoolSize(),
        getExecutor().getParallelism(),
        getExecutor().getActiveThreadCount(),
        getExecutor().getRunningThreadCount(),
        getExecutor().getQueuedSubmissionCount(),
        getExecutor().getQueuedTaskCount(),
        getExecutor().getStealCount()
      );
      out.endTable();
    }

    super.spy(out);
  }

////////////////////////////////////////////////////////////////
// Thread pool
////////////////////////////////////////////////////////////////

  /**
   * Get the job executor.
   */
  public ForkJoinPool getExecutor()
  {
    return executor;
  }

  protected void startThreadPool(int threads)
  {
    // Create the ForkJoinPool
    executor = new ForkJoinPool(threads, NreForkJoinWorkerThreadFactory.DEFAULT_INSTANCE, exceptionHandler, true);

    // Start the monitoring thread
    long monitorIntervalMs = AccessController.doPrivileged((PrivilegedAction<Long>)() ->
      Long.getLong("niagara.job.thread.monitor.intervalMs", DEFAULT_THREAD_MONITOR_INTERVAL_MS));

    // Avoid creating the monitor thread when not used
    if (monitorIntervalMs <= 0)
    {
      return;
    }

    // NCCB-11375: JobMonitor is diagnostic information only, create as lower priority daemon thread suitable for background task
    monitorWorker = new MonitorWorker(executor, monitorIntervalMs);
    monitorWorker.setPriority(Thread.NORM_PRIORITY - 1);
    monitorWorker.setDaemon(true);
    monitorWorker.start();
  }

  /**
   * Shut down the thread pool and monitor.
   */
  protected void stopThreadPool()
  {
    if (executor == null)
    {
      return;
    }

    // Shut down the pool
    executor.shutdownNow();

    if (monitorWorker == null)
    {
      return;
    }

    // Shut down the monitor thread
    monitorWorker.requestStop();

    try
    {
      // Let worker stop
      monitorWorker.join(2000L);
    }
    catch (InterruptedException e)
    {
      if (LOGGER.isLoggable(Level.FINE))
      {
        LOGGER.log(Level.FINE, "Monitor join interrupted on stop", e);
      }
    }
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  protected ForkJoinPool executor;
  protected MonitorWorker monitorWorker;
  protected UncaughtJobExceptionHandler exceptionHandler = new UncaughtJobExceptionHandler();

  protected static final long DEFAULT_THREAD_MONITOR_INTERVAL_MS = 2000L;

  private static final Logger LOGGER = Logger.getLogger("job.thread.monitor");
  private static final BIcon ICON = BIcon.std("jobService.png");

////////////////////////////////////////////////////////////////
// Thread pool support class
////////////////////////////////////////////////////////////////

  /**
   * Handle ForkJoinPool uncaught exceptions
   */
  protected static class UncaughtJobExceptionHandler
    implements Thread.UncaughtExceptionHandler
  {
    /**
     * Uncaught exception handler for thread pool
     *
     * @param t thread
     * @param e Throwable
     */
    @Override
    public void uncaughtException(Thread t, Throwable e)
    {
      LOGGER.log(Level.SEVERE, "Uncaught exception from thread " + t + "\n" + e);
    }
  }

  /**
   * Monitor the thread pool
   */
  protected static class MonitorWorker
    extends Thread
  {
    public MonitorWorker(ForkJoinPool executor, long monitorIntervalMs)
    {
      super("JobService:MonitorWorker");

      this.executor = executor;
      this.monitorIntervalMs = monitorIntervalMs;
    }

    public void requestStop()
    {
      run = false;
      interrupt();
    }

    @Override
    public void run()
    {
      // Return immediately on improper configuration
      if (monitorIntervalMs <= 0)
      {
        return;
      }

      // Run monitor until stop is requested
      while (run)
      {
        if (LOGGER.isLoggable(Level.FINE))
        {
          String traceMessage =
            String.format(
              "[%d/%d] Active: %d, Running: %d, Submitted: %d, Queued: %d, Steals: %d",
              executor.getPoolSize(),
              executor.getParallelism(),
              executor.getActiveThreadCount(),
              executor.getRunningThreadCount(),
              executor.getQueuedSubmissionCount(),
              executor.getQueuedTaskCount(),
              executor.getStealCount()
            );
          LOGGER.log(Level.FINE, traceMessage);
        }

        try
        {
          Thread.sleep(monitorIntervalMs);
        }
        catch (InterruptedException e)
        {
          if (LOGGER.isLoggable(Level.FINE))
          {
            LOGGER.log(Level.FINE, "MonitorWorker sleep interrupted", e);
          }
        }
      }
    }

    private final ForkJoinPool executor;
    private final long monitorIntervalMs;
    private volatile boolean run = true;
  }
}
