/*
 * Copyright 2024 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.export;

import static javax.baja.bacnet.enums.BBacnetErrorClass.property;
import static javax.baja.bacnet.enums.BBacnetErrorCode.invalidArrayIndex;
import static javax.baja.bacnet.enums.BBacnetErrorCode.other;
import static javax.baja.bacnet.enums.BBacnetErrorCode.unknownProperty;
import static javax.baja.bacnet.enums.BBacnetErrorCode.valueNotInitialized;
import static javax.baja.bacnet.enums.BBacnetErrorCode.valueOutOfRange;
import static javax.baja.bacnet.enums.BBacnetErrorCode.writeAccessDenied;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.macAddress;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.maxBvlcLengthAccepted;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.maxNpduLengthAccepted;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scConnectWaitTimeout;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scDirectConnectAcceptEnable;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scDirectConnectAcceptUris;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scDirectConnectBinding;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scDirectConnectConnectionStatus;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scDirectConnectInitiateEnable;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scDisconnectWaitTimeout;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scFailedConnectionRequests;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scFailoverHubConnectionStatus;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scFailoverHubUri;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scHeartbeatTimeout;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scHubConnectorState;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scHubFunctionAcceptUris;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scHubFunctionBinding;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scHubFunctionConnectionStatus;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scHubFunctionEnable;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scMaximumReconnectTime;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scMinimumReconnectTime;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scPrimaryHubConnectionStatus;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.scPrimaryHubUri;
import static javax.baja.bacnet.enums.BBacnetScConnectionState.disconnectedWithErrors;
import static javax.baja.bacnet.enums.BBacnetScConnectionState.failedToConnect;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeBooleanReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeCharStringReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeConstructedReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeEnumReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeListOfReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeOctetStringReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeReadError;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeUnsignedReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.readArray;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import javax.baja.bacnet.BacnetException;
import javax.baja.bacnet.datatypes.BBacnetArray;
import javax.baja.bacnet.datatypes.BBacnetDateTime;
import javax.baja.bacnet.datatypes.BBacnetHostAddress;
import javax.baja.bacnet.datatypes.BBacnetHostNPort;
import javax.baja.bacnet.datatypes.BBacnetListOf;
import javax.baja.bacnet.datatypes.BBacnetOctetString;
import javax.baja.bacnet.datatypes.BBacnetScDirectConnection;
import javax.baja.bacnet.datatypes.BBacnetScFailedConnectionRequest;
import javax.baja.bacnet.datatypes.BBacnetScHubConnection;
import javax.baja.bacnet.datatypes.BBacnetScHubFunctionConnection;
import javax.baja.bacnet.datatypes.BErrorType;
import javax.baja.bacnet.datatypes.BIBacnetDataType;
import javax.baja.bacnet.enums.BBacnetPropertyIdentifier;
import javax.baja.bacnet.enums.BBacnetScConnectionState;
import javax.baja.bacnet.io.AsnException;
import javax.baja.bacnet.io.ErrorException;
import javax.baja.bacnet.io.ErrorType;
import javax.baja.bacnet.io.PropertyValue;
import javax.baja.net.BInternetAddress;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BBoolean;
import javax.baja.sys.BRelTime;
import javax.baja.sys.BString;
import javax.baja.sys.BValue;
import javax.baja.sys.Context;
import javax.baja.sys.LocalizableRuntimeException;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.BUuid;
import javax.baja.web.BWebService;

import com.tridium.bacnet.asn.AsnUtil;
import com.tridium.bacnet.asn.NErrorType;
import com.tridium.bacnet.asn.NReadPropertyResult;
import com.tridium.bacnet.stack.link.sc.BAbstractConnectionManager;
import com.tridium.bacnet.stack.link.sc.BHubConnector;
import com.tridium.bacnet.stack.link.sc.BHubFunction;
import com.tridium.bacnet.stack.link.sc.BNodeSwitch;
import com.tridium.bacnet.stack.link.sc.BScConfiguration;
import com.tridium.bacnet.stack.link.sc.BScLinkLayer;
import com.tridium.bacnet.stack.link.sc.connection.BAbstractConnection;
import com.tridium.bacnet.stack.link.sc.connection.BDirectAcceptingConnection;
import com.tridium.bacnet.stack.link.sc.connection.BDirectInitiatingConnection;
import com.tridium.bacnet.stack.link.sc.connection.BHubAcceptingConnection;
import com.tridium.bacnet.stack.link.sc.connection.BHubInitiatingConnection;
import com.tridium.bacnet.stack.link.sc.message.ScMessageUtil;
import com.tridium.nre.util.IPAddressUtil;

/**
 * BBacnetScPortDescriptor exports a BNetworkPort that contains a BScLinkLayer as a BACnet
 * Network Port object with the Network Type set to SC.
 *
 * @author Sandipan Aich on 9 April 2024
 * @since Niagara 4.15
 */
@NiagaraType
public class BBacnetScPortDescriptor
  extends BBacnetNetworkPortDescriptor
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.export.BBacnetScPortDescriptor(2979906276)1.0$ @*/
/* Generated Fri Oct 25 09:36:46 CDT 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetScPortDescriptor.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  //region BBacnetNetworkPortDescriptor

  @Override
  protected int getNetworkType()
  {
    // To be uncommented once PR 24 is implemented: NCCB-69442
    //return BBacnetNetworkType.SECURE_CONNECT;
    return 64;
  }

  @Override
  protected int[] getOptionalProps()
  {
    return getHubFunction() == null ?
      OPTIONAL_PROPS_WITHOUT_HUB_FUNCTION :
      OPTIONAL_PROPS_WITH_HUB_FUNCTION;
  }

  @Override
  protected boolean isArrayProp(int propId)
  {
    switch (propId)
    {
      case BBacnetPropertyIdentifier.SC_HUB_FUNCTION_ACCEPT_URIS:
      case BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_ACCEPT_URIS:
      case BBacnetPropertyIdentifier.PROPERTY_LIST:
        return true;
      default:
        return false;
    }
  }

  @Override
  protected boolean isListProp(int propId)
  {
    switch (propId)
    {
      case BBacnetPropertyIdentifier.ROUTING_TABLE:
      case BBacnetPropertyIdentifier.SC_HUB_FUNCTION_CONNECTION_STATUS:
      case BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_CONNECTION_STATUS:
      case BBacnetPropertyIdentifier.SC_FAILED_CONNECTION_REQUESTS:
        return true;
      default:
        return false;
    }
  }

  //endregion

  //region ReadProperty

  @Override
  protected PropertyValue readOptionalProperty(int propertyId, int index)
  {
    switch (propertyId)
    {
      case BBacnetPropertyIdentifier.NETWORK_NUMBER:
        return readNetworkNumber();

      case BBacnetPropertyIdentifier.NETWORK_NUMBER_QUALITY:
        return readNetworkNumberQuality();

      case BBacnetPropertyIdentifier.APDU_LENGTH:
        return readApduLength();

      case BBacnetPropertyIdentifier.MAC_ADDRESS:
        return readMacAddress();

      case BBacnetPropertyIdentifier.MAX_BVLC_LENGTH_ACCEPTED:
        return readMaxBvlcLengthAccepted();

      case BBacnetPropertyIdentifier.MAX_NPDU_LENGTH_ACCEPTED:
        return readMaxNpduLengthAccepted();

      case BBacnetPropertyIdentifier.SC_PRIMARY_HUB_URI:
        return readScPrimaryHubUri();

      case BBacnetPropertyIdentifier.SC_FAILOVER_HUB_URI:
        return readScFailoverHubUri();

      case BBacnetPropertyIdentifier.SC_MINIMUM_RECONNECT_TIME:
        return readScMinimumReconnectTime();

      case BBacnetPropertyIdentifier.SC_MAXIMUM_RECONNECT_TIME:
        return readScMaximumReconnectTime();

      case BBacnetPropertyIdentifier.SC_CONNECT_WAIT_TIMEOUT:
        return readScConnectWaitTimeout();

      case BBacnetPropertyIdentifier.SC_DISCONNECT_WAIT_TIMEOUT:
        return readScDisconnectWaitTimeout();

      case BBacnetPropertyIdentifier.SC_HEARTBEAT_TIMEOUT:
        return readScHeartbeatTimeout();

      case BBacnetPropertyIdentifier.SC_HUB_CONNECTOR_STATE:
        return readScHubConnectorState();

      case BBacnetPropertyIdentifier.ROUTING_TABLE:
        return readRoutingTable();

      case BBacnetPropertyIdentifier.SC_PRIMARY_HUB_CONNECTION_STATUS:
        return readScPrimaryHubConnectionStatus();

      case BBacnetPropertyIdentifier.SC_FAILOVER_HUB_CONNECTION_STATUS:
        return readScFailoverHubConnectionStatus();

      case BBacnetPropertyIdentifier.SC_HUB_FUNCTION_ENABLE:
        return readScHubFunctionEnable();

      case BBacnetPropertyIdentifier.SC_HUB_FUNCTION_ACCEPT_URIS:
        return readScHubFunctionAcceptUris(index);

      case BBacnetPropertyIdentifier.SC_HUB_FUNCTION_BINDING:
        return readScHubFunctionBinding();

      case BBacnetPropertyIdentifier.SC_HUB_FUNCTION_CONNECTION_STATUS:
        return readScHubFunctionConnectionStatus();

      case BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_INITIATE_ENABLE:
        return readScDirectConnectInitiateEnable();

      case BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_ACCEPT_ENABLE:
        return readScDirectConnectAcceptEnable();

      case BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_ACCEPT_URIS:
        return readScDirectConnectAcceptUris(index);

      case BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_BINDING:
        return readScDirectConnectBinding();

      case BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_CONNECTION_STATUS:
        return readScDirectConnectConnectionStatus();

      case BBacnetPropertyIdentifier.SC_FAILED_CONNECTION_REQUESTS:
        return readScFailedConnectionRequests();

      default:
        return new NReadPropertyResult(propertyId, new NErrorType(property, unknownProperty));
    }
  }

  //endregion

  //region ReadRange

  @Override
  protected List<? extends BIBacnetDataType> getListElements(int propertyId)
    throws ErrorException
  {
    switch (propertyId)
    {
      case BBacnetPropertyIdentifier.ROUTING_TABLE:
        return getRouterEntries();
      case BBacnetPropertyIdentifier.SC_HUB_FUNCTION_CONNECTION_STATUS:
        // This method would not have been called if getHubFunction returned null.
        return getScHubFunctionConnections(getHubFunction());
      case BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_CONNECTION_STATUS:
        return getScDirectConnections();
      case BBacnetPropertyIdentifier.SC_FAILED_CONNECTION_REQUESTS:
        return getFailedConnections();
    }
    throw new IllegalArgumentException("Invalid propertyId value " + propertyId);
  }

  //endregion

  //region WriteProperty

  @Override
  protected ErrorType writeOptionalProperty(int propertyId, int index, byte[] value, int priority)
    throws BacnetException
  {
    switch (propertyId)
    {
      case BBacnetPropertyIdentifier.NETWORK_NUMBER_QUALITY:
      case BBacnetPropertyIdentifier.APDU_LENGTH:
      case BBacnetPropertyIdentifier.MAC_ADDRESS:
      case BBacnetPropertyIdentifier.MAX_BVLC_LENGTH_ACCEPTED:
      case BBacnetPropertyIdentifier.MAX_NPDU_LENGTH_ACCEPTED:
      case BBacnetPropertyIdentifier.SC_HUB_CONNECTOR_STATE:
      case BBacnetPropertyIdentifier.ROUTING_TABLE:
      case BBacnetPropertyIdentifier.SC_PRIMARY_HUB_CONNECTION_STATUS:
      case BBacnetPropertyIdentifier.SC_FAILOVER_HUB_CONNECTION_STATUS:
      case BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_BINDING:
      case BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_CONNECTION_STATUS:
      case BBacnetPropertyIdentifier.SC_FAILED_CONNECTION_REQUESTS:
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(this + ": attempted to write read-only property " + BBacnetPropertyIdentifier.tag(propertyId));
        }
        return new NErrorType(property, writeAccessDenied);

      case BBacnetPropertyIdentifier.NETWORK_NUMBER:
        return writeNetworkNumber(value);

      case BBacnetPropertyIdentifier.SC_PRIMARY_HUB_URI:
        return writeScPrimaryHubUri(value);

      case BBacnetPropertyIdentifier.SC_FAILOVER_HUB_URI:
        return writeScFailoverHubUri(value);

      case BBacnetPropertyIdentifier.SC_MINIMUM_RECONNECT_TIME:
        return writeScMinimumReconnectTime(value);

      case BBacnetPropertyIdentifier.SC_MAXIMUM_RECONNECT_TIME:
        return writeScMaximumReconnectTime(value);

      case BBacnetPropertyIdentifier.SC_CONNECT_WAIT_TIMEOUT:
        return writeScConnectWaitTimeout(value);

      case BBacnetPropertyIdentifier.SC_DISCONNECT_WAIT_TIMEOUT:
        return writeScDisconnectWaitTimeout(value);

      case BBacnetPropertyIdentifier.SC_HEARTBEAT_TIMEOUT:
        return writeScHeartbeatTimeout(value);

      case BBacnetPropertyIdentifier.SC_HUB_FUNCTION_ENABLE:
        return writeScHubFunctionEnable(value);

      case BBacnetPropertyIdentifier.SC_HUB_FUNCTION_ACCEPT_URIS:
        return writeScHubFunctionAcceptUris(index, value);

      case BBacnetPropertyIdentifier.SC_HUB_FUNCTION_BINDING:
        return writeScHubFunctionBinding();

      case BBacnetPropertyIdentifier.SC_HUB_FUNCTION_CONNECTION_STATUS:
        return writeScHubFunctionConnectionStatus();

      case BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_INITIATE_ENABLE:
        return writeScDirectConnectInitiateEnable(value);

      case BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_ACCEPT_ENABLE:
        return writeScDirectConnectAcceptEnable(value);

      case BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_ACCEPT_URIS:
        return writeScDirectConnectAcceptUris(index, value);

      default:
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(this + ": attempted to write unknown property " + BBacnetPropertyIdentifier.tag(propertyId));
        }
        return new NErrorType(property, unknownProperty);
    }
  }

  //endregion

  //region Validate/Activate Changes

  @Override
  protected void validateSetPendingChange(String name, BValue value, Context context)
  {
    switch (name)
    {
      case NETWORK_NUMBER:
        validateSetNetworkNumber(value, context);
        break;
      case SC_PRIMARY_HUB_URI:
        validateSetPrimaryHubUri(value, context);
        break;
      case SC_FAILOVER_HUB_URI:
        validateSetFailoverHubUri(value, context);
        break;
      case SC_HUB_FUNCTION_ENABLE:
        validateSetHubFunctionEnable(value, context);
        break;
      case SC_DIRECT_CONNECT_INITIATE_ENABLE:
        validateSetDirectConnectInitiateEnable(value, context);
        break;
      case SC_DIRECT_CONNECT_ACCEPT_ENABLE:
        validateSetDirectConnectAcceptEnable(value, context);
        break;
      default:
        throw new LocalizableRuntimeException(
          "bacnet",
          "networkPortDescriptor.pendingChange.unknown",
          new Object[] { name });
    }
  }

  @Override
  public void validateChanges(Context context)
    throws ValidateChangesException
  {
    if (!hasPendingChanges())
    {
      return;
    }

    checkCanReadPendingChanges(context);

    validateNetworkNumber(context);
    validateScPrimaryHubUri(context);
    validateScFailoverHubUri(context);
    validateScHubFunctionEnable(context);
    validateScDirectConnectInitiateEnable(context);
    validateScDirectConnectAcceptEnable(context);
  }

  @Override
  public void activateChanges(Context context)
    throws Exception
  {
    if (!hasPendingChanges())
    {
      return;
    }

    activateNetworkNumber(context);
    activateScPrimaryHubUri(context);
    activateScFailoverHubUri(context);
    activateScHubFunctionEnable(context);
    activateScDirectConnectInitiateEnable(context);
    activateScDirectConnectAcceptEnable(context);
  }

  //endregion

  //region MacAddress

  private PropertyValue readMacAddress()
  {
    byte[] macBytes = getScLink().getMacAddress();
    if (macBytes == null || macBytes.length != 6)
    {
      return makeReadError(macAddress, property, valueNotInitialized);
    }
    return makeOctetStringReadResult(macAddress, macBytes);
  }

  //endregion

  //region MaxBvlcLengthAccepted

  private PropertyValue readMaxBvlcLengthAccepted()
  {
    return makeUnsignedReadResult(maxBvlcLengthAccepted, getScLink().getMaxBvlcLength());
  }

  //endregion

  //region MaxNpduLengthAccepted

  private PropertyValue readMaxNpduLengthAccepted()
  {
    return makeUnsignedReadResult(maxNpduLengthAccepted, getScLink().getMaxNpduLength());
  }

  //endregion

  //region ScPrimaryHubUri

  private PropertyValue readScPrimaryHubUri()
  {
    BString pending = getPendingChange(SC_PRIMARY_HUB_URI, BString.class);
    if (pending != null)
    {
      return makeCharStringReadResult(scPrimaryHubUri, pending.getString());
    }
    return readUri(scPrimaryHubUri, getPrimaryConnection());
  }

  private ErrorType writeScPrimaryHubUri(byte[] value)
    throws AsnException
  {
    String uri = AsnUtil.fromAsnCharacterString(value);
    try
    {
      validateUri(uri, scPrimaryHubUri);
    }
    catch (ValidateChangesException e)
    {
      return e.getError();
    }

    addPendingChange(SC_PRIMARY_HUB_URI, BString.make(uri), BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  private void validateSetPrimaryHubUri(BValue value, Context context)
  {
    checkPendingChangeType(SC_PRIMARY_HUB_URI, value, BString.TYPE);
    checkCanWriteUri(getPrimaryConnection(), context);
  }

  private void validateScPrimaryHubUri(Context context)
    throws ValidateChangesException
  {
    BString change = getPendingChange(SC_PRIMARY_HUB_URI, BString.class);
    if (change != null)
    {
      validateUri(change.getString(), scPrimaryHubUri);
      validateCanWriteUri(getPrimaryConnection(), scPrimaryHubUri, context);
    }
  }

  private void activateScPrimaryHubUri(Context context)
    throws URISyntaxException
  {
    BString change = getPendingChange(SC_PRIMARY_HUB_URI, BString.class);
    if (change != null)
    {
      activateUri(change.getString(), getPrimaryConnection(), context);
    }
  }

  //endregion

  //region ScFailoverHubUri

  private PropertyValue readScFailoverHubUri()
  {
    BString pending = getPendingChange(SC_FAILOVER_HUB_URI, BString.class);
    if (pending != null)
    {
      return makeCharStringReadResult(scFailoverHubUri, pending.getString());
    }
    return readUri(scFailoverHubUri, getFailoverConnection());
  }

  private ErrorType writeScFailoverHubUri(byte[] value)
    throws AsnException
  {
    String uri = AsnUtil.fromAsnCharacterString(value);
    try
    {
      validateUri(uri, scFailoverHubUri);
    }
    catch (ValidateChangesException e)
    {
      return e.getError();
    }

    addPendingChange(SC_FAILOVER_HUB_URI, BString.make(uri), BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  private void validateSetFailoverHubUri(BValue value, Context context)
  {
    checkPendingChangeType(SC_FAILOVER_HUB_URI, value, BString.TYPE);
    checkCanWriteUri(getFailoverConnection(), context);
  }

  private void validateScFailoverHubUri(Context context)
    throws ValidateChangesException
  {
    BString change = getPendingChange(SC_FAILOVER_HUB_URI, BString.class);
    if (change != null)
    {
      validateUri(change.getString(), scFailoverHubUri);
      validateCanWriteUri(getFailoverConnection(), scFailoverHubUri, context);
    }
  }

  private void activateScFailoverHubUri(Context context)
    throws URISyntaxException
  {
    BString change = getPendingChange(SC_FAILOVER_HUB_URI, BString.class);
    if (change != null)
    {
      activateUri(change.getString(), getFailoverConnection(), context);
    }
  }

  //endregion

  //region ScMinimumReconnectTime

  private PropertyValue readScMinimumReconnectTime()
  {
    return readConfigTime(scMinimumReconnectTime, getScConfig().getMinimumReconnectTime().getSeconds());
  }

  private ErrorType writeScMinimumReconnectTime(byte[] value)
    throws AsnException
  {
    return writeConfigTime(BScConfiguration.minimumReconnectTime, value, 1, 600);
  }

  //endregion

  //region ScMaximumReconnectTime

  private PropertyValue readScMaximumReconnectTime()
  {
    return readConfigTime(scMaximumReconnectTime, getScConfig().getMaximumReconnectTime().getSeconds());
  }

  private ErrorType writeScMaximumReconnectTime(byte[] value)
    throws AsnException
  {
    return writeConfigTime(BScConfiguration.maximumReconnectTime, value, 1, 600);
  }

  //endregion

  //region ScConnectWaitTimeout

  private PropertyValue readScConnectWaitTimeout()
  {
    return readConfigTime(scConnectWaitTimeout, getScConfig().getConnectWaitTimeout().getSeconds());
  }

  private ErrorType writeScConnectWaitTimeout(byte[] value)
    throws AsnException
  {
    return writeConfigTime(BScConfiguration.connectWaitTimeout, value, 1, Integer.MAX_VALUE);
  }

  //endregion

  //region ScDisconnectWaitTimeout

  private PropertyValue readScDisconnectWaitTimeout()
  {
    return readConfigTime(scDisconnectWaitTimeout, getScConfig().getDisconnectWaitTimeout().getSeconds());
  }

  private ErrorType writeScDisconnectWaitTimeout(byte[] value)
    throws AsnException
  {
    return writeConfigTime(BScConfiguration.disconnectWaitTimeout, value, 1, Integer.MAX_VALUE);
  }

  //endregion

  //region ScHeartbeatTimeout

  private PropertyValue readScHeartbeatTimeout()
  {
    return readConfigTime(scHeartbeatTimeout, getScConfig().getInitiatingHeartbeatTimeout().getSeconds());
  }

  private ErrorType writeScHeartbeatTimeout(byte[] value)
    throws AsnException
  {
    return writeConfigTime(BScConfiguration.initiatingHeartbeatTimeout, value, 1, Integer.MAX_VALUE);
  }

  //endregion

  //region ScHubConnectorState

  private PropertyValue readScHubConnectorState()
  {
    return makeEnumReadResult(scHubConnectorState, getHubConnector().getState());
  }

  //endregion

  //region ScPrimaryHubConnectionStatus

  private PropertyValue readScPrimaryHubConnectionStatus()
  {
    return readHubConnectionStatus(scPrimaryHubConnectionStatus, getPrimaryConnection());
  }

  private static PropertyValue readHubConnectionStatus(BBacnetPropertyIdentifier propertyIdentifier, BHubInitiatingConnection connection)
  {
    BBacnetScHubConnection scHubConnection = new BBacnetScHubConnection();

    scHubConnection.setConnectionState(connection.getState());
    scHubConnection.setConnectTimestamp(makeDateTime(connection.getLastConnect()));
    scHubConnection.setDisconnectTimestamp(makeDateTime(connection.getLastDisconnect()));

    BErrorType error = connection.getError();
    if (error.getErrorClass() != NOT_USED)
    {
      scHubConnection.setError((BErrorType) connection.getError().newCopy());
      scHubConnection.setErrorDetails(connection.getErrorDetails());
    }

    return makeConstructedReadResult(propertyIdentifier, scHubConnection);
  }

  //endregion

  //region ScFailoverHubConnectionStatus

  private PropertyValue readScFailoverHubConnectionStatus()
  {
    return readHubConnectionStatus(scFailoverHubConnectionStatus, getFailoverConnection());
  }

  //endregion

  //region ScHubFunctionEnable

  private PropertyValue readScHubFunctionEnable()
  {
    BHubFunction hubFunction = getHubFunction();
    if (hubFunction == null)
    {
      return makeReadError(scHubFunctionEnable, property, unknownProperty);
    }

    BBoolean pending = getPendingChange(SC_HUB_FUNCTION_ENABLE, BBoolean.class);
    return makeBooleanReadResult(
      scHubFunctionEnable,
      pending != null ? pending.getBoolean() : hubFunction.getEnabled());
  }

  private ErrorType writeScHubFunctionEnable(byte[] value)
    throws AsnException
  {
    BHubFunction hubFunction = getHubFunction();
    if (hubFunction == null)
    {
      return new NErrorType(property, unknownProperty);
    }

    addPendingChange(
      SC_HUB_FUNCTION_ENABLE,
      BBoolean.make(AsnUtil.fromAsnBoolean(value)),
      BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  private void validateSetHubFunctionEnable(BValue value, Context context)
  {
    checkPendingChangeType(SC_HUB_FUNCTION_ENABLE, value, BBoolean.TYPE);
    checkCanWrite(getHubFunction(), BHubFunction.enabled, context);
  }

  private void validateScHubFunctionEnable(Context context)
    throws ValidateChangesException
  {
    BHubFunction hubFunction = getHubFunction();
    if (hubFunction == null)
    {
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ unknownProperty,
        /* property */ scHubFunctionEnable,
        /* details */ "Port does not support the BACnet/SC hub function.");
    }

    validateCanWrite(hubFunction, BHubFunction.enabled, scHubFunctionEnable, context);
  }

  private void activateScHubFunctionEnable(Context context)
  {
    BBoolean change = getPendingChange(SC_HUB_FUNCTION_ENABLE, BBoolean.class);
    if (change != null)
    {
      // Validation would have failed if getHubFunction returned null
      getHubFunction().setBoolean(BHubFunction.enabled, change.getBoolean(), context);
    }
  }

  //endregion

  //region ScHubFunctionAcceptUris

  private PropertyValue readScHubFunctionAcceptUris(int index)
  {
    BHubFunction hubFunction = getHubFunction();
    if (hubFunction == null)
    {
      return makeReadError(scHubFunctionAcceptUris, property, unknownProperty);
    }

    return readArray(scHubFunctionAcceptUris, index, hubFunction.getAcceptUris());
  }

  private ErrorType writeScHubFunctionAcceptUris(int index, byte[] value)
    throws AsnException
  {
    BHubFunction hubFunction = getHubFunction();
    if (hubFunction == null)
    {
      return new NErrorType(property, unknownProperty);
    }

    return writeAcceptUris(hubFunction.getAcceptUris(), index, value);
  }

  //endregion

  //region ScHubFunctionBinding

  private PropertyValue readScHubFunctionBinding()
  {
    BHubFunction hubFunction = getHubFunction();
    if (hubFunction == null)
    {
      return makeReadError(scHubFunctionBinding, property, unknownProperty);
    }

    return readHttpsPortBinding(scHubFunctionBinding, hubFunction);
  }

  private ErrorType writeScHubFunctionBinding()
  {
    return writeHubFunctionReadOnlyProperty(scHubFunctionBinding);
  }

  //endregion

  //region ScHubFunctionConnectionStatus

  private PropertyValue readScHubFunctionConnectionStatus()
  {
    BHubFunction hubFunction = getHubFunction();
    if (hubFunction == null)
    {
      return makeReadError(scHubFunctionConnectionStatus, property, unknownProperty);
    }

    BBacnetListOf list = new BBacnetListOf(BBacnetScHubFunctionConnection.TYPE);
    for (BBacnetScHubFunctionConnection connection : getScHubFunctionConnections(hubFunction))
    {
      list.addListElement(connection, null);
    }
    return makeListOfReadResult(scHubFunctionConnectionStatus, list);
  }

  private static List<BBacnetScHubFunctionConnection> getScHubFunctionConnections(BHubFunction hubFunction)
  {
    BHubAcceptingConnection[] connections = hubFunction.getConnections().getChildren(BHubAcceptingConnection.class);
    List<BBacnetScHubFunctionConnection> scConnections = new ArrayList<>(connections.length);
    for (BHubAcceptingConnection connection : connections)
    {
      BBacnetScConnectionState connectionState = connection.getState();
      if (!connectionState.equals(failedToConnect))
      {
        BBacnetScHubFunctionConnection scConnection = new BBacnetScHubFunctionConnection();
        scConnection.setConnectionState(connectionState);
        scConnection.setConnectTimestamp(makeDateTime(connection.getLastConnect()));
        scConnection.setDisconnectTimestamp(makeDateTime(connection.getLastDisconnect()));
        scConnection.setPeerAddress(makeHostNPort(connection.getNodeAddress()));
        scConnection.setPeerVmac(connection.getNodeVmac());
        scConnection.setPeerUuid(BBacnetOctetString.make(connection.getNodeUuid().getBytes()));

        if (connectionState.equals(disconnectedWithErrors))
        {
          scConnection.setError((BErrorType) connection.getError().newCopy());
          scConnection.setErrorDetails(connection.getErrorDetails());
        }

        scConnections.add(scConnection);
      }
    }
    return scConnections;
  }

  private ErrorType writeScHubFunctionConnectionStatus()
  {
    return writeHubFunctionReadOnlyProperty(scHubFunctionConnectionStatus);
  }

  //endregion

  //region ScDirectConnectInitiateEnable

  private PropertyValue readScDirectConnectInitiateEnable()
  {
    BBoolean pending = getPendingChange(SC_DIRECT_CONNECT_INITIATE_ENABLE, BBoolean.class);
    return makeBooleanReadResult(
      scDirectConnectInitiateEnable,
      pending != null ? pending.getBoolean() : getNodeSwitch().getInitiateEnabled());
  }

  private ErrorType writeScDirectConnectInitiateEnable(byte[] value)
    throws AsnException
  {
    addPendingChange(
      SC_DIRECT_CONNECT_INITIATE_ENABLE,
      BBoolean.make(AsnUtil.fromAsnBoolean(value)),
      BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  private void validateSetDirectConnectInitiateEnable(BValue value, Context context)
  {
    checkPendingChangeType(SC_DIRECT_CONNECT_INITIATE_ENABLE, value, BBoolean.TYPE);
    checkCanWrite(getNodeSwitch(), BNodeSwitch.initiateEnabled, context);
  }

  private void validateScDirectConnectInitiateEnable(Context context)
    throws ValidateChangesException
  {
    BBoolean pending = getPendingChange(SC_DIRECT_CONNECT_INITIATE_ENABLE, BBoolean.class);
    if (pending != null)
    {
      validateCanWrite(getNodeSwitch(), BNodeSwitch.initiateEnabled, scDirectConnectInitiateEnable, context);
    }
  }

  private void activateScDirectConnectInitiateEnable(Context context)
  {
    BBoolean change = getPendingChange(SC_DIRECT_CONNECT_INITIATE_ENABLE, BBoolean.class);
    if (change != null)
    {
      getNodeSwitch().setBoolean(BNodeSwitch.initiateEnabled, change.getBoolean(), context);
    }
  }

  //endregion

  //region ScDirectConnectAcceptEnable

  private PropertyValue readScDirectConnectAcceptEnable()
  {
    BBoolean pending = getPendingChange(SC_DIRECT_CONNECT_ACCEPT_ENABLE, BBoolean.class);
    return makeBooleanReadResult(
      scDirectConnectAcceptEnable,
      pending != null ? pending.getBoolean() : getNodeSwitch().getAcceptEnabled());
  }

  private ErrorType writeScDirectConnectAcceptEnable(byte[] value)
    throws AsnException
  {
    addPendingChange(
      SC_DIRECT_CONNECT_ACCEPT_ENABLE,
      BBoolean.make(AsnUtil.fromAsnBoolean(value)),
      BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  private void validateSetDirectConnectAcceptEnable(BValue value, Context context)
  {
    checkPendingChangeType(SC_DIRECT_CONNECT_ACCEPT_ENABLE, value, BBoolean.TYPE);
    checkCanWrite(getNodeSwitch(), BNodeSwitch.acceptEnabled, context);
  }

  private void validateScDirectConnectAcceptEnable(Context context)
    throws ValidateChangesException
  {
    BBoolean pending = getPendingChange(SC_DIRECT_CONNECT_ACCEPT_ENABLE, BBoolean.class);
    if (pending != null)
    {
      validateCanWrite(getNodeSwitch(), BNodeSwitch.acceptEnabled, scDirectConnectAcceptEnable, context);
    }
  }

  private void activateScDirectConnectAcceptEnable(Context context)
  {
    BBoolean change = getPendingChange(SC_DIRECT_CONNECT_ACCEPT_ENABLE, BBoolean.class);
    if (change != null)
    {
      getNodeSwitch().setBoolean(BNodeSwitch.acceptEnabled, change.getBoolean(), context);
    }
  }

  //endregion

  //region ScDirectConnectBinding

  private PropertyValue readScDirectConnectBinding()
  {
    return readHttpsPortBinding(scDirectConnectBinding, getNodeSwitch());
  }

  //endregion

  //region ScDirectConnectAcceptUris

  private PropertyValue readScDirectConnectAcceptUris(int index)
  {
    return readArray(scDirectConnectAcceptUris, index, getNodeSwitch().getAcceptUris());
  }

  private ErrorType writeScDirectConnectAcceptUris(int index, byte[] value)
    throws AsnException
  {
    return writeAcceptUris(getNodeSwitch().getAcceptUris(), index, value);
  }

  //endregion

  //region ScDirectConnectConnectionStatus

  private PropertyValue readScDirectConnectConnectionStatus()
  {
    BBacnetListOf list = new BBacnetListOf(BBacnetScDirectConnection.TYPE);
    for (BBacnetScDirectConnection connection : getScDirectConnections())
    {
      list.addListElement(connection, null);
    }
    return makeListOfReadResult(scDirectConnectConnectionStatus, list);
  }

  private List<BBacnetScDirectConnection> getScDirectConnections()
  {
    BAbstractConnection[] connections = getNodeSwitch().getConnections().getChildren(BAbstractConnection.class);
    List<BBacnetScDirectConnection> scConnections = new ArrayList<>(connections.length);
    for (BAbstractConnection connection : connections)
    {
      // Can only be a BDirectInitiatingConnection or a BDirectAcceptingConnection
      // All BDirectInitiatingConnections are added
      // Only BDirectAcceptingConnection that are not "failedToConnect" are added
      BBacnetScConnectionState connectionState = connection.getState();
      if (connection instanceof BDirectInitiatingConnection || !connectionState.equals(failedToConnect))
      {
        BBacnetScDirectConnection scConnection = new BBacnetScDirectConnection();
        if (connection instanceof BDirectInitiatingConnection)
        {
          try
          {
            scConnection.setUri(((BDirectInitiatingConnection) connection).getURI().toString());
          }
          catch (URISyntaxException e)
          {
            if (logger.isLoggable(Level.FINE))
            {
              logger.log(Level.FINE, this + ": failed to write direct initiating connection URI", e);
            }
          }
        }

        scConnection.setConnectionState(connectionState);

        if (connectionState.equals(failedToConnect))
        {
          scConnection.setConnectTimestamp(makeDateTime(connection.getLastFailureToConnect()));
        }
        else
        {
          scConnection.setConnectTimestamp(makeDateTime(connection.getLastConnect()));
        }

        scConnection.setDisconnectTimestamp(makeDateTime(connection.getLastDisconnect()));

        BInternetAddress peerAddress;
        BBacnetOctetString peerVmac;
        BUuid peerUuid;
        if (connection instanceof BDirectInitiatingConnection)
        {
          BDirectInitiatingConnection initiatingConnection = (BDirectInitiatingConnection) connection;
          peerAddress = initiatingConnection.getPeerAddress();
          peerVmac = initiatingConnection.getPeerVmac();
          peerUuid = initiatingConnection.getPeerUuid();
        }
        else
        {
          @SuppressWarnings("CastConflictsWithInstanceof")
          BDirectAcceptingConnection acceptingConnection = (BDirectAcceptingConnection) connection;
          peerAddress = acceptingConnection.getPeerAddress();
          peerVmac = acceptingConnection.getPeerVmac();
          peerUuid = acceptingConnection.getPeerUuid();
        }
        scConnection.setPeerAddress(makeHostNPort(peerAddress));
        scConnection.setPeerVmac(peerVmac);
        if (!peerUuid.equals(BUuid.DEFAULT))
        {
          scConnection.setPeerUuid(BBacnetOctetString.make(peerUuid.getBytes()));
        }

        if (connectionState.equals(failedToConnect) || connectionState.equals(disconnectedWithErrors))
        {
          scConnection.setError((BErrorType) connection.getError().newCopy());
          scConnection.setErrorDetails(connection.getErrorDetails());
        }

        scConnections.add(scConnection);
      }
    }
    return scConnections;
  }

  //endregion

  //region ScFailedConnectionRequests

  private PropertyValue readScFailedConnectionRequests()
  {
    BBacnetListOf list = new BBacnetListOf(BBacnetScFailedConnectionRequest.TYPE);
    for (BBacnetScFailedConnectionRequest connection : getFailedConnections())
    {
      list.addListElement(connection, null);
    }
    return makeListOfReadResult(scFailedConnectionRequests, list);
  }

  private List<BBacnetScFailedConnectionRequest> getFailedConnections()
  {
    List<BBacnetScFailedConnectionRequest> failedConnections = new ArrayList<>();

    BHubFunction hubFunction = getHubFunction();
    if (hubFunction != null)
    {
      for (BHubAcceptingConnection connection : hubFunction.getConnections().getChildren(BHubAcceptingConnection.class))
      {
        if (connection.getState().equals(failedToConnect))
        {
          BBacnetScFailedConnectionRequest failedConnection = new BBacnetScFailedConnectionRequest();
          failedConnection.setTimestamp(makeDateTime(connection.getLastFailureToConnect()));
          failedConnection.setPeerAddress(makeHostNPort(connection.getNodeAddress()));
          failedConnection.setPeerVmac(connection.getNodeVmac());

          BUuid nodeUuid = connection.getNodeUuid();
          if (!nodeUuid.equals(BUuid.DEFAULT))
          {
            failedConnection.setPeerUuid(BBacnetOctetString.make(connection.getNodeUuid().getBytes()));
          }

          failedConnection.setError((BErrorType) connection.getError().newCopy());
          failedConnection.setErrorDetails(connection.getErrorDetails());
          failedConnections.add(failedConnection);
        }
      }
    }

    BDirectAcceptingConnection[] connections = getNodeSwitch().getConnections().getChildren(BDirectAcceptingConnection.class);
    for (BDirectAcceptingConnection connection : connections)
    {
      if (connection.getState().equals(failedToConnect))
      {
        BBacnetScFailedConnectionRequest failedConnection = new BBacnetScFailedConnectionRequest();
        failedConnection.setTimestamp(makeDateTime(connection.getLastFailureToConnect()));
        failedConnection.setPeerAddress(makeHostNPort(connection.getPeerAddress()));
        failedConnection.setPeerVmac(connection.getPeerVmac());

        BUuid peerUuid = connection.getPeerUuid();
        if (!peerUuid.equals(BUuid.DEFAULT))
        {
          failedConnection.setPeerUuid(BBacnetOctetString.make(connection.getPeerUuid().getBytes()));
        }

        failedConnection.setError((BErrorType) connection.getError().newCopy());
        failedConnection.setErrorDetails(connection.getErrorDetails());
        failedConnections.add(failedConnection);
      }
    }
    return failedConnections;
  }

  //endregion

  //region Utility

  private BScLinkLayer getScLink()
  {
    return (BScLinkLayer) networkPort.getLink();
  }

  private BScConfiguration getScConfig()
  {
    return getScLink().getConfig();
  }

  private BNodeSwitch getNodeSwitch()
  {
    return getScLink().getNodeSwitch();
  }

  private BHubConnector getHubConnector()
  {
    return getScLink().getHubConnector();
  }

  private BHubFunction getHubFunction()
  {
    return getScLink().getHubFunction();
  }

  private BHubInitiatingConnection getPrimaryConnection()
  {
    return getHubConnector().getPrimaryConnection();
  }

  private BHubInitiatingConnection getFailoverConnection()
  {
    return getHubConnector().getFailoverConnection();
  }

  private PropertyValue readUri(BBacnetPropertyIdentifier propertyId, BHubInitiatingConnection connection)
  {
    try
    {
      return makeCharStringReadResult(propertyId, connection.getURI().toString());
    }
    catch (URISyntaxException e)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.log(Level.FINE, this + ": failed to retrieve connection URI for " + propertyId, e);
      }
      return makeReadError(propertyId, property, other);
    }
  }

  private static void validateUri(String uri, BBacnetPropertyIdentifier propertyId)
    throws ValidateChangesException
  {
    String error = ScMessageUtil.checkWebSocketUri(uri);
    if (error != null)
    {
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ valueOutOfRange,
        /* property */ propertyId,
        /* details */ "Web Socket URI " + uri + " is not valid; error: " + error);
    }
  }

  private void checkCanWriteUri(BHubInitiatingConnection connection, Context context)
  {
    checkCanWrite(connection, BHubInitiatingConnection.hubUriAddress, context);
    checkCanWrite(connection, BHubInitiatingConnection.hubUriPath, context);
    checkCanWrite(connection, BHubInitiatingConnection.hubUriQuery, context);
  }

  private void validateCanWriteUri(
    BHubInitiatingConnection connection,
    BBacnetPropertyIdentifier propertyId,
    Context context)
    throws ValidateChangesException
  {
    validateCanWrite(connection, BHubInitiatingConnection.hubUriAddress, propertyId, context);
    validateCanWrite(connection, BHubInitiatingConnection.hubUriPath, propertyId, context);
    validateCanWrite(connection, BHubInitiatingConnection.hubUriQuery, propertyId, context);
  }

  private static void activateUri(String uriStr, BHubInitiatingConnection connection, Context context)
    throws URISyntaxException
  {
    URI uri = new URI(uriStr);
    connection.set(
      BHubInitiatingConnection.hubUriAddress,
      new BInternetAddress(uri.getHost(), uri.getPort()),
      context);
    String path = uri.getPath();
    connection.setString(BHubInitiatingConnection.hubUriPath, path != null ? path : "", context);
    String query = uri.getQuery();
    connection.setString(BHubInitiatingConnection.hubUriQuery, query != null ? query : "", context);
  }

  private static PropertyValue readConfigTime(BBacnetPropertyIdentifier propertyId, int seconds)
  {
    if (seconds < 1)
    {
      return makeReadError(propertyId, property, valueNotInitialized);
    }
    return makeUnsignedReadResult(propertyId, seconds);
  }

  @SuppressWarnings("SameParameterValue")
  private ErrorType writeConfigTime(Property configProp, byte[] value, int min, int max)
    throws AsnException
  {
    int seconds = AsnUtil.fromAsnUnsignedInt(value);
    if (seconds < min || seconds > max)
    {
      return new NErrorType(property, valueOutOfRange);
    }

    getScConfig().set(configProp, BRelTime.makeSeconds(seconds), BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  private static BBacnetDateTime makeDateTime(BAbsTime absTime)
  {
    return absTime.equals(BAbsTime.DEFAULT) ?
      new BBacnetDateTime() :
      new BBacnetDateTime(absTime);
  }

  private static PropertyValue readHttpsPortBinding(BBacnetPropertyIdentifier propertyId, BAbstractConnectionManager connectionManager)
  {
    BWebService webService = connectionManager.getWebSocketAcceptor().getWebService();
    if (webService == null)
    {
      return makeReadError(propertyId, property, valueNotInitialized);
    }

    return makeCharStringReadResult(propertyId, Integer.toString(webService.getHttpsPort().getPublicServerPort()));
  }

  private static BBacnetHostNPort makeHostNPort(BInternetAddress internetAddress)
  {
    String host = internetAddress.getHost();
    if (host == null)
    {
      return new BBacnetHostNPort();
    }

    BBacnetHostAddress hostAddress;
    byte[] hostBytes = IPAddressUtil.numericStringToByteArray(host);
    if (hostBytes != null && (hostBytes.length == 4 || hostBytes.length == 16))
    {
      hostAddress = BBacnetHostAddress.makeWithIpAddress(hostBytes);
    }
    else
    {
      hostAddress = BBacnetHostAddress.makeWithHostName(host);
    }

    return new BBacnetHostNPort(hostAddress, Math.max(0, internetAddress.getPort()));
  }

  private static ErrorType writeAcceptUris(BBacnetArray array, int index, byte[] value)
    throws AsnException
  {
    if (index == 0)
    {
      int newSize = AsnUtil.fromAsnUnsignedInt(value);
      array.setInt(BBacnetArray.size, newSize, BLocalBacnetDevice.getBacnetContext());
      return null;
    }
    else if (index == NOT_USED)
    {
      // Replace the entire array
      BBacnetArray newArray = new BBacnetArray(BString.TYPE);
      newArray = (BBacnetArray) AsnUtil.fromAsn(ASN_BACNET_ARRAY, value, newArray);
      for (BString element : newArray.getChildren(BString.class))
      {
        if (ScMessageUtil.checkWebSocketUri(element.getString()) != null)
        {
          return new NErrorType(property, valueOutOfRange);
        }
      }
      array.getParent().set(array.getPropertyInParent(), newArray, BLocalBacnetDevice.getBacnetContext());
      return null;
    }
    else if (index >= 1 && index <= array.getSize())
    {
      String uri = AsnUtil.fromAsnCharacterString(value);
      if (ScMessageUtil.checkWebSocketUri(uri) != null)
      {
        return new NErrorType(property, valueOutOfRange);
      }
      array.set("element" + index, BString.make(uri), BLocalBacnetDevice.getBacnetContext());
      return null;
    }
    else
    {
      return new NErrorType(property, invalidArrayIndex);
    }
  }

  private ErrorType writeHubFunctionReadOnlyProperty(BBacnetPropertyIdentifier propertyId)
  {
    BHubFunction hubFunction = getHubFunction();
    if (hubFunction == null)
    {
      return new NErrorType(property, unknownProperty);
    }
    else
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(this + ": attempted to write read-only property " + propertyId);
      }
      return new NErrorType(property, writeAccessDenied);
    }
  }

  //endregion

  //region Fields

  private static final String SC_PRIMARY_HUB_URI = "ScPrimaryHubUri";
  private static final String SC_FAILOVER_HUB_URI = "ScFailoverHubUri";
  private static final String SC_HUB_FUNCTION_ENABLE = "ScHubFunctionEnable";
  private static final String SC_DIRECT_CONNECT_INITIATE_ENABLE = "ScDirectConnectInitiateEnable";
  private static final String SC_DIRECT_CONNECT_ACCEPT_ENABLE = "ScDirectConnectAcceptEnable";

  private static final int[] OPTIONAL_PROPS_WITHOUT_HUB_FUNCTION = {
    BBacnetPropertyIdentifier.DESCRIPTION,
    BBacnetPropertyIdentifier.COMMAND,
    BBacnetPropertyIdentifier.NETWORK_NUMBER,
    BBacnetPropertyIdentifier.NETWORK_NUMBER_QUALITY,
    BBacnetPropertyIdentifier.APDU_LENGTH,
    BBacnetPropertyIdentifier.MAC_ADDRESS,
    BBacnetPropertyIdentifier.MAX_BVLC_LENGTH_ACCEPTED,
    BBacnetPropertyIdentifier.MAX_NPDU_LENGTH_ACCEPTED,
    BBacnetPropertyIdentifier.SC_PRIMARY_HUB_URI,
    BBacnetPropertyIdentifier.SC_FAILOVER_HUB_URI,
    BBacnetPropertyIdentifier.SC_MINIMUM_RECONNECT_TIME,
    BBacnetPropertyIdentifier.SC_MAXIMUM_RECONNECT_TIME,
    BBacnetPropertyIdentifier.SC_CONNECT_WAIT_TIMEOUT,
    BBacnetPropertyIdentifier.SC_DISCONNECT_WAIT_TIMEOUT,
    BBacnetPropertyIdentifier.SC_HEARTBEAT_TIMEOUT,
    BBacnetPropertyIdentifier.SC_HUB_CONNECTOR_STATE,
    BBacnetPropertyIdentifier.ROUTING_TABLE,
    BBacnetPropertyIdentifier.SC_PRIMARY_HUB_CONNECTION_STATUS,
    BBacnetPropertyIdentifier.SC_FAILOVER_HUB_CONNECTION_STATUS,
    BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_INITIATE_ENABLE,
    BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_ACCEPT_ENABLE,
    BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_ACCEPT_URIS,
    BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_BINDING,
    BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_CONNECTION_STATUS,
    BBacnetPropertyIdentifier.SC_FAILED_CONNECTION_REQUESTS
  };

  private static final int[] OPTIONAL_PROPS_WITH_HUB_FUNCTION = {
    BBacnetPropertyIdentifier.DESCRIPTION,
    BBacnetPropertyIdentifier.COMMAND,
    BBacnetPropertyIdentifier.NETWORK_NUMBER,
    BBacnetPropertyIdentifier.NETWORK_NUMBER_QUALITY,
    BBacnetPropertyIdentifier.APDU_LENGTH,
    BBacnetPropertyIdentifier.MAC_ADDRESS,
    BBacnetPropertyIdentifier.MAX_BVLC_LENGTH_ACCEPTED,
    BBacnetPropertyIdentifier.MAX_NPDU_LENGTH_ACCEPTED,
    BBacnetPropertyIdentifier.SC_PRIMARY_HUB_URI,
    BBacnetPropertyIdentifier.SC_FAILOVER_HUB_URI,
    BBacnetPropertyIdentifier.SC_MINIMUM_RECONNECT_TIME,
    BBacnetPropertyIdentifier.SC_MAXIMUM_RECONNECT_TIME,
    BBacnetPropertyIdentifier.SC_CONNECT_WAIT_TIMEOUT,
    BBacnetPropertyIdentifier.SC_DISCONNECT_WAIT_TIMEOUT,
    BBacnetPropertyIdentifier.SC_HEARTBEAT_TIMEOUT,
    BBacnetPropertyIdentifier.SC_HUB_CONNECTOR_STATE,
    BBacnetPropertyIdentifier.ROUTING_TABLE,
    BBacnetPropertyIdentifier.SC_PRIMARY_HUB_CONNECTION_STATUS,
    BBacnetPropertyIdentifier.SC_FAILOVER_HUB_CONNECTION_STATUS,
    BBacnetPropertyIdentifier.SC_HUB_FUNCTION_ENABLE,
    BBacnetPropertyIdentifier.SC_HUB_FUNCTION_ACCEPT_URIS,
    BBacnetPropertyIdentifier.SC_HUB_FUNCTION_BINDING,
    BBacnetPropertyIdentifier.SC_HUB_FUNCTION_CONNECTION_STATUS,
    BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_INITIATE_ENABLE,
    BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_ACCEPT_ENABLE,
    BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_ACCEPT_URIS,
    BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_BINDING,
    BBacnetPropertyIdentifier.SC_DIRECT_CONNECT_CONNECTION_STATUS,
    BBacnetPropertyIdentifier.SC_FAILED_CONNECTION_REQUESTS
  };

  //endregion
}
