/*
 * Copyright 2023 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.export;

import static javax.baja.bacnet.enums.BBacnetErrorClass.object;
import static javax.baja.bacnet.enums.BBacnetErrorClass.property;
import static javax.baja.bacnet.enums.BBacnetErrorClass.services;
import static javax.baja.bacnet.enums.BBacnetErrorCode.busy;
import static javax.baja.bacnet.enums.BBacnetErrorCode.inconsistentParameters;
import static javax.baja.bacnet.enums.BBacnetErrorCode.invalidArrayIndex;
import static javax.baja.bacnet.enums.BBacnetErrorCode.invalidDataType;
import static javax.baja.bacnet.enums.BBacnetErrorCode.invalidValueInThisState;
import static javax.baja.bacnet.enums.BBacnetErrorCode.listItemNotNumbered;
import static javax.baja.bacnet.enums.BBacnetErrorCode.listItemNotTimestamped;
import static javax.baja.bacnet.enums.BBacnetErrorCode.notConfigured;
import static javax.baja.bacnet.enums.BBacnetErrorCode.optionalFunctionalityNotSupported;
import static javax.baja.bacnet.enums.BBacnetErrorCode.other;
import static javax.baja.bacnet.enums.BBacnetErrorCode.parameterOutOfRange;
import static javax.baja.bacnet.enums.BBacnetErrorCode.propertyIsNotA_List;
import static javax.baja.bacnet.enums.BBacnetErrorCode.propertyIsNotAnArray;
import static javax.baja.bacnet.enums.BBacnetErrorCode.readAccessDenied;
import static javax.baja.bacnet.enums.BBacnetErrorCode.unknownProperty;
import static javax.baja.bacnet.enums.BBacnetErrorCode.valueOutOfRange;
import static javax.baja.bacnet.enums.BBacnetErrorCode.writeAccessDenied;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.apduLength;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.networkNumber;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.networkNumberQuality;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.routingTable;
import static javax.baja.bacnet.enums.BBacnetReliability.processError;
import static javax.baja.bacnet.enums.BBacnetRouterStatus.available;
import static javax.baja.bacnet.enums.BBacnetRouterStatus.disconnected;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeAddListElementError;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeEnumReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeListOfReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeReadError;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeRemoveListElementError;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeUnsignedReadResult;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.baja.bacnet.BBacnetNetwork;
import javax.baja.bacnet.BacnetException;
import javax.baja.bacnet.datatypes.BBacnetBitString;
import javax.baja.bacnet.datatypes.BBacnetListOf;
import javax.baja.bacnet.datatypes.BBacnetObjectIdentifier;
import javax.baja.bacnet.datatypes.BBacnetRoutingTableEntry;
import javax.baja.bacnet.datatypes.BBacnetUnsigned;
import javax.baja.bacnet.datatypes.BIBacnetDataType;
import javax.baja.bacnet.enums.BBacnetErrorCode;
import javax.baja.bacnet.enums.BBacnetNetworkPortCommand;
import javax.baja.bacnet.enums.BBacnetNetworkType;
import javax.baja.bacnet.enums.BBacnetObjectType;
import javax.baja.bacnet.enums.BBacnetPropertyIdentifier;
import javax.baja.bacnet.enums.BBacnetProtocolLevel;
import javax.baja.bacnet.enums.BBacnetReliability;
import javax.baja.bacnet.enums.BBacnetRouterStatus;
import javax.baja.bacnet.io.AsnException;
import javax.baja.bacnet.io.ChangeListError;
import javax.baja.bacnet.io.ErrorException;
import javax.baja.bacnet.io.ErrorType;
import javax.baja.bacnet.io.OutOfRangeException;
import javax.baja.bacnet.io.PropertyReference;
import javax.baja.bacnet.io.PropertyValue;
import javax.baja.bacnet.io.RangeData;
import javax.baja.bacnet.io.RangeReference;
import javax.baja.bacnet.io.RejectException;
import javax.baja.naming.BOrd;
import javax.baja.nre.annotations.Facet;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraAction;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.security.BPermissions;
import javax.baja.security.PermissionException;
import javax.baja.status.BStatus;
import javax.baja.sys.Action;
import javax.baja.sys.BBoolean;
import javax.baja.sys.BComplex;
import javax.baja.sys.BComponent;
import javax.baja.sys.BDynamicEnum;
import javax.baja.sys.BEnum;
import javax.baja.sys.BEnumRange;
import javax.baja.sys.BFacets;
import javax.baja.sys.BIcon;
import javax.baja.sys.BObject;
import javax.baja.sys.BValue;
import javax.baja.sys.BajaException;
import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.BasicContext;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.LocalizableRuntimeException;
import javax.baja.sys.Property;
import javax.baja.sys.Slot;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.user.BUser;

import com.tridium.bacnet.BacUtil;
import com.tridium.bacnet.asn.AsnOutputStream;
import com.tridium.bacnet.asn.AsnUtil;
import com.tridium.bacnet.asn.NErrorType;
import com.tridium.bacnet.asn.NReadPropertyResult;
import com.tridium.bacnet.services.BacnetConfirmedRequest;
import com.tridium.bacnet.services.confirmed.ReadRangeAck;
import com.tridium.bacnet.stack.BBacnetStack;
import com.tridium.bacnet.stack.network.BBacnetNetworkLayer;
import com.tridium.bacnet.stack.network.BBacnetRouterEntry;
import com.tridium.bacnet.stack.network.BNetworkPort;
import com.tridium.bacnet.stack.network.BRouterStatus;
import com.tridium.bacnet.stack.server.BBacnetExportTable;

/**
 * BBacnetNetworkPortDescriptor exposes a BNetworkPort as a Network Port object on the BACnet
 * network.
 *
 * @author Uday Rapuru on 19-Sep-2023
 * @since Niagara 4.15
 */
@NiagaraType
/*
 Status for Niagara server-side behavior
 */
@NiagaraProperty(
  name = "status",
  type = "BStatus",
  defaultValue = "BStatus.ok",
  flags = Flags.READONLY | Flags.TRANSIENT | Flags.DEFAULT_ON_CLONE
)
/*
 Description of a fault with server-side behavior
 */
@NiagaraProperty(
  name = "faultCause",
  type = "String",
  defaultValue = "",
  flags = Flags.READONLY | Flags.TRANSIENT | Flags.DEFAULT_ON_CLONE
)
/*
 The ord of the BNetworkPort being exposed as a BACnet Network Port object.
 */
@NiagaraProperty(
  name = "networkPortOrd",
  type = "BOrd",
  defaultValue = "BOrd.DEFAULT",
  flags = Flags.READONLY | Flags.DEFAULT_ON_CLONE,
  facets = @Facet(name = "BFacets.TARGET_TYPE", value = "\"bacnet:NetworkPort\"")
)
/*
 Identifies this BACnet Network Port object within this device. When the Protocol_Level property has
 a value of BACNET_APPLICATION, the instance number shall correspond to the Port ID of the
 associated network.
 */
@NiagaraProperty(
  name = "objectId",
  type = "BBacnetObjectIdentifier",
  defaultValue = "BBacnetObjectIdentifier.make(BBacnetObjectType.NETWORK_PORT)",
  flags = Flags.READONLY | Flags.TRANSIENT | Flags.DEFAULT_ON_CLONE,
  facets = @Facet("BBacnetObjectType.getObjectIdFacets(BBacnetObjectType.NETWORK_PORT)")
)
/*
 A name for the object that is unique within the BACnet device that maintains it.
 */
@NiagaraProperty(
  name = "objectName",
  type = "String",
  defaultValue = "",
  flags = Flags.DEFAULT_ON_CLONE
)
@NiagaraProperty(
  name = "description",
  type = "String",
  defaultValue = ""
)
/*
 Provides an indication of whether the Network Port object, the network port, and the network
 connected to the port are "reliable" as far as the BACnet device can determine and, if not, why.
 */
@NiagaraProperty(
  name = "reliability",
  type = "BEnum",
  defaultValue = "BBacnetReliability.noFaultDetected",
  flags = Flags.READONLY | Flags.TRANSIENT
)
/*
 Used to request that the Network Port object perform various actions.
 */
@NiagaraProperty(
  name = "command",
  type = "BEnum",
  defaultValue = "BDynamicEnum.make(BBacnetNetworkPortCommand.IDLE, BEnumRange.make(BBacnetNetworkPortCommand.TYPE))",
  flags = Flags.READONLY | Flags.TRANSIENT
)
/*
 Holds pending changes until they are activated.
 */
@NiagaraProperty(
  name = "pendingChanges",
  type = "BBacnetNetworkPortPendingChanges",
  defaultValue = "new BBacnetNetworkPortPendingChanges()"
)
/*
 Execute a NetworkPort command as if writing to the Command property.
 */
@NiagaraAction(
  name = "executeCommand",
  parameterType = "BDynamicEnum",
  defaultValue = "BDynamicEnum.make(BBacnetNetworkPortCommand.idle)"
)
/*
 Activates pending changes on this NetworkPort object as if initiating a ReinitializeDevice service
 request. Set the action parameter to true to apply changes that require a reboot to complete
 activation.
 */
@NiagaraAction(
  name = "activatePendingChanges",
  parameterType = "BBoolean",
  defaultValue = "BBoolean.FALSE"
)
public abstract class BBacnetNetworkPortDescriptor
  extends BComponent
  implements BIBacnetExportObject,
             BacnetPropertyListProvider
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.export.BBacnetNetworkPortDescriptor(988438606)1.0$ @*/
/* Generated Tue Nov 05 12:56:20 CST 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Property "status"

  /**
   * Slot for the {@code status} property.
   * Status for Niagara server-side behavior
   * @see #getStatus
   * @see #setStatus
   */
  @Generated
  public static final Property status = newProperty(Flags.READONLY | Flags.TRANSIENT | Flags.DEFAULT_ON_CLONE, BStatus.ok, null);

  /**
   * Get the {@code status} property.
   * Status for Niagara server-side behavior
   * @see #status
   */
  @Generated
  public BStatus getStatus() { return (BStatus)get(status); }

  /**
   * Set the {@code status} property.
   * Status for Niagara server-side behavior
   * @see #status
   */
  @Generated
  public void setStatus(BStatus v) { set(status, v, null); }

  //endregion Property "status"

  //region Property "faultCause"

  /**
   * Slot for the {@code faultCause} property.
   * Description of a fault with server-side behavior
   * @see #getFaultCause
   * @see #setFaultCause
   */
  @Generated
  public static final Property faultCause = newProperty(Flags.READONLY | Flags.TRANSIENT | Flags.DEFAULT_ON_CLONE, "", null);

  /**
   * Get the {@code faultCause} property.
   * Description of a fault with server-side behavior
   * @see #faultCause
   */
  @Generated
  public String getFaultCause() { return getString(faultCause); }

  /**
   * Set the {@code faultCause} property.
   * Description of a fault with server-side behavior
   * @see #faultCause
   */
  @Generated
  public void setFaultCause(String v) { setString(faultCause, v, null); }

  //endregion Property "faultCause"

  //region Property "networkPortOrd"

  /**
   * Slot for the {@code networkPortOrd} property.
   * The ord of the BNetworkPort being exposed as a BACnet Network Port object.
   * @see #getNetworkPortOrd
   * @see #setNetworkPortOrd
   */
  @Generated
  public static final Property networkPortOrd = newProperty(Flags.READONLY | Flags.DEFAULT_ON_CLONE, BOrd.DEFAULT, BFacets.make(BFacets.TARGET_TYPE, "bacnet:NetworkPort"));

  /**
   * Get the {@code networkPortOrd} property.
   * The ord of the BNetworkPort being exposed as a BACnet Network Port object.
   * @see #networkPortOrd
   */
  @Generated
  public BOrd getNetworkPortOrd() { return (BOrd)get(networkPortOrd); }

  /**
   * Set the {@code networkPortOrd} property.
   * The ord of the BNetworkPort being exposed as a BACnet Network Port object.
   * @see #networkPortOrd
   */
  @Generated
  public void setNetworkPortOrd(BOrd v) { set(networkPortOrd, v, null); }

  //endregion Property "networkPortOrd"

  //region Property "objectId"

  /**
   * Slot for the {@code objectId} property.
   * Identifies this BACnet Network Port object within this device. When the Protocol_Level property has
   * a value of BACNET_APPLICATION, the instance number shall correspond to the Port ID of the
   * associated network.
   * @see #getObjectId
   * @see #setObjectId
   */
  @Generated
  public static final Property objectId = newProperty(Flags.READONLY | Flags.TRANSIENT | Flags.DEFAULT_ON_CLONE, BBacnetObjectIdentifier.make(BBacnetObjectType.NETWORK_PORT), BBacnetObjectType.getObjectIdFacets(BBacnetObjectType.NETWORK_PORT));

  /**
   * Get the {@code objectId} property.
   * Identifies this BACnet Network Port object within this device. When the Protocol_Level property has
   * a value of BACNET_APPLICATION, the instance number shall correspond to the Port ID of the
   * associated network.
   * @see #objectId
   */
  @Generated
  public BBacnetObjectIdentifier getObjectId() { return (BBacnetObjectIdentifier)get(objectId); }

  /**
   * Set the {@code objectId} property.
   * Identifies this BACnet Network Port object within this device. When the Protocol_Level property has
   * a value of BACNET_APPLICATION, the instance number shall correspond to the Port ID of the
   * associated network.
   * @see #objectId
   */
  @Generated
  public void setObjectId(BBacnetObjectIdentifier v) { set(objectId, v, null); }

  //endregion Property "objectId"

  //region Property "objectName"

  /**
   * Slot for the {@code objectName} property.
   * A name for the object that is unique within the BACnet device that maintains it.
   * @see #getObjectName
   * @see #setObjectName
   */
  @Generated
  public static final Property objectName = newProperty(Flags.DEFAULT_ON_CLONE, "", null);

  /**
   * Get the {@code objectName} property.
   * A name for the object that is unique within the BACnet device that maintains it.
   * @see #objectName
   */
  @Generated
  public String getObjectName() { return getString(objectName); }

  /**
   * Set the {@code objectName} property.
   * A name for the object that is unique within the BACnet device that maintains it.
   * @see #objectName
   */
  @Generated
  public void setObjectName(String v) { setString(objectName, v, null); }

  //endregion Property "objectName"

  //region Property "description"

  /**
   * Slot for the {@code description} property.
   * @see #getDescription
   * @see #setDescription
   */
  @Generated
  public static final Property description = newProperty(0, "", null);

  /**
   * Get the {@code description} property.
   * @see #description
   */
  @Generated
  public String getDescription() { return getString(description); }

  /**
   * Set the {@code description} property.
   * @see #description
   */
  @Generated
  public void setDescription(String v) { setString(description, v, null); }

  //endregion Property "description"

  //region Property "reliability"

  /**
   * Slot for the {@code reliability} property.
   * Provides an indication of whether the Network Port object, the network port, and the network
   * connected to the port are "reliable" as far as the BACnet device can determine and, if not, why.
   * @see #getReliability
   * @see #setReliability
   */
  @Generated
  public static final Property reliability = newProperty(Flags.READONLY | Flags.TRANSIENT, BBacnetReliability.noFaultDetected, null);

  /**
   * Get the {@code reliability} property.
   * Provides an indication of whether the Network Port object, the network port, and the network
   * connected to the port are "reliable" as far as the BACnet device can determine and, if not, why.
   * @see #reliability
   */
  @Generated
  public BEnum getReliability() { return (BEnum)get(reliability); }

  /**
   * Set the {@code reliability} property.
   * Provides an indication of whether the Network Port object, the network port, and the network
   * connected to the port are "reliable" as far as the BACnet device can determine and, if not, why.
   * @see #reliability
   */
  @Generated
  public void setReliability(BEnum v) { set(reliability, v, null); }

  //endregion Property "reliability"

  //region Property "command"

  /**
   * Slot for the {@code command} property.
   * Used to request that the Network Port object perform various actions.
   * @see #getCommand
   * @see #setCommand
   */
  @Generated
  public static final Property command = newProperty(Flags.READONLY | Flags.TRANSIENT, BDynamicEnum.make(BBacnetNetworkPortCommand.IDLE, BEnumRange.make(BBacnetNetworkPortCommand.TYPE)), null);

  /**
   * Get the {@code command} property.
   * Used to request that the Network Port object perform various actions.
   * @see #command
   */
  @Generated
  public BEnum getCommand() { return (BEnum)get(command); }

  /**
   * Set the {@code command} property.
   * Used to request that the Network Port object perform various actions.
   * @see #command
   */
  @Generated
  public void setCommand(BEnum v) { set(command, v, null); }

  //endregion Property "command"

  //region Property "pendingChanges"

  /**
   * Slot for the {@code pendingChanges} property.
   * Holds pending changes until they are activated.
   * @see #getPendingChanges
   * @see #setPendingChanges
   */
  @Generated
  public static final Property pendingChanges = newProperty(0, new BBacnetNetworkPortPendingChanges(), null);

  /**
   * Get the {@code pendingChanges} property.
   * Holds pending changes until they are activated.
   * @see #pendingChanges
   */
  @Generated
  public BBacnetNetworkPortPendingChanges getPendingChanges() { return (BBacnetNetworkPortPendingChanges)get(pendingChanges); }

  /**
   * Set the {@code pendingChanges} property.
   * Holds pending changes until they are activated.
   * @see #pendingChanges
   */
  @Generated
  public void setPendingChanges(BBacnetNetworkPortPendingChanges v) { set(pendingChanges, v, null); }

  //endregion Property "pendingChanges"

  //region Action "executeCommand"

  /**
   * Slot for the {@code executeCommand} action.
   * Execute a NetworkPort command as if writing to the Command property.
   * @see #executeCommand(BDynamicEnum parameter)
   */
  @Generated
  public static final Action executeCommand = newAction(0, BDynamicEnum.make(BBacnetNetworkPortCommand.idle), null);

  /**
   * Invoke the {@code executeCommand} action.
   * Execute a NetworkPort command as if writing to the Command property.
   * @see #executeCommand
   */
  @Generated
  public void executeCommand(BDynamicEnum parameter) { invoke(executeCommand, parameter, null); }

  //endregion Action "executeCommand"

  //region Action "activatePendingChanges"

  /**
   * Slot for the {@code activatePendingChanges} action.
   * Activates pending changes on this NetworkPort object as if initiating a ReinitializeDevice service
   * request. Set the action parameter to true to apply changes that require a reboot to complete
   * activation.
   * @see #activatePendingChanges(BBoolean parameter)
   */
  @Generated
  public static final Action activatePendingChanges = newAction(0, BBoolean.FALSE, null);

  /**
   * Invoke the {@code activatePendingChanges} action.
   * Activates pending changes on this NetworkPort object as if initiating a ReinitializeDevice service
   * request. Set the action parameter to true to apply changes that require a reboot to complete
   * activation.
   * @see #activatePendingChanges
   */
  @Generated
  public void activatePendingChanges(BBoolean parameter) { invoke(activatePendingChanges, parameter, null); }

  //endregion Action "activatePendingChanges"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetNetworkPortDescriptor.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  //region BComponent

  @Override
  public void started()
    throws Exception
  {
    super.started();

    checkFatalFault();

    oldId = getObjectId();
    oldName = getObjectName();
    resolveNetworkPort();

    if (Sys.isStationStarted())
    {
      // Exporting when the descriptor is added to a running station. Do not export before the
      // NetworkPort has started and updated this descriptor's object ID with its port ID.
      checkConfiguration();
      if (!exportFault)
      {
        BBacnetNetwork.localDevice().incrementDatabaseRevision();
      }
    }
  }

  @Override
  public void stopped()
    throws Exception
  {
    super.stopped();

    // Clear the local copies.
    BLocalBacnetDevice local = BBacnetNetwork.localDevice();
    local.unexport(oldId, oldName, this);
    networkPort = null;
    oldId = null;
    oldName = null;

    // Increment the Database_Revision when a network port is removed from a running station.
    if (local.isRunning())
    {
      local.incrementDatabaseRevision();
    }
  }

  @Override
  public void changed(Property p, Context cx)
  {
    super.changed(p, cx);

    if (!isRunning())
    {
      return;
    }

    if (p.equals(objectId) && cx != skipExport)
    {
      BBacnetObjectIdentifier newId = getObjectId();
      if (networkPort != null)
      {
        int portId = networkPort.getPortId();
        if (portId != newId.getInstanceNumber())
        {
          logger.info(this + ": resetting object ID " + newId + " based on port ID " + portId);
          newId = makeObjectId(portId);
          set(objectId, newId, skipExport);
        }

        BLocalBacnetDevice local = BBacnetNetwork.localDevice();
        local.unexport(oldId, oldName, this);
        oldId = newId;
        checkConfiguration();
        if (cx != atStarted && !exportFault)
        {
          local.incrementDatabaseRevision();
        }
      }
      else
      {
        logger.info(this + ": resetting object ID " + newId + " instance number to -1 because network port cannot be resolved");
        newId = makeObjectId(NOT_USED);
        set(objectId, newId, skipExport);
      }
    }
    else if (p.equals(objectName) && cx != skipExport)
    {
      BLocalBacnetDevice local = BBacnetNetwork.localDevice();
      local.unexport(oldId, oldName, this);
      oldName = getObjectName();
      checkConfiguration();
      if (!exportFault)
      {
        local.incrementDatabaseRevision();
      }
    }
    else if (p.equals(networkPortOrd))
    {
      BLocalBacnetDevice local = BBacnetNetwork.localDevice();
      local.unexport(oldId, oldName, this);
      resolveNetworkPort();
      if (networkPort != null)
      {
        oldId = makeObjectId(networkPort.getPortId());
      }
      else
      {
        oldId = makeObjectId(NOT_USED);
      }
      set(objectId, oldId, skipExport);
      checkConfiguration();
      if (!exportFault)
      {
        local.incrementDatabaseRevision();
      }
    }
    else if (p.equals(reliability))
    {
      boolean hasFault = !getReliability().equals(BBacnetReliability.noFaultDetected);
      setStatus(BStatus.makeFault(getStatus(), hasFault));
    }
  }

  public static String makeDescriptorName(String portName)
  {
    return "NetworkPort_" + portName;
  }

  public static BBacnetObjectIdentifier makeObjectId(int portId)
  {
    return BBacnetObjectIdentifier.make(BBacnetObjectType.NETWORK_PORT, portId);
  }

  private static final BIcon icon = BIcon.make(BIcon.std("usbPort.png"), BIcon.std("badges/export.png"));

  @Override
  public BIcon getIcon()
  {
    return icon;
  }

  @Override
  public String toString(Context c)
  {
    return getObjectName() + " [" + getObjectId() + ']';
  }

  //endregion

  //region BIBacnetExportObject

  @Override
  public BObject getObject()
  {
    return networkPort;
  }

  @Override
  public BOrd getObjectOrd()
  {
    return getNetworkPortOrd();
  }

  @Override
  public void setObjectOrd(BOrd objectOrd, Context cx)
  {
    set(networkPortOrd, objectOrd, cx);
  }

  @Override
  public boolean isFatalFault()
  {
    return fatalFault;
  }

  private void checkFatalFault()
  {
    // Short circuit if already in fatal fault
    if (fatalFault)
    {
      return;
    }

    // Find the parent export table and local device
    BBacnetExportTable exportTable = null;
    BLocalBacnetDevice localDevice = null;
    BComplex parent = getParent();
    while (parent != null)
    {
      if (parent instanceof BBacnetExportTable)
      {
        exportTable = (BBacnetExportTable) parent;
      }
      else if (parent instanceof BLocalBacnetDevice)
      {
        localDevice = (BLocalBacnetDevice) parent;
        break;
      }
      parent = parent.getParent();
    }

    // Check if mounted in local device's export table
    if (exportTable == null || localDevice == null)
    {
      fatalFault = true;
      setFaultCause("Not under LocalBacnetDevice Export Table");
      return;
    }

    // Check local device fatal fault
    if (localDevice.isFatalFault())
    {
      fatalFault = true;
      setFaultCause("LocalDevice fault: " + localDevice.getFaultCause());
      return;
    }

    // Check mounted in network
    BComplex localDeviceParent = localDevice.getParent();
    if (!(localDeviceParent instanceof BBacnetNetwork))
    {
      fatalFault = true;
      setFaultCause("Not under BacnetNetwork");
      return;
    }
    BBacnetNetwork network = (BBacnetNetwork) localDeviceParent;

    // Check network fatal fault
    if (network.isFatalFault())
    {
      fatalFault = true;
      setFaultCause("Network fault: " + network.getFaultCause());
      return;
    }

    // Check license
    if (!network.hasServerLicense())
    {
      fatalFault = true;
      setFaultCause("Server capability not licensed");
      return;
    }

    // No fatal faults
    setFaultCause("");
  }

  @Override
  public void checkConfiguration()
  {
    if (fatalFault)
    {
      exportFault = true;
      updateReliability();
      logger.warning(this + ": is in fatal fault");
      return;
    }

    if (networkPort == null)
    {
      setStatusFaulted("Cannot find exported network port");
      return;
    }

    if (!getObjectId().isValid())
    {
      setStatusFaulted("Invalid Object ID");
      return;
    }

    if (getObjectName() == null || getObjectName().trim().isEmpty())
    {
      setStatusFaulted("Invalid Object Name");
      return;
    }

    String exportError = BBacnetNetwork.localDevice().export(this);
    if (exportError != null)
    {
      setStatusFaulted(exportError);
      return;
    }

    exportFault = false;

    setFaultCause("");
    updateReliability();
    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(this + ": configuration is ok");
    }
  }

  private void setStatusFaulted(String faultCause)
  {
    setFaultCause(faultCause);
    exportFault = true;
    updateReliability();
    logger.warning(this + ": " + faultCause);
  }

  @Override
  public final ChangeListError addListElements(PropertyValue propertyValue)
    throws BacnetException
  {
    int propertyId = propertyValue.getPropertyId();
    if (!hasProperty(propertyId))
    {
      return makeAddListElementError(property, unknownProperty);
    }

    if (!isListProp(propertyId))
    {
      return makeAddListElementError(services, propertyIsNotA_List);
    }

    // No list properties are an array of lists
    if (propertyValue.getPropertyArrayIndex() != NOT_USED)
    {
      return makeAddListElementError(property, propertyIsNotAnArray);
    }

    return addListElements(propertyId, propertyValue.getPropertyValue());
  }

  protected ChangeListError addListElements(int propertyId, byte[] value)
  {
    return makeAddListElementError(property, writeAccessDenied);
  }

  @Override
  public final ChangeListError removeListElements(PropertyValue propertyValue)
    throws BacnetException
  {
    int propertyId = propertyValue.getPropertyId();
    if (!hasProperty(propertyId))
    {
      return makeRemoveListElementError(property, unknownProperty);
    }

    if (!isListProp(propertyId))
    {
      return makeRemoveListElementError(services, propertyIsNotA_List);
    }

    // No list properties are an array of lists
    if (propertyValue.getPropertyArrayIndex() != NOT_USED)
    {
      return makeRemoveListElementError(property, propertyIsNotAnArray);
    }

    return removeListElements(propertyId, propertyValue.getPropertyValue());
  }

  protected ChangeListError removeListElements(int propertyId, byte[] value)
  {
    return makeRemoveListElementError(property, writeAccessDenied);
  }

  @Override
  public final int[] getPropertyList()
  {
    return BacnetPropertyList.makePropertyList(REQUIRED_PROPS, getOptionalProps());
  }

  protected abstract int[] getOptionalProps();
  protected abstract boolean isArrayProp(int propId);
  protected abstract boolean isListProp(int propId);

  protected boolean hasProperty(int propertyId)
  {
    for (int id : REQUIRED_PROPS)
    {
      if (id == propertyId)
      {
        return true;
      }
    }

    for (int id : getOptionalProps())
    {
      if (id == propertyId)
      {
        return true;
      }
    }

    // Property List is not included in either required or optional so check
    // that last.
    return propertyId == BBacnetPropertyIdentifier.PROPERTY_LIST;
  }

  //endregion

  //region ReadProperty

  @Override
  public final PropertyValue readProperty(PropertyReference ref)
    throws RejectException
  {
    return readProperty(ref.getPropertyId(), ref.getPropertyArrayIndex());
  }

  private PropertyValue readProperty(int propertyId, int index)
  {
    // Check for array index on non-array property.
    if (index >= 0)
    {
      if (!isArrayProp(propertyId))
      {
        return new NReadPropertyResult(propertyId, index, new NErrorType(property, propertyIsNotAnArray));
      }
    }
    else if (index < NOT_USED)
    {
      return new NReadPropertyResult(propertyId, index, new NErrorType(property, invalidArrayIndex));
    }

    switch (propertyId)
    {
      case BBacnetPropertyIdentifier.OBJECT_IDENTIFIER:
        return new NReadPropertyResult(propertyId, AsnUtil.toAsnObjectId(getObjectId()));

      case BBacnetPropertyIdentifier.OBJECT_NAME:
        return new NReadPropertyResult(propertyId, AsnUtil.toAsnCharacterString(getObjectName()));

      case BBacnetPropertyIdentifier.OBJECT_TYPE:
        return new NReadPropertyResult(propertyId, AsnUtil.toAsnEnumerated(BBacnetObjectType.networkPort));

      case BBacnetPropertyIdentifier.DESCRIPTION:
        return new NReadPropertyResult(propertyId, AsnUtil.toAsnCharacterString(getDescription()));

      case BBacnetPropertyIdentifier.STATUS_FLAGS:
        return new NReadPropertyResult(propertyId, AsnUtil.statusToAsnStatusFlags(getStatus()));

      case BBacnetPropertyIdentifier.RELIABILITY:
        return new NReadPropertyResult(propertyId, AsnUtil.toAsnEnumerated(getReliability()));

      case BBacnetPropertyIdentifier.OUT_OF_SERVICE:
        return new NReadPropertyResult(propertyId, AsnUtil.toAsnBoolean(!networkPort.getEnabled()));

      case BBacnetPropertyIdentifier.NETWORK_TYPE:
        return new NReadPropertyResult(propertyId, AsnUtil.toAsnEnumerated(getNetworkType()));

      case BBacnetPropertyIdentifier.PROTOCOL_LEVEL:
        return new NReadPropertyResult(propertyId, AsnUtil.toAsnEnumerated(BBacnetProtocolLevel.bacnetApplication));

      case BBacnetPropertyIdentifier.CHANGES_PENDING:
        return new NReadPropertyResult(propertyId, AsnUtil.toAsnBoolean(hasPendingChanges()));

      case BBacnetPropertyIdentifier.COMMAND:
        return makeEnumReadResult(BBacnetPropertyIdentifier.command, getCommand());

      case BBacnetPropertyIdentifier.PROPERTY_LIST:
        return readPropertyList(index);

      default:
        return readOptionalProperty(propertyId, index);
    }
  }

  protected abstract int getNetworkType();
  protected abstract PropertyValue readOptionalProperty(int propertyId, int index);

  //endregion

  //region ReadPropertyMultiple

  @Override
  public final PropertyValue[] readPropertyMultiple(PropertyReference[] propertyReferences)
    throws RejectException
  {
    ArrayList<PropertyValue> results = new ArrayList<>(propertyReferences.length);
    for (PropertyReference propertyReference : propertyReferences)
    {
      int propertyId = propertyReference.getPropertyId();
      switch (propertyId)
      {
        case BBacnetPropertyIdentifier.ALL:
          for (int prop : REQUIRED_PROPS)
          {
            results.add(readProperty(prop, NOT_USED));
          }
          for (int prop : getOptionalProps())
          {
            results.add(readProperty(prop, NOT_USED));
          }
          break;

        case BBacnetPropertyIdentifier.OPTIONAL:
          for (int prop : getOptionalProps())
          {
            results.add(readProperty(prop, NOT_USED));
          }
          break;

        case BBacnetPropertyIdentifier.REQUIRED:
          for (int prop : REQUIRED_PROPS)
          {
            results.add(readProperty(prop, NOT_USED));
          }
          break;

        default:
          results.add(
            readProperty(
              propertyId,
              propertyReference.getPropertyArrayIndex()));
          break;
      }
    }

    return results.toArray(EMPTY_PROPERTY_VALUE_ARRAY);
  }

  //endregion

  //region ReadRange

  @Override
  public final RangeData readRange(RangeReference rangeReference)
    throws RejectException
  {
    int propertyId = rangeReference.getPropertyId();
    if (!hasProperty(propertyId))
    {
      return new ReadRangeAck(property, unknownProperty);
    }

    if (!isListProp(propertyId))
    {
      return new ReadRangeAck(services, propertyIsNotA_List);
    }

    // No list properties are an array of lists
    if (rangeReference.getPropertyArrayIndex() != NOT_USED)
    {
      return new ReadRangeAck(property, propertyIsNotAnArray);
    }

    try
    {
      switch (rangeReference.getRangeType())
      {
        case RangeReference.BY_SEQUENCE_NUMBER:
          return new ReadRangeAck(property, listItemNotNumbered);
        case RangeReference.BY_TIME:
          return new ReadRangeAck(property, listItemNotTimestamped);
        case RangeReference.BY_POSITION:
          return readRangeByPosition(rangeReference, getListElements(propertyId));
        case NOT_USED:
          return readRangeAll(rangeReference, getListElements(propertyId));
        default:
          return new ReadRangeAck(services, parameterOutOfRange);
      }
    }
    catch (ErrorException e)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.log(Level.FINE, this + ": errorException readRange property " + propertyId + ": " + e);
      }
      return new ReadRangeAck(e.getErrorType().getErrorClass(), e.getErrorType().getErrorCode());
    }
  }

  protected abstract List<? extends BIBacnetDataType> getListElements(int propertyId)
    throws ErrorException;

  private RangeData readRangeByPosition(
    RangeReference rangeReference,
    List<? extends BIBacnetDataType> elements)
  {
    int elementCount = elements.size();
    int refIndex = (int) rangeReference.getReferenceIndex();
    if (refIndex < 1 || refIndex > elementCount)
    {
      return new ReadRangeAck(
        getObjectId(),
        rangeReference.getPropertyId(),
        NOT_USED,
        BBacnetBitString.emptyBitString(3),
        /* itemCount */ 0,
        EMPTY_BYTE_ARRAY);
    }

    int count = rangeReference.getCount();
    if (count == 0)
    {
      return new ReadRangeAck(services, inconsistentParameters);
    }

    return readRange(rangeReference, elements, refIndex, count);
  }

  private RangeData readRangeAll(
    RangeReference rangeReference,
    List<? extends BIBacnetDataType> elements)
  {
    return readRange(rangeReference, elements, 1, elements.size());
  }

  private ReadRangeAck readRange(
    RangeReference rangeReference,
    List<? extends BIBacnetDataType> elements,
    int refIndex,
    int count)
  {
    int maxDataLength = Integer.MAX_VALUE;
    if (rangeReference instanceof BacnetConfirmedRequest)
    {
      maxDataLength = ((BacnetConfirmedRequest) rangeReference).getMaxDataLength()
        // We need to subtract the size of the ReadRangeAck application headers.
        - ReadRangeAck.READ_RANGE_ACK_MAX_APP_HEADER_SIZE
        // We also add back in the length of the unused fields.
        + 3 // we don't use propertyArrayIndex here
        + 5; // we don't use sequenceNumber here
    }

    boolean firstItem = false;
    boolean lastItem = false;
    boolean moreItems = false;

    int itemLimit = Math.abs(count);
    int elementCount = elements.size();
    int itemCount = 0;
    ByteArrayOutputStream itemData = new ByteArrayOutputStream();

    AsnOutputStream asnOut = AsnOutputStream.make();
    try
    {
      if (count > 0)
      {
        // Count is positive: search from refIndex to end until we find count items.
        int i;
        for (i = refIndex - 1; i < elementCount && itemCount < itemLimit; ++i)
        {
          asnOut.reset();
          elements.get(i).writeAsn(asnOut);
          if (asnOut.size() + itemData.size() > maxDataLength)
          {
            moreItems = true;
            break;
          }

          asnOut.writeTo(itemData);
          ++itemCount;
        }

        if (itemData.size() > 0)
        {
          // Update the flags if we've encoded something
          firstItem = refIndex == 1;
          lastItem = i == elementCount;
        }
      }
      else
      {
        // Count is negative: search from refIndex to beginning until we find Math.abs(count) items.
        ByteArrayOutputStream temp = new ByteArrayOutputStream();

        int i;
        for (i = refIndex - 1; i >= 0 && itemCount < itemLimit; --i)
        {
          asnOut.reset();
          elements.get(i).writeAsn(asnOut);
          if (asnOut.size() + itemData.size() > maxDataLength)
          {
            moreItems = true;
            break;
          }

          // Prepend the element bytes
          temp.reset();
          itemData.writeTo(temp);
          itemData.reset();
          asnOut.writeTo(itemData);
          temp.writeTo(itemData);
          ++itemCount;
        }

        if (itemData.size() > 0)
        {
          // Update the flags if we've encoded something
          firstItem = (i + 1) == 0;
          lastItem = refIndex == elementCount;
        }
      }
    }
    catch (IOException e)
    {
      logger.log(
        Level.WARNING,
        this + ": error writing encoded read range item to item data",
        logger.isLoggable(Level.FINE) ? e : null);
      return new ReadRangeAck(services, other);
    }
    finally
    {
      asnOut.release();
    }

    return new ReadRangeAck(
      getObjectId(),
      rangeReference.getPropertyId(),
      NOT_USED,
      BBacnetBitString.make(new boolean[]{ firstItem, lastItem, moreItems }),
      itemCount,
      itemData.toByteArray());
  }

  //endregion

  //region WriteProperty

  @Override
  public final ErrorType writeProperty(PropertyValue val)
    throws BacnetException
  {
    return writeProperty(
      val.getPropertyId(),
      val.getPropertyArrayIndex(),
      val.getPropertyValue(),
      val.getPriority());
  }

  private ErrorType writeProperty(int propertyId, int index, byte[] value, int priority)
    throws BacnetException
  {
    // Check for array index on non-array property.
    if (index >= 0)
    {
      if (!isArrayProp(propertyId))
      {
        return new NErrorType(property, propertyIsNotAnArray);
      }
    }
    else if (index < NOT_USED)
    {
      return new NErrorType(property, invalidArrayIndex);
    }

    try
    {
      switch (propertyId)
      {
        case BBacnetPropertyIdentifier.OBJECT_IDENTIFIER:
        case BBacnetPropertyIdentifier.OBJECT_TYPE:
        case BBacnetPropertyIdentifier.STATUS_FLAGS:
        case BBacnetPropertyIdentifier.NETWORK_TYPE:
        case BBacnetPropertyIdentifier.PROTOCOL_LEVEL:
        case BBacnetPropertyIdentifier.CHANGES_PENDING:
        case BBacnetPropertyIdentifier.PROPERTY_LIST:
          if (logger.isLoggable(Level.FINE))
          {
            logger.fine(getObjectId() + ": attempted to write read-only property " + BBacnetPropertyIdentifier.tag(propertyId));
          }
          return new NErrorType(property, writeAccessDenied);

        case BBacnetPropertyIdentifier.OBJECT_NAME:
          BacUtil.setObjectName(this, objectName, value);
          return null;

        case BBacnetPropertyIdentifier.DESCRIPTION:
          setString(description, AsnUtil.fromAsnCharacterString(value), BLocalBacnetDevice.getBacnetContext());
          return null;

        case BBacnetPropertyIdentifier.RELIABILITY:
          return writeReliability(value);

        case BBacnetPropertyIdentifier.OUT_OF_SERVICE:
          return writeOutOfService(value);

        case BBacnetPropertyIdentifier.COMMAND:
          return writeCommand(value);

        default:
          return writeOptionalProperty(propertyId, index, value, priority);
      }
    }
    catch (OutOfRangeException e)
    {
      logger.log(
        Level.WARNING,
        this + ": OutOfRangeException writing property " + BBacnetPropertyIdentifier.tag(propertyId) + "- " + e,
        logger.isLoggable(Level.FINE) ? e : null);
      return new NErrorType(property, valueOutOfRange);
    }
    catch (AsnException e)
    {
      logger.log(
        Level.WARNING,
        this + ": AsnException writing property " + BBacnetPropertyIdentifier.tag(propertyId) + "- " + e,
        logger.isLoggable(Level.FINE) ? e : null);
      return new NErrorType(property, invalidDataType);
    }
    catch (PermissionException e)
    {
      logger.log(
        Level.WARNING,
        this + ": PermissionException writing property " + BBacnetPropertyIdentifier.tag(propertyId) + "- " + e,
        logger.isLoggable(Level.FINE) ? e : null);
      return new NErrorType(property, writeAccessDenied);
    }
  }

  protected abstract ErrorType writeOptionalProperty(int propertyId, int index, byte[] value, int priority)
    throws BacnetException;

  //endregion

  //region NetworkNumber

  protected PropertyValue readNetworkNumber()
  {
    int value = getNetworkNumber();
    if (value < 0 || value >= BBacnetUnsigned.MAX_UNSIGNED16_VALUE)
    {
      return makeReadError(networkNumber, property, valueOutOfRange);
    }
    return makeUnsignedReadResult(networkNumber, value);
  }

  private int getNetworkNumber()
  {
    BBacnetUnsigned pending = getPendingChange(NETWORK_NUMBER, BBacnetUnsigned.class);
    return pending != null ? pending.getInt() : networkPort.getNetworkNumber();
  }

  protected ErrorType writeNetworkNumber(byte[] value)
    throws AsnException
  {
    BBacnetUnsigned unsigned = AsnUtil.fromAsnUnsigned(value);
    if (!getNetworkLayer().isValidNetworkNumber(unsigned.getInt()))
    {
      return new NErrorType(property, valueOutOfRange);
    }

    addPendingChange(NETWORK_NUMBER, unsigned, BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  protected void validateSetNetworkNumber(BValue value, Context context)
  {
    checkPendingChangeType(NETWORK_NUMBER, value, BBacnetUnsigned.TYPE);
    checkCanWrite(networkPort, BNetworkPort.networkNumber, context);
  }

  protected void validateNetworkNumber(Context context)
    throws ValidateChangesException
  {
    BBacnetUnsigned change = getPendingChange(NETWORK_NUMBER, BBacnetUnsigned.class);
    if (change != null)
    {
      int value = change.getInt();

      if (isDuplicateNetworkNumber(value))
      {
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ other,
          /* property */ networkNumber,
          /* details */ "Value " + value + " is already used by another in-service network port");
      }

      if (!getNetworkLayer().isValidNetworkNumber(value))
      {
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ valueOutOfRange,
          /* property */ networkNumber,
          /* details */ "Value " + value + " is not in the range 0..65534 or is zero and device is a router");
      }

      validateCanWrite(networkPort, BNetworkPort.networkNumber, networkNumber, context);
    }
  }

  private boolean isDuplicateNetworkNumber(int pendingValue)
  {
    BBacnetStack comm = (BBacnetStack)BBacnetNetwork.bacnet().getBacnetComm();
    BNetworkPort[] ports = comm.getNetwork().getChildren(BNetworkPort.class);
    for (BNetworkPort port : ports)
    {
      BBacnetNetworkPortDescriptor descriptor = port.findDescriptor();
      if (descriptor == this)
      {
        continue;
      }

      if (port.getEnabled())
      {
        if (descriptor != null)
        {
          if (pendingValue == descriptor.getNetworkNumber())
          {
            return true;
          }
        }
        else if (pendingValue == port.getNetworkNumber())
        {
          return true;
        }
      }
    }
    return false;
  }

  protected void activateNetworkNumber(Context context)
  {
    BBacnetUnsigned change = getPendingChange(NETWORK_NUMBER, BBacnetUnsigned.class);
    if (change != null)
    {
      networkPort.setInt(BNetworkPort.networkNumber, change.getInt(), context);
    }
  }

  //endregion

  //region NetworkNumberQuality

  protected PropertyValue readNetworkNumberQuality()
  {
    return makeEnumReadResult(networkNumberQuality, networkPort.getNetworkNumberQuality());
  }

  //endregion

  //region ApduLength

  protected PropertyValue readApduLength()
  {
    int value = networkPort.getLink().getMaxAPDULengthAccepted();
    if (value < 0)
    {
      return makeReadError(apduLength, property, notConfigured);
    }
    return makeUnsignedReadResult(apduLength, value);
  }

  //endregion

  //region RoutingTable

  protected PropertyValue readRoutingTable()
  {
    BBacnetListOf routerEntries = new BBacnetListOf(BBacnetRoutingTableEntry.TYPE);
    for (BBacnetRoutingTableEntry routerEntry : getRouterEntries())
    {
      routerEntries.addListElement(routerEntry, null);
    }
    return makeListOfReadResult(routingTable, routerEntries);
  }

  protected List<BBacnetRoutingTableEntry> getRouterEntries()
  {
    int portId = networkPort.getPortId();
    BBacnetNetworkLayer networkLayer = getNetworkLayer();
    BBacnetRouterEntry[] entries = networkLayer.getRouterTable().getChildren(BBacnetRouterEntry.class);
    List<BBacnetRoutingTableEntry> routerEntries = new ArrayList<>();
    for (BBacnetRouterEntry entry : entries)
    {
      if (portId != entry.getPortId())
      {
        continue;
      }

      BBacnetRoutingTableEntry routerEntry = new BBacnetRoutingTableEntry();
      routerEntry.setNetworkNumber(BBacnetUnsigned.make(entry.getDnet()));
      routerEntry.setMacAddress(entry.getRouterAddress().getMacAddress());

      switch (entry.getRouterStatus().getOrdinal())
      {
        case BRouterStatus.OK:
          routerEntry.setStatus(available);
          break;
        case BRouterStatus.ROUTER_BUSY:
          routerEntry.setStatus(BBacnetRouterStatus.busy);
          break;
        case BRouterStatus.ROUTER_UNAVAILABLE:
        case BRouterStatus.ROUTER_NOT_CONNECTED:
        case BRouterStatus.UNKNOWN:
        default:
          routerEntry.setStatus(disconnected);
          break;
      }

      routerEntry.setPerformanceIndex(BBacnetUnsigned.make(entry.getPerformanceIndex()));
      routerEntries.add(routerEntry);
    }

    return routerEntries;
  }

  //endregion

  //region Pending Changes

  public boolean hasPendingChanges()
  {
    return getPendingChanges().getChildren(BValue.class).length > 0;
  }

  protected void addPendingChange(String propertyName, BValue value, Context context)
  {
    BBacnetNetworkPortPendingChanges pendingChanges = getPendingChanges();
    Property existing = pendingChanges.getProperty(propertyName);
    if (existing != null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        BValue oldValue = pendingChanges.get(existing);
        logger.fine(getObjectId() + ": replacing pending change value for " + propertyName +
          "; oldValue: " + oldValue +
          ", newValue: " + value);
      }
      pendingChanges.set(existing, value, context);
      return;
    }

    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(getObjectId() + ": adding pending change value for " + propertyName +
        "; newValue: " + value);
    }
    pendingChanges.add(propertyName, value, context);
  }

  @SuppressWarnings("unchecked")
  protected final <T extends BValue> T getPendingChange(String name, Class<T> clazz)
  {
    BBacnetNetworkPortPendingChanges pendingChanges = getPendingChanges();
    BValue pendingChange = pendingChanges.get(name);
    if (pendingChange == null)
    {
      return null;
    }

    if (!pendingChange.getClass().equals(clazz))
    {
      logger.warning(getObjectId() + ": Removing PendingChange with incorrect type" +
        "; name: " + name +
        ", expected class: " + clazz +
        ", actual class: " + pendingChange.getClass());
      pendingChanges.remove(name);
      return null;
    }

    return (T) pendingChange;
  }

  public void discardPendingChanges(Context context)
  {
    getPendingChanges().removeAll(context);
  }

  //endregion

  //region Reliability

  private ErrorType writeReliability(byte[] value)
    throws AsnException
  {
    BBacnetReliability reliabilityValue = BBacnetReliability.make(AsnUtil.fromAsnEnumerated(value));

    if (networkPort.getEnabled())
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(this + ": attempt to write reliability when port is not out-of-service");
      }
      return new NErrorType(property, writeAccessDenied);
    }

    set(reliability, reliabilityValue, BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  public void updateReliability()
  {
    if (networkPort != null && !networkPort.getEnabled())
    {
      // When out-of-service, do not change the reliability value.
      return;
    }

    if ((networkPort != null && networkPort.getStatus().isFault()) ||
        fatalFault ||
        exportFault)
    {
      setReliability(BBacnetReliability.configurationError);
      return;
    }

    setReliability(BBacnetReliability.noFaultDetected);
  }

  //endregion

  //region OutOfService

  private ErrorType writeOutOfService(byte[] value)
    throws AsnException
  {
    boolean outOfServiceValue = AsnUtil.fromAsnBoolean(value);
    // All BACnet communication through the port shall be disabled When Out_Of_Service is TRUE
    networkPort.setBoolean(BNetworkPort.enabled, !outOfServiceValue, BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  //endregion

  //region Command

  private ErrorType writeCommand(byte[] value)
    throws AsnException
  {
    return handleCommand(AsnUtil.fromAsnEnumerated(value), BLocalBacnetDevice.getBacnetContext());
  }

  public void doExecuteCommand(BDynamicEnum command, Context context)
    throws Exception
  {
    ErrorType error = handleCommand(command.getOrdinal(), context);
    if (error != null)
    {
      throw new BajaException(this + ": could not execute command " + command + "; error: " + error);
    }
  }

  private ErrorType handleCommand(int command, Context context)
  {
    if (getCommand().getOrdinal() != BBacnetNetworkPortCommand.IDLE)
    {
      if (logger.isLoggable(Level.INFO))
      {
        logger.info(this + ": attempt to write command value " + BBacnetNetworkPortCommand.tag(command) +
          " when current value is not Idle");
      }
      return new NErrorType(object, busy);
    }

    if (command == BBacnetNetworkPortCommand.IDLE)
    {
      if (logger.isLoggable(Level.INFO))
      {
        logger.info(this + ": attempt to write command value Idle");
      }
      return new NErrorType(property, valueOutOfRange);
    }

    if (hasPendingChanges() &&
        command != BBacnetNetworkPortCommand.DISCARD_CHANGES &&
        command != BBacnetNetworkPortCommand.VALIDATE_CHANGES)
    {
      if (logger.isLoggable(Level.INFO))
      {
        logger.info(this + ": attempt to write command value " + BBacnetNetworkPortCommand.tag(command) +
          " when there are pending changes; only the Discard Changes or Validate Changes commands are allowed");
      }
      return new NErrorType(property, invalidValueInThisState);
    }

    if (!networkPort.getEnabled())
    {
      switch (command)
      {
        case BBacnetNetworkPortCommand.RESTART_PORT:
        case BBacnetNetworkPortCommand.DISCONNECT:
        case BBacnetNetworkPortCommand.DISCARD_CHANGES:
          // Ok when port is disabled (out-of-service)
          break;
        default:
          if (logger.isLoggable(Level.INFO))
          {
            logger.info(this + ": attempt to write command value " + BBacnetNetworkPortCommand.tag(command) +
              "when port is out of service");
          }
          return new NErrorType(property, valueOutOfRange);
      }
    }

    switch (command)
    {
      case BBacnetNetworkPortCommand.DISCARD_CHANGES:
        return handleDiscardChangesCommand(context);

      case BBacnetNetworkPortCommand.RENEW_FD_REGISTRATION:
        return handleRenewFdRegistrationCommand(context);

      case BBacnetNetworkPortCommand.RESTART_SLAVE_DISCOVERY:
        return handleRestartSlaveDiscovery();

      case BBacnetNetworkPortCommand.RENEW_DHCP:
        return handleRenewDhcpCommand(context);

      case BBacnetNetworkPortCommand.RESTART_AUTONEGOTIATION:
        return handleRestartAutoNegotiationCommand();

      case BBacnetNetworkPortCommand.DISCONNECT:
        return handleDisconnectCommand();

      case BBacnetNetworkPortCommand.RESTART_PORT:
        return handleRestartPortCommand(context);

      case BBacnetNetworkPortCommand.GENERATE_CSR_FILE:
        return handleGenerateCsrFileCommand(context);

      case BBacnetNetworkPortCommand.VALIDATE_CHANGES:
        return handleValidateChangesCommand(context);

      default:
        return handleProprietaryCommand(command);
    }
  }

  private ErrorType handleDiscardChangesCommand(Context context)
  {
    set(command, BBacnetNetworkPortCommand.discardChanges, context);
    try
    {
      if (hasPendingChanges())
      {
        discardPendingChanges(context);
      }

      if (networkPort.getEnabled())
      {
        updateReliability();
      }
    }
    catch (PermissionException e)
    {
      if (logger.isLoggable(Level.INFO))
      {
        logger.log(
          Level.INFO,
          this + ": attempt to write command value Discard Changes but user does not have permission to remove pending changes.",
          logger.isLoggable(Level.FINE) ? e : null);
      }

      if (networkPort.getEnabled())
      {
        setReliability(processError);
      }
    }
    finally
    {
      set(command, BBacnetNetworkPortCommand.idle, context);
    }
    return null;
  }

  protected ErrorType handleRenewFdRegistrationCommand(Context context)
  {
    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(this + ": attempt to write command value Renew FD Registration" +
        " when network type is not IPv4 or IPv6: " + BBacnetNetworkType.tag(getNetworkType()));
    }
    return new NErrorType(property, valueOutOfRange);
  }

  protected ErrorType handleRestartSlaveDiscovery()
  {
    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(this + ": attempt to write command value Restart Slave Discovery" +
        " when network type is not MSTP: " + BBacnetNetworkType.tag(getNetworkType()));
    }
    return new NErrorType(property, valueOutOfRange);
  }

  protected ErrorType handleRenewDhcpCommand(Context context)
  {
    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(this + ": attempt to write command value Renew DHCP" +
        " when network type is not IPv4 or IPv6: " + BBacnetNetworkType.tag(getNetworkType()));
    }
    return new NErrorType(property, valueOutOfRange);
  }

  private ErrorType handleRestartAutoNegotiationCommand()
  {
    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(this + ": Restart Auto-Negotiation command is not supported");
    }
    return new NErrorType(property, optionalFunctionalityNotSupported);
  }

  private ErrorType handleDisconnectCommand()
  {
    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(this + ": attempt to write command value Disconnect" +
        " when network type is not PTP: " + BBacnetNetworkType.tag(getNetworkType()));
    }
    return new NErrorType(property, valueOutOfRange);
  }

  private ErrorType handleRestartPortCommand(Context context)
  {
    set(command, BBacnetNetworkPortCommand.restartPort, context);
    try
    {
      if (networkPort.getEnabled())
      {
        networkPort.invoke(BNetworkPort.disable, null, context);
      }
      networkPort.invoke(BNetworkPort.enable, null, context);

      // Cannot set the reliability to restartFailed if enabling the port fails because the port is
      // out-of-service and reliability is decoupled from the port. Can reset the reliability if the
      // command is successful and the port is not out-of-service.
      if (networkPort.getEnabled())
      {
        updateReliability();
      }
    }
    catch (PermissionException e)
    {
      if (logger.isLoggable(Level.INFO))
      {
        logger.log(
          Level.INFO,
          this + ": attempt to write command value Restart Port but user does not have permission on network port.",
          logger.isLoggable(Level.FINE) ? e : null);
      }

      if (networkPort.getEnabled())
      {
        setReliability(processError);
      }
    }
    finally
    {
      set(command, BBacnetNetworkPortCommand.idle, context);
    }
    return null;
  }

  protected ErrorType handleGenerateCsrFileCommand(Context context)
  {
    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(this + ": Generate CSR File command is not supported");
    }
    return new NErrorType(property, optionalFunctionalityNotSupported);
  }

  private ErrorType handleValidateChangesCommand(Context context)
  {
    set(command, BBacnetNetworkPortCommand.validateChanges, context);
    try
    {
      validateChanges(context);
    }
    catch (ValidateChangesException e)
    {
      if (logger.isLoggable(Level.INFO))
      {
        logger.log(
          Level.INFO, this + ": Validate Changes failed; exception: " + e,
          logger.isLoggable(Level.FINE) ? e : null);
      }
      return e.getError();
    }
    set(command, BBacnetNetworkPortCommand.idle, context);
    return null;
  }

  private ErrorType handleProprietaryCommand(int commandValue)
  {
    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(this + ": unknown command value " + commandValue);
    }
    return new NErrorType(property, valueOutOfRange);
  }

  //endregion

  //region Validate/Activate Changes

  public boolean activateChangesRequiresReboot()
  {
    return false;
  }

  protected abstract void validateSetPendingChange(String name, BValue value, Context context);
  public abstract void validateChanges(Context context)
    throws ValidateChangesException;
  public abstract void activateChanges(Context context)
    throws Exception;

  public void doActivatePendingChanges(BBoolean activateIfRebootRequired, Context context)
  {
    BNetworkPort networkPort = this.networkPort;
    if (networkPort == null)
    {
      throw new LocalizableRuntimeException("bacnet", "networkPortDescriptor.activatePendingChanges.noNetworkPort");
    }

    if (!hasPendingChanges())
    {
      throw new LocalizableRuntimeException("bacnet", "networkPortDescriptor.activatePendingChanges.noPendingChanges");
    }

    boolean rebootRequired = activateChangesRequiresReboot();
    if (rebootRequired && !activateIfRebootRequired.getBoolean())
    {
      throw new LocalizableRuntimeException("bacnet", "networkPortDescriptor.activatePendingChanges.requireReboot");
    }

    try
    {
      validateChanges(context);

      boolean wasEnabled = networkPort.getEnabled();
      networkPort.setBoolean(BNetworkPort.enabled, false, context);
      activateChanges(context);
      discardPendingChanges(context);
      if (wasEnabled)
      {
        networkPort.setBoolean(BNetworkPort.enabled, true, context);
      }
    }
    catch (Exception e)
    {
      throw new LocalizableRuntimeException("bacnet", "networkPortDescriptor.activatePendingChanges.failed", new Object[] { e }, e);
    }

    if (rebootRequired)
    {
      throw new LocalizableRuntimeException("bacnet", "networkPortDescriptor.activatePendingChanges.manualReboot");
    }
  }

  //endregion

  //region Utility

  private void resolveNetworkPort()
  {
    BNetworkPort resolved = null;
    BOrd networkPortOrd = getNetworkPortOrd();
    try
    {
      BObject target = networkPortOrd.get(this);
      if (target instanceof BNetworkPort)
      {
        resolved = (BNetworkPort)target;
      }
      else
      {
        logger.warning(getObjectId() + ": Network Port Ord does not resolve to a BNetworkPort: " + networkPortOrd);
      }
    }
    catch (BajaRuntimeException e)
    {
      logger.log(
        Level.WARNING,
        getObjectId() + ": unable to resolve Network Port Ord: " + networkPortOrd,
        logger.isLoggable(Level.FINE) ? e : null);
    }
    networkPort = resolved;
  }

  private BBacnetNetworkLayer getNetworkLayer()
  {
    return (BBacnetNetworkLayer)networkPort.getParent();
  }

  protected void checkCanWrite(BComponent target, Context context)
  {
    BUser user = context != null ? context.getUser() : null;
    if (user != null)
    {
      user.check(target, BPermissions.adminWrite);
    }
  }

  protected void checkCanWrite(BComponent target, Slot slot, Context context)
  {
    BUser user = context != null ? context.getUser() : null;
    if (user != null)
    {
      user.checkWrite(target, slot);
    }
  }

  protected void checkCanReadPendingChanges(Context context)
    throws ValidateChangesException
  {
    BUser user = getUser(context, readAccessDenied);
    if (!user.getPermissionsFor(getPendingChanges()).hasAdminRead())
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(this + ": User missing admin read permissions on pending changes");
      }
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ readAccessDenied,
        /* property */ null,
        /* details */ "User missing read permissions on pending changes");
    }
  }

  protected void validateCanWrite(
    BComponent target,
    BBacnetPropertyIdentifier propertyId,
    Context context)
      throws ValidateChangesException
  {
    BUser user = getUser(context, writeAccessDenied);
    try
    {
      user.check(target, BPermissions.adminWrite);
    }
    catch (PermissionException e)
    {
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ writeAccessDenied,
        /* property */ propertyId,
        /* details */ "User missing write permissions");
    }
  }

  protected void validateCanWrite(
    BComponent target,
    Slot slot,
    BBacnetPropertyIdentifier propertyId,
    Context context)
      throws ValidateChangesException
  {
    BUser user = getUser(context, writeAccessDenied);
    try
    {
      user.checkWrite(target, slot);
    }
    catch (PermissionException e)
    {
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ writeAccessDenied,
        /* property */ propertyId,
        /* details */ "User missing write permissions");
    }
  }

  protected BUser getUser(Context context, BBacnetErrorCode errorCode)
    throws ValidateChangesException
  {
    BUser user = context != null ? context.getUser() : null;
    if (user == null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(this + ": missing context or context user");
      }
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ errorCode,
        /* property */ null,
        /* details */ "Missing write or read permissions");
    }

    return user;
  }

  protected void checkPendingChangeType(String name, BValue value, Type type)
  {
    if (!value.getType().is(type))
    {
      throw new LocalizableRuntimeException(
        "bacnet",
        "networkPortDescriptor.pendingChange.wrongType",
        new Object[] { name, value.getType(), type });
    }
  }

  //endregion

  //region Fields

  public static final Context atStarted = new BasicContext();

  protected static final String NETWORK_NUMBER = "NetworkNumber";

  protected static final Logger logger = Logger.getLogger("bacnet.server");

  private static final Context skipExport = new BasicContext();

  private static final int[] REQUIRED_PROPS = {
    BBacnetPropertyIdentifier.OBJECT_IDENTIFIER,
    BBacnetPropertyIdentifier.OBJECT_NAME,
    BBacnetPropertyIdentifier.OBJECT_TYPE,
    BBacnetPropertyIdentifier.STATUS_FLAGS,
    BBacnetPropertyIdentifier.RELIABILITY,
    BBacnetPropertyIdentifier.OUT_OF_SERVICE,
    BBacnetPropertyIdentifier.NETWORK_TYPE,
    BBacnetPropertyIdentifier.PROTOCOL_LEVEL,
    BBacnetPropertyIdentifier.CHANGES_PENDING
  };

  protected BNetworkPort networkPort;
  protected boolean fatalFault;

  private BBacnetObjectIdentifier oldId = BBacnetObjectIdentifier.make(BBacnetObjectType.NETWORK_PORT);
  private String oldName = "";
  private boolean exportFault;

  //endregion
}
