/*
 * Copyright 2024 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.export;

import static javax.baja.bacnet.enums.BBacnetErrorClass.property;
import static javax.baja.bacnet.enums.BBacnetErrorCode.optionalFunctionalityNotSupported;
import static javax.baja.bacnet.enums.BBacnetErrorCode.unknownProperty;
import static javax.baja.bacnet.enums.BBacnetErrorCode.valueOutOfRange;
import static javax.baja.bacnet.enums.BBacnetErrorCode.writeAccessDenied;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.linkSpeed;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.linkSpeedAutonegotiate;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.linkSpeeds;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.macAddress;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.maxInfoFrames;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.maxMaster;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.networkInterfaceName;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeBooleanReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeCharStringReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeOctetStringReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeRealReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeUnsignedReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.readArray;

import java.util.List;
import java.util.logging.Level;

import javax.baja.bacnet.BacnetException;
import javax.baja.bacnet.datatypes.BBacnetArray;
import javax.baja.bacnet.datatypes.BBacnetOctetString;
import javax.baja.bacnet.datatypes.BBacnetUnsigned;
import javax.baja.bacnet.datatypes.BIBacnetDataType;
import javax.baja.bacnet.enums.BBacnetNetworkType;
import javax.baja.bacnet.enums.BBacnetPropertyIdentifier;
import javax.baja.bacnet.io.AsnException;
import javax.baja.bacnet.io.ErrorException;
import javax.baja.bacnet.io.ErrorType;
import javax.baja.bacnet.io.PropertyValue;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.sys.BFloat;
import javax.baja.sys.BValue;
import javax.baja.sys.Context;
import javax.baja.sys.LocalizableRuntimeException;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.bacnet.asn.AsnUtil;
import com.tridium.bacnet.asn.NErrorType;
import com.tridium.bacnet.asn.NReadPropertyResult;
import com.tridium.bacnet.enums.BBacnetMstpBaudRate;
import com.tridium.bacnet.stack.link.mstp.BBacnetMstpLinkLayer;

/**
 * BBacnetMstpPortDescriptor exports a BNetworkPort that contains a BBacnetMstpLinkLayer as a BACnet
 * Network Port object with the Network Type set to MSTP.
 *
 * @author Sandipan Aich on 22-July-2024
 * @since Niagara 4.15
 */
@NiagaraType
public class BBacnetMstpPortDescriptor
  extends BBacnetNetworkPortDescriptor
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.export.BBacnetMstpPortDescriptor(2979906276)1.0$ @*/
/* Generated Tue Jul 16 14:56:28 EDT 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetMstpPortDescriptor.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  //region BBacnetNetworkPortDescriptor

  @Override
  protected int getNetworkType()
  {
    return BBacnetNetworkType.MSTP;
  }

  @Override
  protected int[] getOptionalProps()
  {
    //noinspection AssignmentOrReturnOfFieldWithMutableType
    return OPTIONAL_PROPS;
  }

  @Override
  protected boolean isArrayProp(int propId)
  {
    switch (propId)
    {
      case BBacnetPropertyIdentifier.LINK_SPEEDS:
      case BBacnetPropertyIdentifier.PROPERTY_LIST:
        return true;
      default:
        return false;
    }
  }

  @Override
  protected boolean isListProp(int propId)
  {
    return propId == BBacnetPropertyIdentifier.ROUTING_TABLE;
  }

  //endregion

  //region ReadProperty

  @Override
  protected PropertyValue readOptionalProperty(int propertyId, int index)
  {
    switch (propertyId)
    {
      case BBacnetPropertyIdentifier.NETWORK_NUMBER:
        return readNetworkNumber();

      case BBacnetPropertyIdentifier.NETWORK_NUMBER_QUALITY:
        return readNetworkNumberQuality();

      case BBacnetPropertyIdentifier.APDU_LENGTH:
        return readApduLength();

      case BBacnetPropertyIdentifier.ROUTING_TABLE:
        return readRoutingTable();

      case BBacnetPropertyIdentifier.MAC_ADDRESS:
        return readMacAddress();

      case BBacnetPropertyIdentifier.MAX_MASTER:
        return readMaxMaster();

      case BBacnetPropertyIdentifier.MAX_INFO_FRAMES:
        return readMaxInfoFrames();

      case BBacnetPropertyIdentifier.LINK_SPEED:
        return readLinkSpeed();

      case BBacnetPropertyIdentifier.LINK_SPEEDS:
        return readLinkSpeeds(index);

      case BBacnetPropertyIdentifier.LINK_SPEED_AUTONEGOTIATE:
        return readLinkSpeedAutoNegotiate();

      case BBacnetPropertyIdentifier.NETWORK_INTERFACE_NAME:
        return readNetworkInterfaceName();

      default:
        return new NReadPropertyResult(propertyId, new NErrorType(property, unknownProperty));
    }
  }
  //endregion

  //region ReadRange

  @Override
  protected List<? extends BIBacnetDataType> getListElements(int propertyId)
    throws ErrorException
  {
    if (propertyId == BBacnetPropertyIdentifier.ROUTING_TABLE)
    {
      return getRouterEntries();
    }

    throw new IllegalArgumentException("Invalid propertyId value " + propertyId);
  }

  //endregion

  //region WriteProperty

  @Override
  protected ErrorType writeOptionalProperty(int propertyId, int index, byte[] value, int priority)
    throws BacnetException
  {
    switch (propertyId)
    {
      case BBacnetPropertyIdentifier.NETWORK_NUMBER_QUALITY:
      case BBacnetPropertyIdentifier.APDU_LENGTH:
      case BBacnetPropertyIdentifier.ROUTING_TABLE:
      case BBacnetPropertyIdentifier.LINK_SPEEDS:
      case BBacnetPropertyIdentifier.LINK_SPEED_AUTONEGOTIATE:
      case BBacnetPropertyIdentifier.NETWORK_INTERFACE_NAME:
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(this + ": attempted to write read-only property " + BBacnetPropertyIdentifier.tag(propertyId));
        }
        return new NErrorType(property, writeAccessDenied);

      case BBacnetPropertyIdentifier.NETWORK_NUMBER:
        return writeNetworkNumber(value);

      case BBacnetPropertyIdentifier.MAC_ADDRESS:
        return writeMacAddress(value);

      case BBacnetPropertyIdentifier.MAX_MASTER:
        return writeMaxMaster(value);

      case BBacnetPropertyIdentifier.MAX_INFO_FRAMES:
        return writeMaxInfoFrames(value);

      case BBacnetPropertyIdentifier.LINK_SPEED:
        return writeLinkSpeed(value);

      default:
        return new NErrorType(property, unknownProperty);
    }
  }

  //endregion

  //region Validate/Activate Changes

  @Override
  protected void validateSetPendingChange(String name, BValue value, Context context)
  {
    switch(name)
    {
      case NETWORK_NUMBER:
        validateSetNetworkNumber(value, context);
        break;
      case MAC_ADDRESS:
        validateSetMacAddress(value, context);
        break;
      case MAX_MASTER:
        validateSetMaxMaster(value, context);
        break;
      case MAX_INFO_FRAMES:
        validateSetMaxInfoFrames(value, context);
        break;
      case LINK_SPEED:
        validateSetLinkSpeed(value, context);
        break;
      default:
        throw new LocalizableRuntimeException(
          "bacnet",
          "networkPortDescriptor.pendingChange.unknown",
          new Object[] { name });
    }
  }

  @Override
  public void validateChanges(Context context)
    throws ValidateChangesException
  {
    if (!hasPendingChanges())
    {
      return;
    }

    checkCanReadPendingChanges(context);

    validateNetworkNumber(context);
    validateMacAddress(context);
    validateMaxMaster(context);
    validateMaxInfoFrames(context);
    validateLinkSpeed(context);
  }

  @Override
  public void activateChanges(Context context)
    throws Exception
  {
    if (!hasPendingChanges())
    {
      return;
    }

    activateNetworkNumber(context);
    activateMacAddress(context);
    activateMaxMaster(context);
    activateMaxInfoFrames(context);
    activateLinkSpeed(context);
  }

  //endregion

  //region Command

  @Override
  protected ErrorType handleRestartSlaveDiscovery()
  {
    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(this + ": Restart Slave Discovery is not supported");
    }
    return new NErrorType(property, optionalFunctionalityNotSupported);
  }

  //endregion

  //region MacAddress

  private PropertyValue readMacAddress()
  {
    BBacnetOctetString pendingMacAddress = getPendingChange(MAC_ADDRESS, BBacnetOctetString.class);
    if (pendingMacAddress != null)
    {
      return makeOctetStringReadResult(macAddress, pendingMacAddress);
    }
    else
    {
      return makeOctetStringReadResult(macAddress, getMstpLink().getMacAddress());
    }
  }

  private ErrorType writeMacAddress(byte[] value)
    throws AsnException
  {
    BBacnetOctetString macAddress = AsnUtil.fromAsnOctetString(value);
    try
    {
      validateMacAddress(macAddress);
    }
    catch (ValidateChangesException e)
    {
      return e.getError();
    }

    addPendingChange(MAC_ADDRESS, macAddress, BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  private void validateSetMacAddress(BValue value, Context context)
  {
    checkPendingChangeType(MAC_ADDRESS, value, BBacnetOctetString.TYPE);
    checkCanWrite(getMstpLink(), BBacnetMstpLinkLayer.mstpAddress, context);
  }

  private void validateMacAddress(Context context)
    throws ValidateChangesException
  {
    BBacnetOctetString change = getPendingChange(MAC_ADDRESS, BBacnetOctetString.class);
    if (change != null)
    {
      validateMacAddress(change);
      validateCanWrite(getMstpLink(), BBacnetMstpLinkLayer.mstpAddress, macAddress, context);
    }
  }

  private static void validateMacAddress(BBacnetOctetString value)
    throws ValidateChangesException
  {
    if (value.isNull() || value.length() != 1)
    {
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ valueOutOfRange,
        /* property */ macAddress,
        /* details */ "Value \"[" + value + "]\" does not have a length of one.");
    }

    byte element = value.getAddr()[0];
    if (element < 0 || element > 127)
    {
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ valueOutOfRange,
        /* property */ macAddress,
        /* details */ "Value " + element + " is not in the range of 0-127.");
    }
  }

  private void activateMacAddress(Context context)
  {
    BBacnetOctetString change = getPendingChange(MAC_ADDRESS, BBacnetOctetString.class);
    if (change != null)
    {
      getMstpLink().setInt(BBacnetMstpLinkLayer.mstpAddress, change.getAddr()[0], context);
    }
  }

  //endregion

  //region MaxMaster

  public int getMaxMaster()
  {
    BBacnetUnsigned maxMaster = getPendingChange(MAX_MASTER, BBacnetUnsigned.class);
    if (maxMaster != null)
    {
      return maxMaster.getInt();
    }
    BBacnetMstpLinkLayer mstpLink = getMstpLink();
    return mstpLink != null ? mstpLink.getMaxMaster() : 127;
  }

  private PropertyValue readMaxMaster()
  {
    return makeUnsignedReadResult(maxMaster, getMaxMaster());
  }

  private ErrorType writeMaxMaster(byte[] value)
    throws AsnException
  {
    BBacnetUnsigned maxMaster = AsnUtil.fromAsnUnsigned(value);
    try
    {
      validateMaxMaster(maxMaster);
    }
    catch (ValidateChangesException e)
    {
      return e.getError();
    }

    addPendingChange(MAX_MASTER, maxMaster, BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  private void validateSetMaxMaster(BValue value, Context context)
  {
    checkPendingChangeType(MAX_MASTER, value, BBacnetUnsigned.TYPE);
    checkCanWrite(getMstpLink(), BBacnetMstpLinkLayer.maxMaster, context);
  }

  private void validateMaxMaster(Context context)
    throws ValidateChangesException
  {
    BBacnetUnsigned change = getPendingChange(MAX_MASTER, BBacnetUnsigned.class);
    if (change != null)
    {
      validateMaxMaster(change);
      validateCanWrite(getMstpLink(), BBacnetMstpLinkLayer.maxMaster, maxMaster, context);
    }
  }

  private static void validateMaxMaster(BBacnetUnsigned value)
    throws ValidateChangesException
  {
    int intValue = value.getInt();
    if (intValue < 0 || intValue > 127)
    {
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ valueOutOfRange,
        /* property */ maxMaster,
        /* details */ "Value " + value + " is not in the range of 0-127.");
    }
  }

  private void activateMaxMaster(Context context)
  {
    BBacnetUnsigned change = getPendingChange(MAX_MASTER, BBacnetUnsigned.class);
    if (change != null)
    {
      getMstpLink().setInt(BBacnetMstpLinkLayer.maxMaster, change.getInt(), context);
    }
  }

  //endregion

  //region MaxInfoFrames

  public int getMaxInfoFrames()
  {
    BBacnetUnsigned maxInfoFrames = getPendingChange(MAX_INFO_FRAMES, BBacnetUnsigned.class);
    if (maxInfoFrames != null)
    {
      return maxInfoFrames.getInt();
    }
    BBacnetMstpLinkLayer mstpLink = getMstpLink();
    return mstpLink != null ? mstpLink.getMaxInfoFrames() : 1;
  }

  private PropertyValue readMaxInfoFrames()
  {
    return makeUnsignedReadResult(maxInfoFrames, getMaxInfoFrames());
  }

  private ErrorType writeMaxInfoFrames(byte[] value)
    throws AsnException
  {
    BBacnetUnsigned maxInfoFrames = AsnUtil.fromAsnUnsigned(value);
    try
    {
      validateMaxInfoFrames(maxInfoFrames);
    }
    catch (ValidateChangesException e)
    {
      return e.getError();
    }

    addPendingChange(MAX_INFO_FRAMES, maxInfoFrames, BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  private void validateSetMaxInfoFrames(BValue value, Context context)
  {
    checkPendingChangeType(MAX_INFO_FRAMES, value, BBacnetUnsigned.TYPE);
    checkCanWrite(getMstpLink(), BBacnetMstpLinkLayer.maxInfoFrames, context);
  }

  private void validateMaxInfoFrames(Context context)
    throws ValidateChangesException
  {
    BBacnetUnsigned change = getPendingChange(MAX_INFO_FRAMES, BBacnetUnsigned.class);
    if (change != null)
    {
      validateMaxInfoFrames(change);
      validateCanWrite(getMstpLink(), BBacnetMstpLinkLayer.maxInfoFrames, maxInfoFrames, context);
    }
  }

  private static void validateMaxInfoFrames(BBacnetUnsigned value)
    throws ValidateChangesException
  {
    int intValue = value.getInt();
    if (intValue < 1 || intValue > 255)
    {
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ valueOutOfRange,
        /* property */ maxInfoFrames,
        /* details */ "Value " + value + " is not in the range of 1-255.");
    }
  }

  private void activateMaxInfoFrames(Context context)
  {
    BBacnetUnsigned change = getPendingChange(MAX_INFO_FRAMES, BBacnetUnsigned.class);
    if (change != null)
    {
      getMstpLink().setInt(BBacnetMstpLinkLayer.maxInfoFrames, change.getInt(), context);
    }
  }

  //endregion

  //region LinkSpeed

  private PropertyValue readLinkSpeed()
  {
    BFloat pendingChange = getPendingChange(LINK_SPEED, BFloat.class);
    if (pendingChange != null)
    {
      return makeRealReadResult(linkSpeed, pendingChange.getFloat());
    }

    BBacnetMstpBaudRate baudRate = getMstpLink().getBaudRate();
    return makeRealReadResult(linkSpeed, baudRate.getOrdinal());
  }

  private ErrorType writeLinkSpeed(byte[] value)
    throws AsnException
  {
    float linkSpeed = AsnUtil.fromAsnReal(value);
    try
    {
      validateLinkSpeed(linkSpeed);
    }
    catch (ValidateChangesException e)
    {
      return e.getError();
    }

    addPendingChange(LINK_SPEED, BFloat.make(linkSpeed), BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  private void validateSetLinkSpeed(BValue value, Context context)
  {
    checkPendingChangeType(LINK_SPEED, value, BFloat.TYPE);
    checkCanWrite(getMstpLink(), BBacnetMstpLinkLayer.baudRate, context);
  }

  private void validateLinkSpeed(Context context)
    throws ValidateChangesException
  {
    BFloat change = getPendingChange(LINK_SPEED, BFloat.class);
    if (change != null)
    {
      validateLinkSpeed(change.getFloat());
      validateCanWrite(getMstpLink(), BBacnetMstpLinkLayer.baudRate, linkSpeed, context);
    }
  }

  private static void validateLinkSpeed(float value)
    throws ValidateChangesException
  {
    if (!isLinkSpeedValid(value))
    {
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ valueOutOfRange,
        /* property */ linkSpeed,
        /* details */ "Value " + (int) value + " is not one of the valid link speeds.");
    }
  }

  private void activateLinkSpeed(Context context)
  {
    BFloat change = getPendingChange(LINK_SPEED, BFloat.class);
    if (change != null)
    {
      BBacnetMstpBaudRate mstpBaudRate = BBacnetMstpBaudRate.make(change.getInt());
      getMstpLink().set(BBacnetMstpLinkLayer.baudRate, mstpBaudRate, context);
    }
  }

  private static boolean isLinkSpeedValid(float value)
  {
    for (int linkSpeed : VALID_LINK_SPEEDS)
    {
      //noinspection FloatingPointEquality
      if (value == linkSpeed)
      {
        return true;
      }
    }
    return false;
  }

  //endregion

  //region LinkSpeeds

  private static PropertyValue readLinkSpeeds(int index)
  {
    return readArray(linkSpeeds, index, LINK_SPEEDS_ARRAY);
  }

  //endregion

  //region LinkSpeedAutoNegotiate

  private static PropertyValue readLinkSpeedAutoNegotiate()
  {
    return makeBooleanReadResult(linkSpeedAutonegotiate, false);
  }

  //endregion

  //region NetworkInterfaceName

  private PropertyValue readNetworkInterfaceName()
  {
    return makeCharStringReadResult(networkInterfaceName, getMstpLink().getPortName());
  }

  //endregion

  //region Utility

  private BBacnetMstpLinkLayer getMstpLink()
  {
    return networkPort != null ? (BBacnetMstpLinkLayer)networkPort.getLink() : null;
  }

  //endregion

  //region Fields

  private static final String MAC_ADDRESS = "MacAddress";
  private static final String MAX_MASTER = "MaxMaster";
  private static final String MAX_INFO_FRAMES = "MaxInfoFrames";
  private static final String LINK_SPEED = "LinkSpeed";

  private static final int[] OPTIONAL_PROPS = {
    BBacnetPropertyIdentifier.DESCRIPTION,
    BBacnetPropertyIdentifier.COMMAND,
    BBacnetPropertyIdentifier.NETWORK_NUMBER,
    BBacnetPropertyIdentifier.NETWORK_NUMBER_QUALITY,
    BBacnetPropertyIdentifier.APDU_LENGTH,
    BBacnetPropertyIdentifier.ROUTING_TABLE,
    BBacnetPropertyIdentifier.MAC_ADDRESS,
    BBacnetPropertyIdentifier.MAX_MASTER,
    BBacnetPropertyIdentifier.MAX_INFO_FRAMES,
    BBacnetPropertyIdentifier.LINK_SPEED,
    BBacnetPropertyIdentifier.LINK_SPEEDS,
    BBacnetPropertyIdentifier.LINK_SPEED_AUTONEGOTIATE,
    BBacnetPropertyIdentifier.NETWORK_INTERFACE_NAME
  };

  private static final int[] VALID_LINK_SPEEDS = BBacnetMstpBaudRate.DEFAULT.getRange().getOrdinals();
  private static final BBacnetArray LINK_SPEEDS_ARRAY;
  static
  {
    LINK_SPEEDS_ARRAY = new BBacnetArray(BFloat.TYPE);
    for (int validLinkSpeed : VALID_LINK_SPEEDS)
    {
      LINK_SPEEDS_ARRAY.addElement(BFloat.make(validLinkSpeed));
    }
  }

  //endregion
}
