/*
 * Copyright 2023 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.export;

import static javax.baja.bacnet.datatypes.BBacnetHostAddressChoice.none;
import static javax.baja.bacnet.enums.BBacnetErrorClass.property;
import static javax.baja.bacnet.enums.BBacnetErrorCode.invalidArrayIndex;
import static javax.baja.bacnet.enums.BBacnetErrorCode.invalidDataEncoding;
import static javax.baja.bacnet.enums.BBacnetErrorCode.invalidValueInThisState;
import static javax.baja.bacnet.enums.BBacnetErrorCode.optionalFunctionalityNotSupported;
import static javax.baja.bacnet.enums.BBacnetErrorCode.other;
import static javax.baja.bacnet.enums.BBacnetErrorCode.unknownProperty;
import static javax.baja.bacnet.enums.BBacnetErrorCode.valueNotInitialized;
import static javax.baja.bacnet.enums.BBacnetErrorCode.valueOutOfRange;
import static javax.baja.bacnet.enums.BBacnetErrorCode.writeAccessDenied;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.bacnetIpMode;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.bacnetIpUdpPort;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.bbmdAcceptFdRegistrations;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.bbmdBroadcastDistributionTable;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.bbmdForeignDeviceTable;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.fdBbmdAddress;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.fdSubscriptionLifetime;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.ipAddress;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.ipDefaultGateway;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.ipDhcpEnable;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.ipDhcpLeaseTime;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.ipDhcpLeaseTimeRemaining;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.ipDhcpServer;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.ipDnsServer;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.ipSubnetMask;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.macAddress;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.networkInterfaceName;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeBooleanReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeCharStringReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeConstructedReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeEnumReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeListOfReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeOctetStringReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeReadError;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeUnsignedReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.readArray;

import static com.tridium.bacnet.stack.link.ip.BBroadcastDistributionTable.noValidation;
import static com.tridium.bacnet.stack.link.ip.util.BacnetIpLinkUtil.EMPTY_BACNET_IP_OCTET_STRING;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.StringJoiner;
import java.util.logging.Level;

import javax.baja.bacnet.BacnetException;
import javax.baja.bacnet.datatypes.BBacnetArray;
import javax.baja.bacnet.datatypes.BBacnetBdtEntry;
import javax.baja.bacnet.datatypes.BBacnetFdtEntry;
import javax.baja.bacnet.datatypes.BBacnetHostAddress;
import javax.baja.bacnet.datatypes.BBacnetHostAddressChoice;
import javax.baja.bacnet.datatypes.BBacnetHostNPort;
import javax.baja.bacnet.datatypes.BBacnetListOf;
import javax.baja.bacnet.datatypes.BBacnetOctetString;
import javax.baja.bacnet.datatypes.BBacnetUnsigned;
import javax.baja.bacnet.datatypes.BIBacnetDataType;
import javax.baja.bacnet.enums.BBacnetIpMode;
import javax.baja.bacnet.enums.BBacnetNetworkPortCommand;
import javax.baja.bacnet.enums.BBacnetNetworkType;
import javax.baja.bacnet.enums.BBacnetPropertyIdentifier;
import javax.baja.bacnet.enums.BBacnetReliability;
import javax.baja.bacnet.io.AsnException;
import javax.baja.bacnet.io.ErrorException;
import javax.baja.bacnet.io.ErrorType;
import javax.baja.bacnet.io.PropertyValue;
import javax.baja.naming.SlotPath;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.security.PermissionException;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BBoolean;
import javax.baja.sys.BRelTime;
import javax.baja.sys.BString;
import javax.baja.sys.BValue;
import javax.baja.sys.BVector;
import javax.baja.sys.Context;
import javax.baja.sys.LocalizableRuntimeException;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.user.BUser;

import com.tridium.bacnet.asn.AsnUtil;
import com.tridium.bacnet.asn.NErrorType;
import com.tridium.bacnet.asn.NReadPropertyResult;
import com.tridium.bacnet.enums.BIpDeviceType;
import com.tridium.bacnet.stack.link.ip.BBacnetIpLinkLayer;
import com.tridium.bacnet.stack.link.ip.BBacnetIpServerPort;
import com.tridium.bacnet.stack.link.ip.BBdtEntry;
import com.tridium.bacnet.stack.link.ip.BBroadcastDistributionTable;
import com.tridium.bacnet.stack.link.ip.BFdtEntry;
import com.tridium.bacnet.stack.link.ip.BForeignDeviceRegistration;
import com.tridium.bacnet.stack.link.ip.util.BacnetIpLinkUtil;
import com.tridium.nre.platform.OperatingSystemEnum;
import com.tridium.nre.platform.PlatformUtil;
import com.tridium.platform.tcpip.BTcpIpAdapterSettings;
import com.tridium.platform.tcpip.BTcpIpHostSettings;
import com.tridium.platform.tcpip.BTcpIpPlatformService;

/**
 * BBacnetIpPortDescriptor exports a BNetworkPort that contains a BBacnetIpLinkLayer as a BACnet
 * Network Port object with the Network Type set to IPv4.
 *
 * @author Uday Rapuru on 19-Sep-2023
 * @since Niagara 4.15
 */
@NiagaraType
public class BBacnetIpPortDescriptor
  extends BBacnetNetworkPortDescriptor
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.export.BBacnetIpPortDescriptor(2979906276)1.0$ @*/
/* Generated Tue Jan 30 09:01:31 CST 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetIpPortDescriptor.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  public BBacnetIpPortDescriptor()
  {
    // Determine if we were instantiated on a Tridium QNX device that might require address
    // translation
    IS_TRIDIUM_QNX = PlatformUtil.isTridiumPlatform() && OperatingSystemEnum.isOS(OperatingSystemEnum.qnx);
  }

  //region BBacnetNetworkPortDescriptor

  @Override
  protected int getNetworkType()
  {
    return BBacnetNetworkType.IPV_4;
  }

  @Override
  protected int[] getOptionalProps()
  {
    //noinspection AssignmentOrReturnOfFieldWithMutableType
    return OPTIONAL_PROPS;
  }

  @Override
  protected boolean isArrayProp(int propId)
  {
    switch (propId)
    {
      case BBacnetPropertyIdentifier.IP_DNS_SERVER:
      case BBacnetPropertyIdentifier.PROPERTY_LIST:
        return true;
      default:
        return false;
    }
  }

  @Override
  protected boolean isListProp(int propId)
  {
    switch (propId)
    {
      case BBacnetPropertyIdentifier.ROUTING_TABLE:
      case BBacnetPropertyIdentifier.BBMD_BROADCAST_DISTRIBUTION_TABLE:
      case BBacnetPropertyIdentifier.BBMD_FOREIGN_DEVICE_TABLE:
        return true;
      default:
        return false;
    }
  }

  //endregion

  //region ReadProperty

  @Override
  protected PropertyValue readOptionalProperty(int propertyId, int index)
  {
    switch (propertyId)
    {
      case BBacnetPropertyIdentifier.NETWORK_NUMBER:
        return readNetworkNumber();

      case BBacnetPropertyIdentifier.NETWORK_NUMBER_QUALITY:
        return readNetworkNumberQuality();

      case BBacnetPropertyIdentifier.APDU_LENGTH:
        return readApduLength();

      case BBacnetPropertyIdentifier.MAC_ADDRESS:
        return readMacAddress();

      case BBacnetPropertyIdentifier.BACNET_IP_MODE:
        return readIpMode();

      case BBacnetPropertyIdentifier.BACNET_IP_UDP_PORT:
        return readUdpPort();

      case BBacnetPropertyIdentifier.ROUTING_TABLE:
        return readRoutingTable();

      case BBacnetPropertyIdentifier.BBMD_BROADCAST_DISTRIBUTION_TABLE:
        return readBroadcastDistributionTable();

      case BBacnetPropertyIdentifier.BBMD_ACCEPT_FD_REGISTRATIONS:
        return readBbmdAcceptForeignDeviceRegistrations();

      case BBacnetPropertyIdentifier.BBMD_FOREIGN_DEVICE_TABLE:
        return readForeignDeviceTable();

      case BBacnetPropertyIdentifier.FD_BBMD_ADDRESS:
        return readForeignDeviceBbmdAddress();

      case BBacnetPropertyIdentifier.FD_SUBSCRIPTION_LIFETIME:
        return readForeignDeviceSubscriptionLifetime();

      case BBacnetPropertyIdentifier.IP_ADDRESS:
        return readIpAddress();

      case BBacnetPropertyIdentifier.IP_SUBNET_MASK:
        return readSubnetMask();

      case BBacnetPropertyIdentifier.IP_DEFAULT_GATEWAY:
        return readDefaultGateway();

      case BBacnetPropertyIdentifier.IP_DNS_SERVER:
        return readDnsServer(index);

      case BBacnetPropertyIdentifier.IP_DHCP_ENABLE:
        return readDhcpEnable();

      case BBacnetPropertyIdentifier.IP_DHCP_LEASE_TIME:
        return readDhcpLeaseTime();

      case BBacnetPropertyIdentifier.IP_DHCP_LEASE_TIME_REMAINING:
        return readDhcpLeaseTimeRemaining();

      case BBacnetPropertyIdentifier.IP_DHCP_SERVER:
        return readDhcpServer();

      case BBacnetPropertyIdentifier.NETWORK_INTERFACE_NAME:
        return readNetworkInterfaceName();

      default:
        return new NReadPropertyResult(propertyId, new NErrorType(property, unknownProperty));
    }
  }

  //endregion

  //region ReadRange

  @Override
  protected List<? extends BIBacnetDataType> getListElements(int propertyId)
    throws ErrorException
  {
    switch (propertyId)
    {
      case BBacnetPropertyIdentifier.ROUTING_TABLE:
        return getRouterEntries();
      case BBacnetPropertyIdentifier.BBMD_BROADCAST_DISTRIBUTION_TABLE:
        return getBdtEntries();
      case BBacnetPropertyIdentifier.BBMD_FOREIGN_DEVICE_TABLE:
        return getFdtEntries();
    }

    throw new IllegalArgumentException("Invalid propertyId value " + propertyId);
  }

  //endregion

  //region WriteProperty

  @Override
  protected ErrorType writeOptionalProperty(int propertyId, int index, byte[] value, int priority)
    throws BacnetException
  {
    switch (propertyId)
    {
      case BBacnetPropertyIdentifier.NETWORK_NUMBER_QUALITY:
      case BBacnetPropertyIdentifier.APDU_LENGTH:
      // MAC_ADDRESS is read-only because the Network_Type is IPV4 and the
      // Protocol_Level is BACNET_APPLICATION
      case BBacnetPropertyIdentifier.MAC_ADDRESS:
      case BBacnetPropertyIdentifier.ROUTING_TABLE:
      case BBacnetPropertyIdentifier.BBMD_FOREIGN_DEVICE_TABLE:
      case BBacnetPropertyIdentifier.IP_DHCP_LEASE_TIME:
      case BBacnetPropertyIdentifier.IP_DHCP_LEASE_TIME_REMAINING:
      case BBacnetPropertyIdentifier.IP_DHCP_SERVER:
      case BBacnetPropertyIdentifier.NETWORK_INTERFACE_NAME:
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(this + ": attempted to write read-only property " + BBacnetPropertyIdentifier.tag(propertyId));
        }
        return new NErrorType(property, writeAccessDenied);

      case BBacnetPropertyIdentifier.NETWORK_NUMBER:
        return writeNetworkNumber(value);

      case BBacnetPropertyIdentifier.BACNET_IP_MODE:
        return writeIpMode(value);

      case BBacnetPropertyIdentifier.BACNET_IP_UDP_PORT:
        return writeUdpPort(value);

      case BBacnetPropertyIdentifier.BBMD_BROADCAST_DISTRIBUTION_TABLE:
        return writeBroadcastDistributionTable(value);

      case BBacnetPropertyIdentifier.FD_BBMD_ADDRESS:
        return writeForeignDeviceBbmdAddress(value);

      case BBacnetPropertyIdentifier.FD_SUBSCRIPTION_LIFETIME:
        return writeForeignDeviceSubscriptionLifetime(value);

      case BBacnetPropertyIdentifier.BBMD_ACCEPT_FD_REGISTRATIONS:
        return writeBbmdAcceptForeignDeviceRegistrations(value);

      case BBacnetPropertyIdentifier.IP_ADDRESS:
        return writeIpAddress(value);

      case BBacnetPropertyIdentifier.IP_SUBNET_MASK:
        return writeSubnetMask(value);

      case BBacnetPropertyIdentifier.IP_DEFAULT_GATEWAY:
        return writeDefaultGateway(value);

      case BBacnetPropertyIdentifier.IP_DNS_SERVER:
        return writeDnsServer(index, value);

      case BBacnetPropertyIdentifier.IP_DHCP_ENABLE:
        return writeDhcpEnable(value);

      default:
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(this + ": attempted to write unknown property " + BBacnetPropertyIdentifier.tag(propertyId));
        }
        return new NErrorType(property, unknownProperty);
    }
  }

  //endregion

  //region Validate/Activate Changes

  @Override
  public boolean activateChangesRequiresReboot()
  {
    return hasAdapterChanges();
  }

  private boolean hasAdapterChanges()
  {
    BBacnetNetworkPortPendingChanges pendingChanges = getPendingChanges();
    return pendingChanges.getProperty(IP_ADDRESS) != null ||
           pendingChanges.getProperty(SUBNET_MASK) != null ||
           pendingChanges.getProperty(DHCP_ENABLE) != null ||
           pendingChanges.getProperty(DEFAULT_GATEWAY) != null ||
           pendingChanges.getProperty(DNS_SERVER) != null;
  }

  @Override
  protected void validateSetPendingChange(String name, BValue value, Context context)
  {
    switch (name)
    {
      case NETWORK_NUMBER:
        validateSetNetworkNumber(value, context);
        break;
      case IP_MODE:
        validateSetIpMode(value, context);
        break;
      case UDP_PORT:
        validateSetUdpPort(value, context);
        break;
      case BBMD_BROADCAST_DISTRIBUTION_TABLE:
        validateSetBbmdBroadcastDistributionTable(value, context);
        break;
      case BBMD_ACCEPT_FD_REGISTRATIONS:
        validateSetBbmdAcceptFdRegistrations(value, context);
        break;
      case FD_BBMD_ADDRESS:
        validateSetFdBbmdAddress(value, context);
        break;
      case FD_SUBSCRIPTION_LIFETIME:
        validateSetFdSubscriptionLifetime(value, context);
        break;
      case IP_ADDRESS:
        validateSetIpAddress(value, context);
        break;
      case SUBNET_MASK:
        validateSetSubnetMask(value, context);
        break;
      case DEFAULT_GATEWAY:
        validateSetDefaultGateway(value, context);
        break;
      case DNS_SERVER:
        validateSetDnsServer(value, context);
        break;
      case DHCP_ENABLE:
        validateSetDhcpEnable(value, context);
        break;
      default:
        throw new LocalizableRuntimeException(
          "bacnet",
          "networkPortDescriptor.pendingChange.unknown",
          new Object[] { name });
    }
  }

  private static void checkCanSaveAdapterSettings(BTcpIpAdapterSettings adapterSettings, Context context)
  {
    if (adapterSettings == null)
    {
      throw new LocalizableRuntimeException("bacnet", "ipPortDescriptor.pendingChange.missingAdapterSettings");
    }

    BUser user = context != null ? context.getUser() : null;
    if (user != null)
    {
      user.checkInvoke(getTcpIpService(), BTcpIpPlatformService.savePlatformServiceProperties);
    }
  }

  private static void checkCanWriteHostSettings(BTcpIpAdapterSettings adapterSettings)
  {
    if (adapterSettings.getHostSettings().getIsReadonly())
    {
      throw new LocalizableRuntimeException("bacnet", "ipPortDescriptor.pendingChange.readonlyAdapterSettings");
    }
  }

  private static void checkCanWriteAdapterSettings(BTcpIpAdapterSettings adapterSettings)
  {
    if (adapterSettings.getIsAdapterReadonly())
    {
      throw new LocalizableRuntimeException("bacnet", "ipPortDescriptor.pendingChange.readonlyAdapterSettings");
    }
  }

  private void validateCanSaveAdapterSettings(BTcpIpAdapterSettings adapterSettings, Context context)
    throws ValidateChangesException
  {
    if (adapterSettings == null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(this + ": cannot find TCP/IP adapter settings");
      }
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ other,
        /* property */ null,
        /* details */ "Cannot find IP adapter settings");
    }

    BUser user = getUser(context, writeAccessDenied);
    try
    {
      user.checkInvoke(getTcpIpService(), BTcpIpPlatformService.savePlatformServiceProperties);
    }
    catch (PermissionException e)
    {
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ writeAccessDenied,
        /* property */ null,
        /* details */ "User missing invoke permissions on TCP/IP service");
    }
  }

  private void validateCanWriteHostSettings(BTcpIpAdapterSettings adapterSettings, BBacnetPropertyIdentifier propertyId)
    throws ValidateChangesException
  {
    if (adapterSettings.getHostSettings().getIsReadonly())
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(this + ": TCP/IP host settings are read only");
      }
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ writeAccessDenied,
        /* property */ propertyId,
        /* details */ "TCP/IP host settings are read only");
    }
  }

  private void validateCanWriteAdapterSettings(BTcpIpAdapterSettings adapterSettings, BBacnetPropertyIdentifier propertyId)
    throws ValidateChangesException
  {
    if (adapterSettings.getIsAdapterReadonly())
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(this + ": TCP/IP adapter settings are read only");
      }
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ writeAccessDenied,
        /* property */ propertyId,
        /* details */ "TCP/IP adapter settings are read only");
    }
  }

  @Override
  public void validateChanges(Context context)
    throws ValidateChangesException
  {
    if (!hasPendingChanges())
    {
      return;
    }

    checkCanReadPendingChanges(context);

    validateNetworkNumber(context);
    validateIpMode(context);
    validateUdpPort(context);
    validateBroadcastDistributionTable(context);
    validateBbmdAcceptFdRegistrations(context);
    validateFdBbmdAddress(context);
    validateFdSubscriptionLifetime(context);

    if (hasAdapterChanges())
    {
      BTcpIpAdapterSettings adapterSettings = findAdapterSettings();
      validateCanSaveAdapterSettings(adapterSettings, context);
      validateIpAddress(adapterSettings, context);
      validateSubnetMask(adapterSettings, context);
      validateDefaultGateway(adapterSettings, context);
      validateDnsServer(adapterSettings, context);
      validateDhcpEnable(adapterSettings, context);
    }
  }

  @Override
  public void activateChanges(Context context)
    throws Exception
  {
    if (!hasPendingChanges())
    {
      return;
    }

    BTcpIpAdapterSettings adapterSettings = null;
    if (hasAdapterChanges())
    {
      adapterSettings = findAdapterSettings();
      if (adapterSettings == null)
      {
        throw new BacnetException(this + ": has pending changes to the TCP/IP adapter but adapter settings could not be found");
      }
    }

    activateNetworkNumber(context);
    activateIpMode(context);
    activateUdpPort(context);
    activateBroadcastDistributionTable();
    activateBbmdAcceptFdRegistrations(context);
    activateFdBbmdAddress(context);
    activateFdSubscriptionLifetime(context);

    if (hasAdapterChanges())
    {
      activateIpAddress(adapterSettings, context);
      activateSubnetMask(adapterSettings, context);
      activateDefaultGateway(adapterSettings, context);
      activateDnsServer(adapterSettings, context);
      activateDhcpEnable(adapterSettings, context);

      // If getTcpIpService had returned null, we would have thrown an exception when
      // findAdapterSettings returned null above.
      getTcpIpService().invoke(BTcpIpPlatformService.savePlatformServiceProperties, null, context);
    }
  }

  private static String makeAddressStr(byte[] bytes)
  {
    if (bytes == null || bytes.length != 4)
    {
      return "null";
    }

    return "" +
      (bytes[0] & 0xFF) + '.' +
      (bytes[1] & 0xFF) + '.' +
      (bytes[2] & 0xFF) + '.' +
      (bytes[3] & 0xFF);
  }

  //endregion

  //region Command

  @Override
  protected ErrorType handleRenewFdRegistrationCommand(Context context)
  {
    BBacnetIpLinkLayer ipLinkLayer = getIpLinkLayer();
    BIpDeviceType ipDeviceType = ipLinkLayer.getIpDeviceType();
    if (!ipDeviceType.equals(BIpDeviceType.foreignDevice))
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(this + ": attempt to write command value Renew FD Registration" +
          " when IP Device Type is not Foreign Device: " + ipDeviceType);
      }
      return new NErrorType(property, valueOutOfRange);
    }

    set(command, BBacnetNetworkPortCommand.renewFdRegistration, context);
    try
    {
      BForeignDeviceRegistration[] registrations = ipLinkLayer.getChildren(BForeignDeviceRegistration.class);
      if (registrations.length > 0)
      {
        registrations[0].invoke(BForeignDeviceRegistration.registerWithBBMD, null, context);
        updateReliability();
      }
      else
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(this + ": attempt to write command value Renew FD Registration" +
            " but no foreign device registrations found");
        }
        setReliability(BBacnetReliability.renewFdRegistrationFailure);
      }
    }
    catch (Exception e)
    {
      setReliability(BBacnetReliability.renewFdRegistrationFailure);
    }
    finally
    {
      set(command, BBacnetNetworkPortCommand.idle, context);
    }
    return null;
  }

  @Override
  protected ErrorType handleRenewDhcpCommand(Context context)
  {
    if (logger.isLoggable(Level.FINE))
    {
      logger.fine(this + ": Renew DHCP command is not supported");
    }
    return new NErrorType(property, optionalFunctionalityNotSupported);
  }

  //endregion

  //region MacAddress

  private PropertyValue readMacAddress()
  {
    // As Protocol_Level is always BACNET_APPLICATION, returning the six octet combination of
    // the IP address and UDP port.
    byte[] macBytes = getMacAddress();
    if (macBytes == null)
    {
      return makeReadError(macAddress, property, valueNotInitialized);
    }

    return makeOctetStringReadResult(macAddress, macBytes);
  }

  private byte[] getMacAddress()
  {
    BBacnetOctetString ipAddress = getPendingChange(IP_ADDRESS, BBacnetOctetString.class);
    BBacnetUnsigned udpPort = getPendingChange(UDP_PORT, BBacnetUnsigned.class);
    byte[] macBytes;
    if (ipAddress != null && udpPort != null)
    {
      macBytes = new byte[6];
    }
    else
    {
      macBytes = getIpLinkLayer().getMacAddress();
      if (macBytes == null || macBytes.length != 6)
      {
        return null;
      }

      if (ipAddress != null || udpPort != null)
      {
        // Don't change the contents of the linkLayer's myMac field
        macBytes = macBytes.clone();
      }
    }

    if (ipAddress != null)
    {
      System.arraycopy(ipAddress.getAddr(), 0, macBytes, 0, 4);
    }

    if (udpPort != null)
    {
      int port = udpPort.getInt();
      macBytes[4] = (byte)((port & 0xFF00) >> 8);
      macBytes[5] = (byte)(port & 0x00FF);
    }
    return macBytes;
  }

  //endregion

  //region IpMode

  private PropertyValue readIpMode()
  {
    return makeEnumReadResult(bacnetIpMode, getIpMode());
  }

  private BBacnetIpMode getIpMode()
  {
    BBacnetIpMode pending = getPendingChange(IP_MODE, BBacnetIpMode.class);
    if (pending != null)
    {
      return pending;
    }

    switch (getIpLinkLayer().getIpDeviceType().getOrdinal())
    {
      case BIpDeviceType.FOREIGN_DEVICE: return BBacnetIpMode.foreign;
      case BIpDeviceType.BBMD:           return BBacnetIpMode.bbmd;
      default:                           return BBacnetIpMode.normal;
    }
  }

  private ErrorType writeIpMode(byte[] value)
    throws AsnException
  {
    addPendingChange(
      IP_MODE,
      BBacnetIpMode.make(AsnUtil.fromAsnEnumerated(value)),
      BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  private void validateSetIpMode(BValue value, Context context)
  {
    checkPendingChangeType(IP_MODE, value, BBacnetIpMode.TYPE);
    checkCanWrite(getIpLinkLayer(), BBacnetIpLinkLayer.ipDeviceType, context);
  }

  private void validateIpMode(Context context)
    throws ValidateChangesException
  {
    BBacnetIpMode change = getPendingChange(IP_MODE, BBacnetIpMode.class);
    if (change != null)
    {
      validateCanWrite(getIpLinkLayer(), BBacnetIpLinkLayer.ipDeviceType, bacnetIpMode, context);
    }
  }

  private void activateIpMode(Context context)
  {
    BBacnetIpMode change = getPendingChange(IP_MODE, BBacnetIpMode.class);
    if (change != null)
    {
      getIpLinkLayer().set(BBacnetIpLinkLayer.ipDeviceType, makeDeviceType(change), context);
    }
  }

  //endregion

  //region UdpPort

  private PropertyValue readUdpPort()
  {
    BBacnetUnsigned pending = getPendingChange(UDP_PORT, BBacnetUnsigned.class);
    if (pending != null)
    {
      if (pending.getInt() < 0 || pending.getInt() > BBacnetUnsigned.MAX_UNSIGNED16_VALUE)
      {
        return makeReadError(bacnetIpUdpPort, property, valueOutOfRange);
      }
      return makeUnsignedReadResult(bacnetIpUdpPort, pending);
    }

    byte[] mac = getIpLinkLayer().getMacAddress();
    if (mac == null || mac.length != 6)
    {
      return makeReadError(bacnetIpUdpPort, property, valueNotInitialized);
    }
    return makeUnsignedReadResult(bacnetIpUdpPort, BacnetIpLinkUtil.getPortFromBacnetIpBytes(mac));
  }

  private ErrorType writeUdpPort(byte[] value)
    throws AsnException
  {
    BBacnetUnsigned udpPort = AsnUtil.fromAsnUnsigned(value);
    if (udpPort.getInt() > 65535)
    {
      return new NErrorType(property, valueOutOfRange);
    }

    addPendingChange(UDP_PORT, udpPort, BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  private void validateSetUdpPort(BValue value, Context context)
  {
    checkPendingChangeType(UDP_PORT, value, BBacnetUnsigned.TYPE);
    checkCanWrite(getIpLinkLayer().getUdpServerPort(), BBacnetIpServerPort.publicServerPort, context);
  }

  private void validateUdpPort(Context context)
    throws ValidateChangesException
  {
    BBacnetUnsigned change = getPendingChange(UDP_PORT, BBacnetUnsigned.class);
    if (change != null)
    {
      int value = change.getInt();
      if (value < 0 || value > 65535)
      {
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ valueOutOfRange,
          /* property */ bacnetIpUdpPort,
          /* details */ "Value " + change.getInt() + " is not a valid Unsigned16");
      }

      validateCanWrite(
        getIpLinkLayer().getUdpServerPort(),
        BBacnetIpServerPort.publicServerPort,
        bacnetIpUdpPort,
        context);
    }
  }

  private void activateUdpPort(Context context)
  {
    BBacnetUnsigned change = getPendingChange(UDP_PORT, BBacnetUnsigned.class);
    if (change != null)
    {
      getIpLinkLayer().getUdpServerPort().setInt(BBacnetIpServerPort.publicServerPort, change.getInt(), context);
    }
  }

  //endregion

  //region BroadcastDistributionTable

  private PropertyValue readBroadcastDistributionTable()
  {
    try
    {
      BBacnetListOf table = getPendingChange(BBMD_BROADCAST_DISTRIBUTION_TABLE, BBacnetListOf.class);
      if (table == null)
      {
        table = new BBacnetListOf(BBacnetBdtEntry.TYPE);
        for (BBacnetBdtEntry bdtEntry : getBdtEntries())
        {
          table.addListElement(bdtEntry, null);
        }
      }
      return makeListOfReadResult(bbmdBroadcastDistributionTable, table);
    }
    catch (ErrorException e)
    {
      return makeReadError(bbmdBroadcastDistributionTable, e.getErrorType());
    }
  }

  private List<BBacnetBdtEntry> getBdtEntries()
    throws ErrorException
  {
    List<BBacnetBdtEntry> bdtEntries = new ArrayList<>();
    BBacnetIpLinkLayer linkLayer = getIpLinkLayer();
    if (!linkLayer.getIpDeviceType().equals(BIpDeviceType.bbmd))
    {
      return bdtEntries;
    }

    BBroadcastDistributionTable table = linkLayer.getBroadcastDistributionTable();
    BBdtEntry[] entries = table.getChildren(BBdtEntry.class);
    for (BBdtEntry entry : entries)
    {
      try
      {
        BBacnetHostNPort bbmdAddress = BBacnetHostNPort.parseBacnetIpAddress(entry.getBacnetIPAddress());
        if (bbmdAddress.getHostAddress().getChoice().equals(none))
        {
          throw new ErrorException(new NErrorType(property, valueNotInitialized));
        }

        byte[] mask = entry.getBdMask();
        BacnetIpLinkUtil.validateAddressLength(mask, 4);

        BBacnetBdtEntry bacnetEntry = new BBacnetBdtEntry();
        bacnetEntry.setBbmdAddress(bbmdAddress);
        bacnetEntry.setBroadcastMask(BBacnetOctetString.make(mask));
        bdtEntries.add(bacnetEntry);
      }
      catch (IllegalArgumentException ex)
      {
        throw new ErrorException(new NErrorType(property, valueNotInitialized));
      }
    }
    return bdtEntries;
  }

  private ErrorType writeBroadcastDistributionTable(byte[] value)
    throws AsnException
  {
    BBacnetListOf bdtEntries = (BBacnetListOf) AsnUtil.fromAsn(ASN_BACNET_LIST, value, new BBacnetListOf(BBacnetBdtEntry.TYPE));
    try
    {
      validateBacnetBdtEntries(bdtEntries.getChildren(BBacnetBdtEntry.class));
    }
    catch (ValidateChangesException e)
    {
      return e.getError();
    }

    addPendingChange(BBMD_BROADCAST_DISTRIBUTION_TABLE, bdtEntries, BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  public void validateSetBbmdBroadcastDistributionTable(BValue value, Context context)
  {
    checkPendingChangeType(BBMD_BROADCAST_DISTRIBUTION_TABLE, value, BBacnetListOf.TYPE);
    checkCanWrite(getIpLinkLayer().getBroadcastDistributionTable(), context);
  }

  private void validateBroadcastDistributionTable(Context context)
    throws ValidateChangesException
  {
    BBacnetListOf change = getPendingChange(BBMD_BROADCAST_DISTRIBUTION_TABLE, BBacnetListOf.class);
    if (change != null)
    {
      BBacnetBdtEntry[] bacnetBdtEntries = change.getChildren(BBacnetBdtEntry.class);
      if (bacnetBdtEntries.length == 0)
      {
        return;
      }

      validateBacnetBdtEntries(bacnetBdtEntries);

      if (!getIpMode().equals(BBacnetIpMode.bbmd))
      {
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ invalidValueInThisState,
          /* property */ bbmdBroadcastDistributionTable,
          /* details */ "Cannot activate a non-empty BDT when the IP mode will not be BBMD");
      }

      byte[] localMac = getMacAddress();
      if (localMac == null)
      {
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ valueNotInitialized,
          /* property */ bbmdBroadcastDistributionTable,
          /* details */ "Port's MAC address has not been initialized");
      }

      int localMatches = 0;
      for (BBacnetBdtEntry bacnetBdtEntry : bacnetBdtEntries)
      {
        byte[] bbmdAddress = bacnetBdtEntry.getBbmdAddress().getBacnetIpAddressBytes();
        if (Arrays.equals(localMac, bbmdAddress))
        {
          ++localMatches;
        }
      }

      if (localMatches != 1)
      {
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ invalidValueInThisState,
          /* property */ bbmdBroadcastDistributionTable,
          /* details */ "One and only one entry must match the port's MAC address");
      }

      validateCanWrite(
        getIpLinkLayer().getBroadcastDistributionTable(),
        bbmdBroadcastDistributionTable,
        context);
    }
  }

  private void validateBacnetBdtEntries(BBacnetBdtEntry[] bdtEntries)
    throws ValidateChangesException
  {
    for (BBacnetBdtEntry bacnetEntry : bdtEntries)
    {
      BBacnetHostNPort hostAddress = bacnetEntry.getBbmdAddress();
      BBacnetHostAddressChoice addressChoice = hostAddress.getHostAddress().getChoice();
      if (addressChoice.equals(none))
      {
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ valueOutOfRange,
          /* property */ bbmdBroadcastDistributionTable,
          /* details */ "BACnetHostAddress choice none is not valid");
      }

      if (addressChoice.equals(BBacnetHostAddressChoice.name))
      {
        // No need to validate host name or try to resolve it here.
        return;
      }

      // If remaining choice is not "ipAddress", throw exception
      if (!addressChoice.equals(BBacnetHostAddressChoice.ipAddress))
      {
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ optionalFunctionalityNotSupported,
          /* property */ bbmdBroadcastDistributionTable,
          /* details */ "Unsupported BACnetHostAddress choice: " + hostAddress.getHostAddress().getChoice());
      }

      try
      {
        // Validate the BacnetIpAddressBytes when the choice is "ipAddress"
        BacnetIpLinkUtil.validateAddressLength(hostAddress.getBacnetIpAddressBytes(), 6);
        BacnetIpLinkUtil.validateAddressLength(bacnetEntry.getBroadcastMask().getBytes(), 4);
      }
      catch (IllegalArgumentException e)
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.log(Level.FINE, this + ": Failed to validate address/mask bytes of BDT entry", e);
        }
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ valueOutOfRange,
          /* property */ bbmdBroadcastDistributionTable,
          /* details */ e.getMessage());
      }
    }
  }

  private void activateBroadcastDistributionTable()
  {
    BBacnetListOf change = getPendingChange(BBMD_BROADCAST_DISTRIBUTION_TABLE, BBacnetListOf.class);
    if (change != null)
    {
      byte[] localMac = getMacAddress();
      BBroadcastDistributionTable table = getIpLinkLayer().getBroadcastDistributionTable();
      table.removeAll(noValidation);
      for (BBacnetBdtEntry bacnetBdtEntry : change.getChildren(BBacnetBdtEntry.class))
      {
        byte[] macAddress = bacnetBdtEntry.getBbmdAddress().getBacnetIpAddressBytes();
        BBdtEntry bdtEntry = new BBdtEntry(macAddress, bacnetBdtEntry.getBroadcastMask().getBytes());
        if (Arrays.equals(localMac, macAddress))
        {
          table.add("localDevice", bdtEntry, noValidation);
        }
        else
        {
          table.add(null, bdtEntry, noValidation);
        }
      }
    }
  }

  //endregion

  //region BbmdAcceptFdRegistrations

  private PropertyValue readBbmdAcceptForeignDeviceRegistrations()
  {
    BBoolean pending = getPendingChange(BBMD_ACCEPT_FD_REGISTRATIONS, BBoolean.class);
    return makeBooleanReadResult(
      bbmdAcceptFdRegistrations,
      pending != null ? pending.getBoolean() : getIpLinkLayer().getAcceptsForeignDeviceRegistrations());
  }

  private ErrorType writeBbmdAcceptForeignDeviceRegistrations(byte[] value)
    throws AsnException
  {
    addPendingChange(
      BBMD_ACCEPT_FD_REGISTRATIONS,
      BBoolean.make(AsnUtil.fromAsnBoolean(value)),
      BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  public void validateSetBbmdAcceptFdRegistrations(BValue value, Context context)
  {
    checkPendingChangeType(BBMD_ACCEPT_FD_REGISTRATIONS, value, BBoolean.TYPE);
    checkCanWrite(getIpLinkLayer(), BBacnetIpLinkLayer.acceptsForeignDeviceRegistrations, context);
  }

  private void validateBbmdAcceptFdRegistrations(Context context)
    throws ValidateChangesException
  {
    BBoolean change = getPendingChange(BBMD_ACCEPT_FD_REGISTRATIONS, BBoolean.class);
    if (change != null)
    {
      validateCanWrite(
        getIpLinkLayer(),
        BBacnetIpLinkLayer.acceptsForeignDeviceRegistrations,
        bbmdAcceptFdRegistrations,
        context);
    }
  }

  private void activateBbmdAcceptFdRegistrations(Context context)
  {
    BBoolean change = getPendingChange(BBMD_ACCEPT_FD_REGISTRATIONS, BBoolean.class);
    if (change != null)
    {
      getIpLinkLayer().set(BBacnetIpLinkLayer.acceptsForeignDeviceRegistrations, change, context);
    }
  }

  //endregion

  //region ForeignDeviceTable

  private PropertyValue readForeignDeviceTable()
  {
    BBacnetListOf fdtEntries = new BBacnetListOf(BBacnetFdtEntry.TYPE);
    for (BBacnetFdtEntry fdtEntry : getFdtEntries())
    {
      fdtEntries.addListElement(fdtEntry, null);
    }
    return makeListOfReadResult(bbmdForeignDeviceTable, fdtEntries);
  }

  private List<BBacnetFdtEntry> getFdtEntries()
  {
    BAbsTime now = BAbsTime.now();
    List<BBacnetFdtEntry> fdtEntries = new ArrayList<>();
    for (BFdtEntry entry : getIpLinkLayer().getForeignDeviceTable().getChildren(BFdtEntry.class))
    {
      BBacnetFdtEntry fdtEntry = new BBacnetFdtEntry();
      fdtEntry.setBacnetIpAddress(getBacnetIpOctetString(entry));

      int timeToLive = coerceToUnsignedShort(entry.getTimeToLive());
      fdtEntry.setTimeToLive(BBacnetUnsigned.make(timeToLive));

      int remaining = now.delta(entry.getPurgeTime()).getSeconds();
      remaining = coerceToUnsignedShort(remaining);
      fdtEntry.setRemainingTimeToLive(BBacnetUnsigned.make(remaining));

      fdtEntries.add(fdtEntry);
    }
    return fdtEntries;
  }

  private BBacnetOctetString getBacnetIpOctetString(BFdtEntry entry)
  {
    try
    {
      byte[] addressBytes = entry.getBIpAddr();
      if (addressBytes != null)
      {
        return BBacnetOctetString.make(addressBytes);
      }

      if (logger.isLoggable(Level.INFO))
      {
        logger.info(this + ": could not map FdtEntry to BacnetFdtEntry with address " + entry.getBacnetIPAddress());
      }
    }
    catch (Exception e)
    {
      if (logger.isLoggable(Level.INFO))
      {
        logger.log(
          Level.INFO,
          this + ": could not map FdtEntry to BacnetFdtEntry with address " + entry.getBacnetIPAddress() + "; error: " + e,
          logger.isLoggable(Level.FINE) ? e : null);
      }
    }
    return EMPTY_BACNET_IP_OCTET_STRING;
  }

  private static int coerceToUnsignedShort(int value)
  {
    if (value < 0)
    {
      return 0;
    }
    else if (value > 65535)
    {
      return 65535;
    }
    else
    {
      return value;
    }
  }

  //endregion

  //region FdBbmdAddress

  private PropertyValue readForeignDeviceBbmdAddress()
  {
    BBacnetHostNPort address = getPendingChange(FD_BBMD_ADDRESS, BBacnetHostNPort.class);
    if (address == null)
    {
      String addressStr;
      BForeignDeviceRegistration[] registrations = getIpLinkLayer().getChildren(BForeignDeviceRegistration.class);
      if (registrations.length > 0)
      {
        addressStr = registrations[0].getBbmdAddress();
      }
      else
      {
        addressStr = getIpLinkLayer().getBbmdAddress();
      }

      byte[] addressBytes = BacnetIpLinkUtil.getBacnetIpBytes(addressStr);
      if (addressBytes == null || addressBytes.length != 6)
      {
        return makeReadError(fdBbmdAddress, property, valueNotInitialized);
      }
      address = BBacnetHostNPort.makeWithBacnetIpBytes(addressBytes);
    }
    return makeConstructedReadResult(fdBbmdAddress, address);
  }

  private ErrorType writeForeignDeviceBbmdAddress(byte[] value)
    throws AsnException
  {
    BBacnetHostNPort hostNPort = (BBacnetHostNPort)AsnUtil.fromAsn(
      ASN_CONSTRUCTED_DATA,
      value,
      new BBacnetHostNPort());
    try
    {
      validateFdBbmdAddress(hostNPort.getHostAddress());
    }
    catch (ValidateChangesException e)
    {
      return e.getError();
    }

    addPendingChange(FD_BBMD_ADDRESS, hostNPort, BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  public void validateSetFdBbmdAddress(BValue value, Context context)
  {
    checkPendingChangeType(FD_BBMD_ADDRESS, value, BBacnetHostNPort.TYPE);
    checkCanWrite(getIpLinkLayer(), BBacnetIpLinkLayer.bbmdAddress, context);
  }

  private void validateFdBbmdAddress(Context context)
    throws ValidateChangesException
  {
    BBacnetHostNPort change = getPendingChange(FD_BBMD_ADDRESS, BBacnetHostNPort.class);
    if (change != null)
    {
      validateFdBbmdAddress(change.getHostAddress());
      if (getIpMode().equals(BBacnetIpMode.foreign) &&
          !change.getHostAddress().getChoice().equals(BBacnetHostAddressChoice.ipAddress))
      {
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ valueOutOfRange,
          /* property */ fdBbmdAddress,
          /* details */ "When IP Mode is Foreign, BACnetHostAddress choice must be IP");
      }

      validateCanWrite(getIpLinkLayer(), BBacnetIpLinkLayer.bbmdAddress, fdBbmdAddress, context);
    }
  }

  private void validateFdBbmdAddress(BBacnetHostAddress hostAddress)
    throws ValidateChangesException
  {
    if (hostAddress.getChoice().equals(BBacnetHostAddressChoice.ipAddress))
    {
      try
      {
        BacnetIpLinkUtil.validateAddressLength(hostAddress.getIpAddress().getAddr(), 4);
      }
      catch (IllegalArgumentException e)
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(this + ": failed to validate foreign device BBMD address bytes: " + hostAddress.getIpAddress());
        }
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ valueOutOfRange,
          /* property */ fdBbmdAddress,
          /* details */ "Invalid BACnetHostAddress IP address");
      }
    }
    else if (!hostAddress.getChoice().equals(none))
    {
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ optionalFunctionalityNotSupported,
        /* property */ fdBbmdAddress,
        /* details */ "Unsupported BACnetHostAddress choice: " + hostAddress.getChoice());
    }
  }

  private void activateFdBbmdAddress(Context context)
  {
    BBacnetHostNPort change = getPendingChange(FD_BBMD_ADDRESS, BBacnetHostNPort.class);
    if (change != null)
    {
      String bbmdAddress = "null";
      BBacnetHostAddress hostAddress = change.getHostAddress();
      if (hostAddress.getChoice().equals(BBacnetHostAddressChoice.ipAddress))
      {
        BBacnetOctetString ipAddress = hostAddress.getIpAddress();
        bbmdAddress = makeAddressStr(ipAddress.getBytes()) + ':' + change.getPort();
      }

      BBacnetIpLinkLayer ipLinkLayer = getIpLinkLayer();
      BForeignDeviceRegistration[] registrations = ipLinkLayer.getChildren(BForeignDeviceRegistration.class);
      if (registrations.length > 0)
      {
        registrations[0].setString(BForeignDeviceRegistration.bbmdAddress, bbmdAddress, context);
      }
      ipLinkLayer.setString(BBacnetIpLinkLayer.bbmdAddress, bbmdAddress, context);
    }
  }

  //endregion

  //region FdSubscriptionLifetime

  private PropertyValue readForeignDeviceSubscriptionLifetime()
  {
    BBacnetUnsigned pending = getPendingChange(FD_SUBSCRIPTION_LIFETIME, BBacnetUnsigned.class);
    int lifetime;
    if (pending != null)
    {
      lifetime = pending.getInt();
    }
    else
    {
      BForeignDeviceRegistration[] registrations = getIpLinkLayer().getChildren(BForeignDeviceRegistration.class);
      if (registrations.length > 0)
      {
        lifetime = registrations[0].getRegistrationLifetime().getSeconds();
      }
      else
      {
        lifetime = getIpLinkLayer().getRegistrationLifetime().getSeconds();
      }

      if (lifetime < 0 || lifetime > 65535)
      {
        return makeReadError(fdSubscriptionLifetime, property, valueNotInitialized);
      }
    }

    return makeUnsignedReadResult(fdSubscriptionLifetime, lifetime);
  }

  private ErrorType writeForeignDeviceSubscriptionLifetime(byte[] value)
    throws AsnException
  {
    BBacnetUnsigned unsigned = AsnUtil.fromAsnUnsigned(value);
    if (unsigned.getInt() > BBacnetUnsigned.MAX_UNSIGNED16_VALUE)
    {
      return new NErrorType(property, valueOutOfRange);
    }

    addPendingChange(FD_SUBSCRIPTION_LIFETIME, unsigned, BLocalBacnetDevice.getBacnetContext());
    return null;
  }

  public void validateSetFdSubscriptionLifetime(BValue value, Context context)
  {
    checkPendingChangeType(FD_SUBSCRIPTION_LIFETIME, value, BBacnetUnsigned.TYPE);
    checkCanWrite(getIpLinkLayer(), BBacnetIpLinkLayer.registrationLifetime, context);
  }

  private void validateFdSubscriptionLifetime(Context context)
    throws ValidateChangesException
  {
    BBacnetUnsigned change = getPendingChange(FD_SUBSCRIPTION_LIFETIME, BBacnetUnsigned.class);
    if (change != null)
    {
      int value = change.getInt();
      if (value < 0 || value > BBacnetUnsigned.MAX_UNSIGNED16_VALUE)
      {
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ valueOutOfRange,
          /* property */ fdSubscriptionLifetime,
          /* details */ "Value " + change.getInt() + " is not a valid Unsigned16");
      }

      validateCanWrite(
        getIpLinkLayer(),
        BBacnetIpLinkLayer.registrationLifetime,
        fdSubscriptionLifetime,
        context);
    }
  }

  private void activateFdSubscriptionLifetime(Context context)
  {
    BBacnetUnsigned change = getPendingChange(FD_SUBSCRIPTION_LIFETIME, BBacnetUnsigned.class);
    if (change != null)
    {
      BRelTime lifetime = BRelTime.makeSeconds(change.getInt());
      BBacnetIpLinkLayer ipLinkLayer = getIpLinkLayer();
      BForeignDeviceRegistration[] registrations = ipLinkLayer.getChildren(BForeignDeviceRegistration.class);
      if (registrations.length > 0)
      {
        registrations[0].set(BForeignDeviceRegistration.registrationLifetime, lifetime, context);
      }
      ipLinkLayer.set(BBacnetIpLinkLayer.registrationLifetime, lifetime, context);
    }
  }

  //endregion

  //region IpAddress

  private PropertyValue readIpAddress()
  {
    BBacnetOctetString pending = getPendingChange(IP_ADDRESS, BBacnetOctetString.class);
    if (pending != null)
    {
      return makeOctetStringReadResult(ipAddress, pending);
    }

    byte[] mac = getIpLinkLayer().getMacAddress();
    if (mac == null || mac.length != 6)
    {
      return makeReadError(ipAddress, property, valueNotInitialized);
    }
    return makeOctetStringReadResult(ipAddress, Arrays.copyOf(mac, 4));
  }

  private ErrorType writeIpAddress(byte[] value)
    throws AsnException
  {
    BBacnetOctetString octetString = AsnUtil.fromAsnOctetString(value);
    try
    {
      validateAddress(octetString, ipAddress);
    }
    catch (ValidateChangesException e)
    {
      return e.getError();
    }

    try
    {
      addPendingChange(IP_ADDRESS, octetString, BLocalBacnetDevice.getBacnetContext());
      return null;
    }
    catch (Exception e)
    {
      return new NErrorType(property, writeAccessDenied);
    }
  }

  public void validateSetIpAddress(BValue value, Context context)
  {
    checkPendingChangeType(IP_ADDRESS, value, BBacnetOctetString.TYPE);
    BTcpIpAdapterSettings adapterSettings = findAdapterSettings();
    checkCanSaveAdapterSettings(adapterSettings, context);
    checkCanWrite(adapterSettings, BTcpIpAdapterSettings.ipAddress, context);
    checkCanWriteAdapterSettings(adapterSettings);
  }

  private void validateIpAddress(BTcpIpAdapterSettings adapterSettings, Context context)
    throws ValidateChangesException
  {
    BBacnetOctetString change = getPendingChange(IP_ADDRESS, BBacnetOctetString.class);
    if (change != null)
    {
      validateAddress(change, ipAddress);
      validateCanWrite(adapterSettings, BTcpIpAdapterSettings.ipAddress, ipAddress, context);
      validateCanWriteAdapterSettings(adapterSettings, ipAddress);
      checkDhcpDisabled(ipAddress);
    }
  }

  private void activateIpAddress(BTcpIpAdapterSettings adapterSettings, Context context)
  {
    BBacnetOctetString change = getPendingChange(IP_ADDRESS, BBacnetOctetString.class);
    if (change != null)
    {
      adapterSettings.setString(BTcpIpAdapterSettings.ipAddress, makeAddressStr(change.getBytes()), context);
    }
  }

  //endregion

  //region SubnetMask

  private PropertyValue readSubnetMask()
  {
    BBacnetOctetString pending = getPendingChange(SUBNET_MASK, BBacnetOctetString.class);
    if (pending != null)
    {
      return makeOctetStringReadResult(ipSubnetMask, pending);
    }

    short networkPrefix = getIpLinkLayer().getNetworkPrefixLength();
    if (networkPrefix < 0 || networkPrefix > 32)
    {
      return makeReadError(ipSubnetMask, property, valueNotInitialized);
    }
    int netmask = BacnetIpLinkUtil.convertNetmask(networkPrefix);
    byte[] subnetMask = BacnetIpLinkUtil.convertIntToAddress(netmask);
    return makeOctetStringReadResult(ipSubnetMask, subnetMask);
  }

  private ErrorType writeSubnetMask(byte[] value)
    throws AsnException
  {
    BBacnetOctetString octetString = AsnUtil.fromAsnOctetString(value);
    try
    {
      validateAddress(octetString, ipSubnetMask);
    }
    catch (ValidateChangesException e)
    {
      return e.getError();
    }

    try
    {
      addPendingChange(SUBNET_MASK, octetString, BLocalBacnetDevice.getBacnetContext());
      return null;
    }
    catch (Exception e)
    {
      return new NErrorType(property, writeAccessDenied);
    }
  }

  public void validateSetSubnetMask(BValue value, Context context)
  {
    checkPendingChangeType(SUBNET_MASK, value, BBacnetOctetString.TYPE);
    BTcpIpAdapterSettings adapterSettings = findAdapterSettings();
    checkCanSaveAdapterSettings(adapterSettings, context);
    checkCanWrite(adapterSettings, BTcpIpAdapterSettings.subnetMask, context);
    checkCanWriteAdapterSettings(adapterSettings);
  }

  private void validateSubnetMask(BTcpIpAdapterSettings adapterSettings, Context context)
    throws ValidateChangesException
  {
    BBacnetOctetString change = getPendingChange(SUBNET_MASK, BBacnetOctetString.class);
    if (change != null)
    {
      validateAddress(change, ipSubnetMask);
      validateCanWrite(adapterSettings, BTcpIpAdapterSettings.subnetMask, ipSubnetMask, context);
      validateCanWriteAdapterSettings(adapterSettings, ipSubnetMask);
      checkDhcpDisabled(ipSubnetMask);
    }
  }

  private void activateSubnetMask(BTcpIpAdapterSettings adapterSettings, Context context)
  {
    BBacnetOctetString change = getPendingChange(SUBNET_MASK, BBacnetOctetString.class);
    if (change != null)
    {
      adapterSettings.setString(BTcpIpAdapterSettings.subnetMask, makeAddressStr(change.getBytes()), context);
    }
  }

  //endregion

  //region IpDefaultGateway

  private PropertyValue readDefaultGateway()
  {
    BBacnetOctetString defaultGateway = getPendingChange(DEFAULT_GATEWAY, BBacnetOctetString.class);
    if (defaultGateway == null)
    {
      defaultGateway = getAdapterDefaultGateway();
      if (defaultGateway == null)
      {
        return makeReadError(ipDefaultGateway, property, other);
      }
    }
    return makeOctetStringReadResult(ipDefaultGateway, defaultGateway);
  }

  private BBacnetOctetString getAdapterDefaultGateway()
  {
    BTcpIpAdapterSettings adapterSettings = findAdapterSettings();
    if (adapterSettings == null)
    {
      return null;
    }

    String defaultGateway = adapterSettings.getUsesAdapterLevelSettings() ?
      adapterSettings.getDefaultGateway().trim() :
      adapterSettings.getHostSettings().getDefaultGateway().trim();
    return defaultGateway.isEmpty() ?
      BBacnetOctetString.make(EMPTY_BYTE_ARRAY) :
      BBacnetOctetString.make(BacnetIpLinkUtil.parseIpBytes(defaultGateway));
  }

  private ErrorType writeDefaultGateway(byte[] value)
    throws AsnException
  {
    BBacnetOctetString octetString = AsnUtil.fromAsnOctetString(value);
    int length = octetString.getBytes().length;
    if (length != 0 && length != 4)
    {
      return new NErrorType(property, invalidDataEncoding);
    }

    try
    {
      addPendingChange(DEFAULT_GATEWAY, octetString, BLocalBacnetDevice.getBacnetContext());
      return null;
    }
    catch (Exception e)
    {
      return new NErrorType(property, writeAccessDenied);
    }
  }

  private void validateSetDefaultGateway(BValue value, Context context)
  {
    checkPendingChangeType(DEFAULT_GATEWAY, value, BBacnetOctetString.TYPE);
    BTcpIpAdapterSettings adapterSettings = findAdapterSettings();
    checkCanSaveAdapterSettings(adapterSettings, context);
    if (adapterSettings.getUsesAdapterLevelSettings())
    {
      checkCanWrite(adapterSettings, BTcpIpAdapterSettings.defaultGateway, context);
      checkCanWriteAdapterSettings(adapterSettings);
    }
    else
    {
      checkCanWrite(adapterSettings.getHostSettings(), BTcpIpHostSettings.defaultGateway, context);
      checkCanWriteHostSettings(adapterSettings);
    }
  }

  private void validateDefaultGateway(BTcpIpAdapterSettings adapterSettings, Context context)
    throws ValidateChangesException
  {
    BBacnetOctetString change = getPendingChange(DEFAULT_GATEWAY, BBacnetOctetString.class);
    if (change != null)
    {
      int length = change.getBytes().length;
      if (length != 0 && length != 4)
      {
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ invalidDataEncoding,
          /* property */ ipDefaultGateway,
          /* details */ "Octet string length is " + length + " instead of 0 or 4");
      }

      if (adapterSettings.getUsesAdapterLevelSettings())
      {
        validateCanWrite(adapterSettings, BTcpIpAdapterSettings.defaultGateway, ipDefaultGateway, context);
        validateCanWriteAdapterSettings(adapterSettings, ipDefaultGateway);
      }
      else
      {
        validateCanWrite(adapterSettings.getHostSettings(), BTcpIpHostSettings.defaultGateway, ipDefaultGateway, context);
        validateCanWriteHostSettings(adapterSettings, ipDefaultGateway);
      }

      checkDhcpDisabled(ipDefaultGateway);
    }
  }

  private void activateDefaultGateway(BTcpIpAdapterSettings adapterSettings, Context context)
  {
    BBacnetOctetString change = getPendingChange(DEFAULT_GATEWAY, BBacnetOctetString.class);
    if (change != null)
    {
      byte[] bytes = change.getBytes();
      String changeStr = bytes.length == 0 ? "" : makeAddressStr(bytes);
      if (adapterSettings.getUsesAdapterLevelSettings())
      {
        adapterSettings.setString(BTcpIpAdapterSettings.defaultGateway, changeStr, context);
      }
      else
      {
        adapterSettings.getHostSettings().setString(BTcpIpHostSettings.defaultGateway, changeStr, context);
      }
    }
  }

  //endregion

  //region IpDnsServer

  private PropertyValue readDnsServer(int index)
  {
    BBacnetArray dnsServer = getDnsServer();
    if (dnsServer == null)
    {
      return makeReadError(ipDnsServer, index, property, other);
    }

    return readArray(BBacnetPropertyIdentifier.IP_DNS_SERVER, index, dnsServer);
  }

  private BBacnetArray getDnsServer()
  {
    BBacnetArray dnsServer = getPendingChange(DNS_SERVER, BBacnetArray.class);
    if (dnsServer == null)
    {
      dnsServer = getAdapterDnsServer();
    }

    if (dnsServer != null && dnsServer.getSize() <= 0)
    {
      dnsServer.addElement(BBacnetOctetString.make(new byte[4]));
    }

    return dnsServer;
  }

  private BBacnetArray getAdapterDnsServer()
  {
    BTcpIpAdapterSettings adapterSettings = findAdapterSettings();
    if (adapterSettings == null)
    {
      return null;
    }

    BVector dnsHosts = adapterSettings.getUsesAdapterLevelSettings() ?
      adapterSettings.getDnsHosts() :
      adapterSettings.getHostSettings().getDnsHosts();

    BBacnetArray dnsServer = new BBacnetArray(BBacnetOctetString.TYPE);
    for (BString dnsHost : dnsHosts.getChildren(BString.class))
    {
      byte[] dnsAddress = BacnetIpLinkUtil.parseIpBytes(dnsHost.getString().trim());
      if (dnsAddress == null || dnsAddress.length != 4)
      {
        dnsAddress = new byte[4];
      }
      dnsServer.addElement(BBacnetOctetString.make(dnsAddress));
    }
    return dnsServer;
  }

  private ErrorType writeDnsServer(int index, byte[] value)
    throws AsnException
  {
    BBacnetArray dnsServer = getDnsServer();
    if (dnsServer == null)
    {
      if (logger.isLoggable(Level.INFO))
      {
        logger.info(this + ": could not find DNS server setting when attempt to write pending change to it");
      }
      return new NErrorType(property, writeAccessDenied);
    }

    if (index == 0)
    {
      int newSize = AsnUtil.fromAsnUnsignedInt(value);
      dnsServer.setSize(newSize);

      // Replace null elements with an empty octet string of size 4
      for (int i = 1; i <= newSize; i++)
      {
        BBacnetOctetString element = (BBacnetOctetString) dnsServer.getElement(i);
        if (element == null)
        {
          dnsServer.add("element" + i, BBacnetOctetString.make(new byte[4]));
        }
        else if (element.isNull())
        {
          dnsServer.setElement(i, BBacnetOctetString.make(new byte[4]));
        }
      }
    }
    else if (index == NOT_USED)
    {
      dnsServer = (BBacnetArray) AsnUtil.fromAsn(ASN_BACNET_ARRAY, value, new BBacnetArray(BBacnetOctetString.TYPE));
    }
    else if (index > dnsServer.getSize())
    {
      return new NErrorType(property, invalidArrayIndex);
    }
    else
    {
      BBacnetOctetString dnsHost = AsnUtil.fromAsnOctetString(value);
      if (dnsHost.length() != 4)
      {
        return new NErrorType(property, invalidDataEncoding);
      }
      dnsServer.setElement(index, dnsHost);
    }

    try
    {
      addPendingChange(DNS_SERVER, dnsServer, BLocalBacnetDevice.getBacnetContext());
      return null;
    }
    catch (Exception e)
    {
      return new NErrorType(property, writeAccessDenied);
    }
  }

  private void validateSetDnsServer(BValue value, Context context)
  {
    checkPendingChangeType(DNS_SERVER, value, BBacnetArray.TYPE);
    BTcpIpAdapterSettings adapterSettings = findAdapterSettings();
    checkCanSaveAdapterSettings(adapterSettings, context);
    if (adapterSettings.getUsesAdapterLevelSettings())
    {
      checkCanWrite(adapterSettings, BTcpIpAdapterSettings.dnsHosts, context);
      checkCanWriteAdapterSettings(adapterSettings);
    }
    else
    {
      checkCanWrite(adapterSettings.getHostSettings(), BTcpIpAdapterSettings.dnsHosts, context);
      checkCanWriteHostSettings(adapterSettings);
    }
  }

  private void validateDnsServer(BTcpIpAdapterSettings adapterSettings, Context context)
    throws ValidateChangesException
  {
    BBacnetArray change = getPendingChange(DNS_SERVER, BBacnetArray.class);
    if (change != null)
    {
      try
      {
        for (BBacnetOctetString dnsAddress : change.getChildren(BBacnetOctetString.class))
        {
          BacnetIpLinkUtil.validateAddressLength(dnsAddress.getBytes(), 4);
        }
      }
      catch (IllegalArgumentException e)
      {
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(this + ": failed to validate dnsAddress element in DNS server array: " + change);
        }
        throw new ValidateChangesException(
          /* errorClass */ property,
          /* errorCode */ invalidDataEncoding,
          /* property */ ipDnsServer,
          /* details */ e.getMessage());
      }

      if (adapterSettings.getUsesAdapterLevelSettings())
      {
        validateCanWrite(adapterSettings, BTcpIpAdapterSettings.dnsHosts, ipDnsServer, context);
        validateCanWriteAdapterSettings(adapterSettings, ipDnsServer);
      }
      else
      {
        validateCanWrite(adapterSettings.getHostSettings(), BTcpIpHostSettings.dnsHosts, ipDnsServer, context);
        validateCanWriteHostSettings(adapterSettings, ipDnsServer);
      }
    }
  }

  private void activateDnsServer(BTcpIpAdapterSettings adapterSettings, Context context)
  {
    BBacnetArray change = getPendingChange(DNS_SERVER, BBacnetArray.class);
    if (change != null)
    {
      BVector dsnHosts = new BVector();
      for (BBacnetOctetString dnsAddress : change.getChildren(BBacnetOctetString.class))
      {
        dsnHosts.add(null, BString.make(makeAddressStr(dnsAddress.getBytes())));
      }

      if (adapterSettings.getUsesAdapterLevelSettings())
      {
        adapterSettings.set(BTcpIpAdapterSettings.dnsHosts, dsnHosts, context);
      }
      else
      {
        adapterSettings.getHostSettings().set(BTcpIpHostSettings.dnsHosts, dsnHosts, context);
      }
    }
  }

  //endregion

  //region IpDhcpEnable

  private PropertyValue readDhcpEnable()
  {
    BBoolean pending = getPendingChange(DHCP_ENABLE, BBoolean.class);
    if (pending != null)
    {
      return makeBooleanReadResult(ipDhcpEnable, pending.getBoolean());
    }

    BTcpIpAdapterSettings adapterSettings = findAdapterSettings();
    if (adapterSettings != null)
    {
      return makeBooleanReadResult(ipDhcpEnable, adapterSettings.getIsDhcpEnabled());
    }

    return makeReadError(ipDhcpEnable, property, other);
  }

  /**
   * Returns the value of a pending change to DHCP_ENABLE or, if not found, the adapter setting.
   * Returns false if there is no pending change and the adapter settings cannot be found.
   */
  private boolean getDhcpEnable()
  {
    BBoolean pending = getPendingChange(DHCP_ENABLE, BBoolean.class);
    if (pending != null)
    {
      return pending.getBoolean();
    }

    BTcpIpAdapterSettings adapterSettings = findAdapterSettings();
    return adapterSettings != null && adapterSettings.getIsDhcpEnabled();
  }

  private ErrorType writeDhcpEnable(byte[] value)
    throws AsnException
  {
    boolean dhcpEnable = AsnUtil.fromAsnBoolean(value);
    try
    {
      addPendingChange(DHCP_ENABLE, BBoolean.make(dhcpEnable), BLocalBacnetDevice.getBacnetContext());
      return null;
    }
    catch (Exception e)
    {
      return new NErrorType(property, writeAccessDenied);
    }
  }

  public void validateSetDhcpEnable(BValue value, Context context)
  {
    checkPendingChangeType(DHCP_ENABLE, value, BBoolean.TYPE);
    BTcpIpAdapterSettings adapterSettings = findAdapterSettings();
    checkCanSaveAdapterSettings(adapterSettings, context);
    checkCanWrite(adapterSettings, BTcpIpAdapterSettings.isDhcpEnabled, context);
    checkCanWriteAdapterSettings(adapterSettings);
  }

  private void validateDhcpEnable(BTcpIpAdapterSettings adapterSettings, Context context)
    throws ValidateChangesException
  {
    BBoolean change = getPendingChange(DHCP_ENABLE, BBoolean.class);
    if (change != null)
    {
      validateCanWrite(adapterSettings, BTcpIpAdapterSettings.isDhcpEnabled, ipDhcpEnable, context);
      validateCanWriteAdapterSettings(adapterSettings, ipDhcpEnable);
    }
  }

  private void activateDhcpEnable(BTcpIpAdapterSettings adapterSettings, Context context)
  {
    BBoolean change = getPendingChange(DHCP_ENABLE, BBoolean.class);
    if (change != null)
    {
      adapterSettings.setBoolean(BTcpIpAdapterSettings.isDhcpEnabled, change.getBoolean(), context);
    }
  }

  //endregion

  //region IpDhcpLeaseTime

  private PropertyValue readDhcpLeaseTime()
  {
    long leaseTime = 0;
    try
    {
      if (getPendingChange(DHCP_ENABLE, BBoolean.class) == null)
      {
        BTcpIpAdapterSettings adapterSettings = findAdapterSettings();
        if (adapterSettings == null)
        {
          return makeReadError(ipDhcpLeaseTime, property, other);
        }

        if (adapterSettings.getIsDhcpEnabled())
        {
          String leastGrantedStr = adapterSettings.getDhcpLeaseGranted().trim();
          if (!leastGrantedStr.isEmpty())
          {
            LocalDateTime leaseGranted = parseDateTime(leastGrantedStr);
            leaseTime = Math.max(0, leaseGranted.until(LocalDateTime.now(), ChronoUnit.SECONDS));
          }
        }
      }
    }
    catch (Exception e)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.log(Level.FINE, this + "Unable to parse the dhcpLeaseGranted", e);
      }
    }
    return makeUnsignedReadResult(ipDhcpLeaseTime, leaseTime);
  }

  //endregion

  //region IpDhcpLeaseTimeRemaining

  private PropertyValue readDhcpLeaseTimeRemaining()
  {
    long timeRemaining = 0;
    try
    {
      if (getPendingChange(DHCP_ENABLE, BBoolean.class) == null)
      {
        BTcpIpAdapterSettings adapterSettings = findAdapterSettings();
        if (adapterSettings == null)
        {
          return makeReadError(ipDhcpLeaseTimeRemaining, property, other);
        }

        if (adapterSettings.getIsDhcpEnabled())
        {
          String leaseExpiresStr = adapterSettings.getDhcpLeaseExpires().trim();
          if (!leaseExpiresStr.isEmpty())
          {
            LocalDateTime leaseExpires = parseDateTime(leaseExpiresStr);
            timeRemaining = Math.max(0, LocalDateTime.now().until(leaseExpires, ChronoUnit.SECONDS));
          }
        }
      }
    }
    catch (Exception e)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.log(Level.FINE, this + "Unable to parse the dhcpLeaseExpires", e);
      }
    }
    return makeUnsignedReadResult(ipDhcpLeaseTimeRemaining, timeRemaining);
  }

  private static LocalDateTime parseDateTime(String dateTimeStr)
  {
    return LocalDateTime.parse(dateTimeStr, DateTimeFormatter.ofPattern("EEE MMM dd HH:mm:ss yyyy"));
  }

  //endregion

  //region IpDhcpServer

  private PropertyValue readDhcpServer()
  {
    BTcpIpAdapterSettings adapterSettings = findAdapterSettings();
    if (adapterSettings == null)
    {
      return makeReadError(ipDhcpServer, property, other);
    }

    byte[] dhcpAddress = BacnetIpLinkUtil.parseIpBytes(adapterSettings.getDhcpHost());
    if (dhcpAddress == null || dhcpAddress.length != 4)
    {
      return makeOctetStringReadResult(ipDhcpServer, new byte[4]);
    }
    else
    {
      return makeOctetStringReadResult(ipDhcpServer, dhcpAddress);
    }
  }

  //endregion

  //region NetworkInterfaceName

  private PropertyValue readNetworkInterfaceName()
  {
    return makeCharStringReadResult(networkInterfaceName, SlotPath.unescape(getIpLinkLayer().getAdapter().getTag()));
  }

  //endregion

  //region Utility

  private BBacnetIpLinkLayer getIpLinkLayer()
  {
    return (BBacnetIpLinkLayer) networkPort.getLink();
  }

  private static BTcpIpPlatformService getTcpIpService()
  {
    try
    {
      return (BTcpIpPlatformService)Sys.getService(BTcpIpPlatformService.TYPE);
    }
    catch (Exception e)
    {
      logger.log(Level.SEVERE, "Failed to resolve BacnetTcpIpPlatformService", logger.isLoggable(Level.FINE) ? e : null);
      return null;
    }
  }

  private BTcpIpAdapterSettings findAdapterSettings()
  {
    BTcpIpPlatformService platformService = getTcpIpService();
    if (platformService == null)
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(this + ": could not find TCP/IP platform service");
      }
      return null;
    }

    platformService.checkPropertiesLoaded();

    String adapterOsName = SlotPath.unescape(getIpLinkLayer().getAdapterId().getTag());
    String adapterId = getAdapterIdFromOsName(adapterOsName);

    BTcpIpHostSettings hostSettings = platformService.getSettings();
    BTcpIpAdapterSettings[] allAdapterSettings = hostSettings.getAdapters().getChildren(BTcpIpAdapterSettings.class);
    for (BTcpIpAdapterSettings adapterSettings : allAdapterSettings)
    {
      if (adapterSettings.getIsAdapterEnabled() &&
          adapterSettings.getAdapterId().equals(adapterId))
      {
        return adapterSettings;
      }
    }

    if (logger.isLoggable(Level.FINE))
    {
      StringJoiner adapterInfo = new StringJoiner(",");
      for (BTcpIpAdapterSettings adapterSettings : allAdapterSettings)
      {
        adapterInfo.add(
          "adapterId: " + adapterSettings.getAdapterId() +
          ", isEnabled? " + adapterSettings.getIsAdapterEnabled());
      }

      logger.fine(this + ": could not find TCP/IP adapter settings" +
        "; port adapter OS name: " + adapterOsName +
        ", port adapter ID: " + adapterId +
        ", available adapters: {" + adapterInfo + "}");
    }
    return null;
  }

  /**
   * Take the osname (dm*) and return the adapterId (en*)
   */
  private String getAdapterIdFromOsName(String adapterOsName)
  {
    // Conversion only necessary for legacy Tridium JACE platforms that use dm1 -> en1 translation (JACE-8000)
    if (logger.isLoggable(Level.FINER))
    {
      logger.finer(this + ": converting adapter OS name to platform service ID" +
        "; OS name: " + adapterOsName +
        ", isTridiumQnx? " + IS_TRIDIUM_QNX +
        ", isTridiumPlatform? " + PlatformUtil.isTridiumPlatform() +
        ", isQnx? " + OperatingSystemEnum.isOS(OperatingSystemEnum.qnx));
    }

    if (IS_TRIDIUM_QNX && adapterOsName.startsWith("dm"))
    {
      return ("en" + adapterOsName.substring(2)).trim();
    }
    return adapterOsName;
  }

  private static void validateAddress(
    BBacnetOctetString octetString,
    BBacnetPropertyIdentifier propertyId)
      throws ValidateChangesException
  {
    try
    {
      BacnetIpLinkUtil.validateAddressLength(octetString.getBytes(), 4);
    }
    catch (IllegalArgumentException e)
    {
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ invalidDataEncoding,
        /* property */ propertyId,
        /* details */ e.getMessage());
    }
  }

  private void checkDhcpDisabled(BBacnetPropertyIdentifier propertyId)
    throws ValidateChangesException
  {
    if (getDhcpEnable())
    {
      if (logger.isLoggable(Level.FINE))
      {
        logger.fine(getObjectId() + ": attempted to write " + propertyId + " when DHCP Enable is true");
      }
      throw new ValidateChangesException(
        /* errorClass */ property,
        /* errorCode */ writeAccessDenied,
        /* property */ propertyId,
        /* details */ "Unable to write " + propertyId + " when DHCP Enable is true");
    }
  }

  private static BIpDeviceType makeDeviceType(BBacnetIpMode ipMode)
  {
    switch (ipMode.getOrdinal())
    {
      case BBacnetIpMode.NORMAL:
        return BIpDeviceType.standard;
      case BBacnetIpMode.FOREIGN:
        return BIpDeviceType.foreignDevice;
      case BBacnetIpMode.BBMD:
      default:
        return BIpDeviceType.bbmd;
    }
  }

  //endregion

  //region Fields

  private final boolean IS_TRIDIUM_QNX;

  private static final String UDP_PORT = "UdpPort";
  private static final String IP_MODE = "IpMode";
  private static final String BBMD_BROADCAST_DISTRIBUTION_TABLE = "BbmdBroadcastDistributionTable";
  private static final String BBMD_ACCEPT_FD_REGISTRATIONS = "BbmdAcceptForeignDeviceRegistrations";
  private static final String FD_BBMD_ADDRESS = "ForeignDeviceBbmdAddress";
  private static final String FD_SUBSCRIPTION_LIFETIME = "ForeignDeviceSubscriptionLifetime";

  private static final String IP_ADDRESS = "IpAddress";
  private static final String SUBNET_MASK = "SubnetMask";
  private static final String DEFAULT_GATEWAY = "DefaultGateway";
  private static final String DNS_SERVER = "DnsServer";
  private static final String DHCP_ENABLE = "DhcpEnable";

  private static final int[] OPTIONAL_PROPS = {
    BBacnetPropertyIdentifier.DESCRIPTION,
    BBacnetPropertyIdentifier.COMMAND,
    BBacnetPropertyIdentifier.NETWORK_NUMBER,
    BBacnetPropertyIdentifier.NETWORK_NUMBER_QUALITY,
    BBacnetPropertyIdentifier.APDU_LENGTH,
    BBacnetPropertyIdentifier.MAC_ADDRESS,
    BBacnetPropertyIdentifier.BACNET_IP_MODE,
    BBacnetPropertyIdentifier.BACNET_IP_UDP_PORT,
    BBacnetPropertyIdentifier.ROUTING_TABLE,
    BBacnetPropertyIdentifier.BBMD_BROADCAST_DISTRIBUTION_TABLE,
    BBacnetPropertyIdentifier.BBMD_ACCEPT_FD_REGISTRATIONS,
    BBacnetPropertyIdentifier.BBMD_FOREIGN_DEVICE_TABLE,
    BBacnetPropertyIdentifier.FD_BBMD_ADDRESS,
    BBacnetPropertyIdentifier.FD_SUBSCRIPTION_LIFETIME,
    BBacnetPropertyIdentifier.IP_ADDRESS,
    BBacnetPropertyIdentifier.IP_SUBNET_MASK,
    BBacnetPropertyIdentifier.IP_DEFAULT_GATEWAY,
    BBacnetPropertyIdentifier.IP_DNS_SERVER,
    BBacnetPropertyIdentifier.IP_DHCP_ENABLE,
    BBacnetPropertyIdentifier.IP_DHCP_LEASE_TIME,
    BBacnetPropertyIdentifier.IP_DHCP_LEASE_TIME_REMAINING,
    BBacnetPropertyIdentifier.IP_DHCP_SERVER,
    BBacnetPropertyIdentifier.NETWORK_INTERFACE_NAME
  };

  //endregion
}
