/*
 * Copyright 2024 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.export;

import static javax.baja.bacnet.enums.BBacnetErrorClass.property;
import static javax.baja.bacnet.enums.BBacnetErrorCode.unknownProperty;
import static javax.baja.bacnet.enums.BBacnetErrorCode.writeAccessDenied;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.macAddress;
import static javax.baja.bacnet.enums.BBacnetPropertyIdentifier.networkInterfaceName;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeCharStringReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeOctetStringReadResult;

import java.util.List;
import java.util.logging.Level;

import javax.baja.bacnet.BacnetException;
import javax.baja.bacnet.datatypes.BIBacnetDataType;
import javax.baja.bacnet.enums.BBacnetNetworkType;
import javax.baja.bacnet.enums.BBacnetPropertyIdentifier;
import javax.baja.bacnet.io.ErrorException;
import javax.baja.bacnet.io.ErrorType;
import javax.baja.bacnet.io.PropertyValue;
import javax.baja.naming.SlotPath;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.sys.BValue;
import javax.baja.sys.Context;
import javax.baja.sys.LocalizableRuntimeException;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.bacnet.asn.NErrorType;
import com.tridium.bacnet.asn.NReadPropertyResult;
import com.tridium.bacnet.stack.link.ethernet.BBacnetEthernetLinkLayer;

/**
 * BBacnetEthernetPortDescriptor exports a BNetworkPort that contains a BBacnetEthernetLinkLayer
 * as a BACnet Network Port object with the Network Type set to Ethernet.
 *
 * @author Tim Urenda on 22 Mar 2024
 * @since Niagara 4.15
 */
@NiagaraType
public class BBacnetEthernetPortDescriptor
  extends BBacnetNetworkPortDescriptor
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.export.BBacnetEthernetPortDescriptor(2979906276)1.0$ @*/
/* Generated Mon Mar 25 14:02:40 EDT 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetEthernetPortDescriptor.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  @Override
  protected int getNetworkType()
  {
    return BBacnetNetworkType.ETHERNET;
  }

  @Override
  protected int[] getOptionalProps()
  {
    //noinspection AssignmentOrReturnOfFieldWithMutableType
    return OPTIONAL_PROPS;
  }

  @Override
  protected boolean isArrayProp(int propId)
  {
    return propId == BBacnetPropertyIdentifier.PROPERTY_LIST;
  }

  @Override
  protected boolean isListProp(int propId)
  {
    return propId == BBacnetPropertyIdentifier.ROUTING_TABLE;
  }

  @Override
  protected List<? extends BIBacnetDataType> getListElements(int propertyId)
    throws ErrorException
  {
    if (propertyId == BBacnetPropertyIdentifier.ROUTING_TABLE)
    {
      return getRouterEntries();
    }

    throw new IllegalArgumentException("Invalid propertyId value " + propertyId);
  }

  @Override
  protected PropertyValue readOptionalProperty(int propertyId, int index)
  {
    switch (propertyId)
    {
      case BBacnetPropertyIdentifier.NETWORK_NUMBER:
        return readNetworkNumber();

      case BBacnetPropertyIdentifier.NETWORK_NUMBER_QUALITY:
        return readNetworkNumberQuality();

      case BBacnetPropertyIdentifier.APDU_LENGTH:
        return readApduLength();

      case BBacnetPropertyIdentifier.ROUTING_TABLE:
        return readRoutingTable();

      case BBacnetPropertyIdentifier.MAC_ADDRESS:
        return readMacAddress();

      case BBacnetPropertyIdentifier.NETWORK_INTERFACE_NAME:
        return readNetworkInterfaceName();

      default:
        return new NReadPropertyResult(propertyId, new NErrorType(property, unknownProperty));
    }
  }

  @Override
  protected ErrorType writeOptionalProperty(int propertyId, int index, byte[] value, int priority)
    throws BacnetException
  {
    switch (propertyId)
    {
      case BBacnetPropertyIdentifier.NETWORK_NUMBER_QUALITY:
      case BBacnetPropertyIdentifier.APDU_LENGTH:
      case BBacnetPropertyIdentifier.ROUTING_TABLE:
      case BBacnetPropertyIdentifier.MAC_ADDRESS:
      case BBacnetPropertyIdentifier.NETWORK_INTERFACE_NAME:
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(this + ": attempted to write read-only property " + BBacnetPropertyIdentifier.tag(propertyId));
        }
        return new NErrorType(property, writeAccessDenied);

      case BBacnetPropertyIdentifier.NETWORK_NUMBER:
        return writeNetworkNumber(value);

      default:
        if (logger.isLoggable(Level.FINE))
        {
          logger.fine(this + ": attempted to write unknown property " + BBacnetPropertyIdentifier.tag(propertyId));
        }
        return new NErrorType(property, unknownProperty);
    }
  }

  @Override
  protected void validateSetPendingChange(String name, BValue value, Context context)
  {
    if (NETWORK_NUMBER.equals(name))
    {
      validateSetNetworkNumber(value, context);
      return;
    }
    throw new LocalizableRuntimeException(
      "bacnet",
      "networkPortDescriptor.pendingChange.unknown",
      new Object[] { name });
  }

  @Override
  public void validateChanges(Context context)
    throws ValidateChangesException
  {
    if (!hasPendingChanges())
    {
      return;
    }

    checkCanReadPendingChanges(context);
    validateNetworkNumber(context);
  }

  @Override
  public void activateChanges(Context context)
    throws Exception
  {
    if (!hasPendingChanges())
    {
      return;
    }
    activateNetworkNumber(context);
  }

  private PropertyValue readMacAddress()
  {
    byte[] macBytes = getLinkLayer().getMacAddress();
    return makeOctetStringReadResult(macAddress, macBytes != null ? macBytes : new byte[0]);
  }

  private PropertyValue readNetworkInterfaceName()
  {
    return makeCharStringReadResult(networkInterfaceName, SlotPath.unescape(getLinkLayer().getAdapterName().getTag()));
  }

  private BBacnetEthernetLinkLayer getLinkLayer()
  {
    return (BBacnetEthernetLinkLayer) networkPort.getLink();
  }

  private static final int[] OPTIONAL_PROPS = {
    BBacnetPropertyIdentifier.DESCRIPTION,
    BBacnetPropertyIdentifier.COMMAND,
    BBacnetPropertyIdentifier.NETWORK_NUMBER,
    BBacnetPropertyIdentifier.NETWORK_NUMBER_QUALITY,
    BBacnetPropertyIdentifier.APDU_LENGTH,
    BBacnetPropertyIdentifier.ROUTING_TABLE,
    BBacnetPropertyIdentifier.MAC_ADDRESS,
    BBacnetPropertyIdentifier.NETWORK_INTERFACE_NAME
  };
}
