/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.export;

import static javax.baja.bacnet.enums.BBacnetErrorClass.property;
import static javax.baja.bacnet.enums.BBacnetErrorCode.writeAccessDenied;
import static javax.baja.bacnet.enums.BBacnetReliability.noFaultDetected;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.makeConstructedReadResult;
import static javax.baja.bacnet.export.BacnetDescriptorUtil.readArray;

import java.util.Vector;

import javax.baja.alarm.BIAlarmSource;
import javax.baja.alarm.ext.BAlarmSourceExt;
import javax.baja.alarm.ext.offnormal.BStringChangeOfStateAlgorithm;
import javax.baja.bacnet.BacnetException;
import javax.baja.bacnet.datatypes.BBacnetAddress;
import javax.baja.bacnet.datatypes.BBacnetArray;
import javax.baja.bacnet.datatypes.BBacnetDateTime;
import javax.baja.bacnet.datatypes.BBacnetObjectIdentifier;
import javax.baja.bacnet.datatypes.BBacnetOptionalCharacterString;
import javax.baja.bacnet.datatypes.BBacnetTimeStamp;
import javax.baja.bacnet.datatypes.BBacnetValueSource;
import javax.baja.bacnet.enums.BBacnetErrorClass;
import javax.baja.bacnet.enums.BBacnetErrorCode;
import javax.baja.bacnet.enums.BBacnetEventType;
import javax.baja.bacnet.enums.BBacnetObjectType;
import javax.baja.bacnet.enums.BBacnetPropertyIdentifier;
import javax.baja.bacnet.io.AsnException;
import javax.baja.bacnet.io.AsnInput;
import javax.baja.bacnet.io.AsnOutput;
import javax.baja.bacnet.io.ErrorType;
import javax.baja.bacnet.io.PropertyValue;
import javax.baja.control.BControlPoint;
import javax.baja.control.BStringPoint;
import javax.baja.control.BStringWritable;
import javax.baja.nre.annotations.AgentOn;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.security.PermissionException;
import javax.baja.status.BStatus;
import javax.baja.status.BStatusString;
import javax.baja.status.BStatusValue;
import javax.baja.sys.BEnum;
import javax.baja.sys.BIcon;
import javax.baja.sys.BObject;
import javax.baja.sys.BString;
import javax.baja.sys.BValue;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Knob;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.bacnet.asn.AsnOutputStream;
import com.tridium.bacnet.asn.AsnUtil;
import com.tridium.bacnet.asn.NErrorType;
import com.tridium.bacnet.asn.NReadPropertyResult;

/**
 * BBacnetCharacterStringDescriptor is the extension that exposes
 * Bacnet CharacterString capability.
 *
 * @author Joseph Chandler
 * @since 12/7/2013
 */
@NiagaraType(
  agent = @AgentOn(
    types = "control:StringWritable"
  )
)
/*
 objectId is the identifier by which this point is known
 to the Bacnet world.
 */
@NiagaraProperty(
  name = "objectId",
  type = "BBacnetObjectIdentifier",
  defaultValue = "BBacnetObjectIdentifier.make(BBacnetObjectType.CHARACTER_STRING_VALUE)",
  flags = Flags.DEFAULT_ON_CLONE,
  override = true
)
@NiagaraProperty(
  name = "bacnetWritable",
  type = "String",
  defaultValue = "BBacnetPointDescriptor.lexNotWritable",
  flags = Flags.READONLY | Flags.HIDDEN
)
/*
 @since Niagara 4.15
 */
@NiagaraProperty(
  name = "valueSourceArray",
  type = "BBacnetArray",
  defaultValue = "new BBacnetArray(BBacnetValueSource.TYPE, 16)",
  flags = Flags.READONLY | Flags.DEFAULT_ON_CLONE | Flags.TRANSIENT
)
/*
 @since Niagara 4.15
 */
@NiagaraProperty(
  name = "lastCommandTime",
  type = "BBacnetTimeStamp",
  defaultValue = "new BBacnetTimeStamp(new BBacnetDateTime())",
  flags = Flags.READONLY | Flags.DEFAULT_ON_CLONE | Flags.TRANSIENT
)
/*
 @since Niagara 4.15
 */
@NiagaraProperty(
  name = "commandTimeArray",
  type = "BBacnetArray",
  defaultValue = "makeEmptyCommandTimeArray()",
  flags = Flags.READONLY | Flags.DEFAULT_ON_CLONE | Flags.TRANSIENT
)
public class BBacnetCharacterStringDescriptor
  extends BBacnetPointDescriptor
  implements BacnetWritableDescriptor
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.export.BBacnetCharacterStringDescriptor(2393838269)1.0$ @*/
/* Generated Tue Oct 22 12:09:24 CDT 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Property "objectId"

  /**
   * Slot for the {@code objectId} property.
   * objectId is the identifier by which this point is known
   * to the Bacnet world.
   * @see #getObjectId
   * @see #setObjectId
   */
  @Generated
  public static final Property objectId = newProperty(Flags.DEFAULT_ON_CLONE, BBacnetObjectIdentifier.make(BBacnetObjectType.CHARACTER_STRING_VALUE), null);

  //endregion Property "objectId"

  //region Property "bacnetWritable"

  /**
   * Slot for the {@code bacnetWritable} property.
   * @see #getBacnetWritable
   * @see #setBacnetWritable
   */
  @Generated
  public static final Property bacnetWritable = newProperty(Flags.READONLY | Flags.HIDDEN, BBacnetPointDescriptor.lexNotWritable, null);

  /**
   * Get the {@code bacnetWritable} property.
   * @see #bacnetWritable
   */
  @Generated
  public String getBacnetWritable() { return getString(bacnetWritable); }

  /**
   * Set the {@code bacnetWritable} property.
   * @see #bacnetWritable
   */
  @Generated
  public void setBacnetWritable(String v) { setString(bacnetWritable, v, null); }

  //endregion Property "bacnetWritable"

  //region Property "valueSourceArray"

  /**
   * Slot for the {@code valueSourceArray} property.
   * @since Niagara 4.15
   * @see #getValueSourceArray
   * @see #setValueSourceArray
   */
  @Generated
  public static final Property valueSourceArray = newProperty(Flags.READONLY | Flags.DEFAULT_ON_CLONE | Flags.TRANSIENT, new BBacnetArray(BBacnetValueSource.TYPE, 16), null);

  /**
   * Get the {@code valueSourceArray} property.
   * @since Niagara 4.15
   * @see #valueSourceArray
   */
  @Generated
  public BBacnetArray getValueSourceArray() { return (BBacnetArray)get(valueSourceArray); }

  /**
   * Set the {@code valueSourceArray} property.
   * @since Niagara 4.15
   * @see #valueSourceArray
   */
  @Generated
  public void setValueSourceArray(BBacnetArray v) { set(valueSourceArray, v, null); }

  //endregion Property "valueSourceArray"

  //region Property "lastCommandTime"

  /**
   * Slot for the {@code lastCommandTime} property.
   * @since Niagara 4.15
   * @see #getLastCommandTime
   * @see #setLastCommandTime
   */
  @Generated
  public static final Property lastCommandTime = newProperty(Flags.READONLY | Flags.DEFAULT_ON_CLONE | Flags.TRANSIENT, new BBacnetTimeStamp(new BBacnetDateTime()), null);

  /**
   * Get the {@code lastCommandTime} property.
   * @since Niagara 4.15
   * @see #lastCommandTime
   */
  @Generated
  public BBacnetTimeStamp getLastCommandTime() { return (BBacnetTimeStamp)get(lastCommandTime); }

  /**
   * Set the {@code lastCommandTime} property.
   * @since Niagara 4.15
   * @see #lastCommandTime
   */
  @Generated
  public void setLastCommandTime(BBacnetTimeStamp v) { set(lastCommandTime, v, null); }

  //endregion Property "lastCommandTime"

  //region Property "commandTimeArray"

  /**
   * Slot for the {@code commandTimeArray} property.
   * @since Niagara 4.15
   * @see #getCommandTimeArray
   * @see #setCommandTimeArray
   */
  @Generated
  public static final Property commandTimeArray = newProperty(Flags.READONLY | Flags.DEFAULT_ON_CLONE | Flags.TRANSIENT, makeEmptyCommandTimeArray(), null);

  /**
   * Get the {@code commandTimeArray} property.
   * @since Niagara 4.15
   * @see #commandTimeArray
   */
  @Generated
  public BBacnetArray getCommandTimeArray() { return (BBacnetArray)get(commandTimeArray); }

  /**
   * Set the {@code commandTimeArray} property.
   * @since Niagara 4.15
   * @see #commandTimeArray
   */
  @Generated
  public void setCommandTimeArray(BBacnetArray v) { set(commandTimeArray, v, null); }

  //endregion Property "commandTimeArray"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetCharacterStringDescriptor.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////////////////////

  public BBacnetCharacterStringDescriptor()
  {
  }

////////////////////////////////////////////////////////////////
// Overrides
////////////////////////////////////////////////////////////////

  /**
   * BBacnetCharacterStringDescriptor may only expose a BStringPoint.
   *
   * @param pt the exposed point
   * @return true if the Niagara point type is legal for this point type.
   */
  @Override
  protected boolean isPointTypeLegal(BControlPoint pt)
  {
    return pt instanceof BStringPoint;
  }

  /**
   * Get the BACnetEventType reported by this object.
   */
  @Override
  public BEnum getEventType()
  {
    return BBacnetEventType.changeOfCharacterstring;
  }

  /**
   * Is the given alarm source ext a valid extension for
   * exporting BACnet alarm properties?  This determines if the
   * given alarm source extension follows the appropriate algorithm
   * defined for the intrinsic alarming of a particular object
   * type as required by the BACnet specification.<p>
   * All BACnet Analog points use an OutOfRange alarm algorithm.
   *
   * @param ext
   * @return true if valid, otherwise false.
   */
  @Override
  public boolean isValidAlarmExt(BIAlarmSource ext)
  {
    if (ext instanceof BAlarmSourceExt)
    {
      return ((BAlarmSourceExt)ext).getOffnormalAlgorithm() instanceof BStringChangeOfStateAlgorithm;
    }

    return false;
  }

  @Override
  protected boolean isCommandable()
  {
    return true;
  }

////////////////////////////////////////////////////////////////
// Bacnet Access
////////////////////////////////////////////////////////////////

  /**
   * Get the value of a property.
   * Subclasses with additional properties override this to check for
   * their properties.  If no match is found, call this superclass
   * method to check these properties.
   *
   * @param pId the requested property-identifier.
   * @param ndx the property array index (-1 if not specified).
   * @return a PropertyValue containing either the encoded value or the error.
   */
  @Override
  protected PropertyValue readProperty(int pId, int ndx)
  {
    BStringWritable pt = (BStringWritable)getPoint();
    if (pt == null)
    {
      return new NReadPropertyResult(pId, ndx, new NErrorType(BBacnetErrorClass.OBJECT,
                                                              BBacnetErrorCode.TARGET_NOT_CONFIGURED));
    }

    // Check for array index on non-array property.
    if (ndx >= 0)
    {
      if (!isArray(pId))
      {
        return new NReadPropertyResult(pId, ndx, new NErrorType(BBacnetErrorClass.PROPERTY,
                                                                BBacnetErrorCode.PROPERTY_IS_NOT_AN_ARRAY));
      }
    }

    switch (pId)
    {
      case BBacnetPropertyIdentifier.PRIORITY_ARRAY:
        return readPriorityArray(ndx);

      case BBacnetPropertyIdentifier.RELINQUISH_DEFAULT:
        return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnCharacterString(pt.getFallback().getValue()));

      case BBacnetPropertyIdentifier.PRESENT_VALUE:
        return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnCharacterString(pt.getOut().getValue()));

      default:
        return super.readProperty(pId, ndx);
    }
  }

  @Override
  protected PropertyValue readOptionalProperty(int pId, int ndx)
  {
    switch (pId)
    {
      case BBacnetPropertyIdentifier.ALARM_VALUES:
        BAlarmSourceExt alarmExt = getAlarmExt();
        if ((alarmExt == null) || !(alarmExt.getOffnormalAlgorithm() instanceof BStringChangeOfStateAlgorithm))
        {
          return new NReadPropertyResult(pId, ndx, new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.UNKNOWN_PROPERTY));
        }

        String alarmExpression = ((BStringChangeOfStateAlgorithm)alarmExt.getOffnormalAlgorithm()).getExpression();

        AsnOutputStream asnOut = new AsnOutputStream();
        new BBacnetOptionalCharacterString(alarmExpression).writeAsn(asnOut);
        byte[] val = asnOut.toByteArray();

        return new NReadPropertyResult(pId, ndx, val);

      case BBacnetPropertyIdentifier.CURRENT_COMMAND_PRIORITY:
        return readCurrentCommandPriority();

      case BBacnetPropertyIdentifier.VALUE_SOURCE:
        return readValueSource(getValueSourceArray());

      case BBacnetPropertyIdentifier.VALUE_SOURCE_ARRAY:
        return readArray(pId, ndx, getValueSourceArray());

      case BBacnetPropertyIdentifier.LAST_COMMAND_TIME:
        return makeConstructedReadResult(BBacnetPropertyIdentifier.lastCommandTime, getLastCommandTime());

      case BBacnetPropertyIdentifier.COMMAND_TIME_ARRAY:
        return readArray(pId, ndx, getCommandTimeArray());
    }

    return super.readOptionalProperty(pId, ndx);
  }

  /**
   * The interface value property is not supported for the characterstring points.
   * @since Niagara 4.14
   */
  @Override
  protected byte[] makeInterfaceValue(BStatusValue proxyValue)
  {
    throw new UnsupportedOperationException("BACnet Characterstring object does not have an interface value property");
  }

  /**
   * Set the value of a property.
   * Subclasses with additional properties override this to check for
   * their properties.  If no match is found, call this superclass
   * method to check these properties.
   *
   * @param pId the requested property-identifier.
   * @param ndx the property array index (-1 if not specified).
   * @param val the Asn-encoded value for the property.
   * @param pri the priority level (only used for commandable properties).
   * @return null if everything goes OK, or
   * an ErrorType describing the error if not.
   */
  @Override
  protected ErrorType writeProperty(int pId,
                                    int ndx,
                                    byte[] val,
                                    int pri)
    throws BacnetException
  {
    BStringWritable pt = (BStringWritable)getPoint();
    if (pt == null)
    {
      return new NErrorType(BBacnetErrorClass.OBJECT,
                            BBacnetErrorCode.TARGET_NOT_CONFIGURED);
    }

    // Check for array index on non-array property.
    if (ndx >= 0)
    {
      if (!isArray(pId))
      {
        return new NErrorType(BBacnetErrorClass.PROPERTY,
                              BBacnetErrorCode.PROPERTY_IS_NOT_AN_ARRAY);
      }
    }

    try
    {
      switch (pId)
      {
        case BBacnetPropertyIdentifier.PRIORITY_ARRAY:
          return new NErrorType(BBacnetErrorClass.PROPERTY,
                                BBacnetErrorCode.WRITE_ACCESS_DENIED);

        case BBacnetPropertyIdentifier.RELINQUISH_DEFAULT:
          BStatusString fb = pt.getFallback();
          fb.setString(BStatusString.value,
                       AsnUtil.fromAsnCharacterString(val),
                       BLocalBacnetDevice.getBacnetContext());
          fb.setStatusNull(false);
          return null;

        default:
          return super.writeProperty(pId, ndx, val, pri);
      }
    }
    catch (AsnException e)
    {
      log.warning("AsnException writing property " + pId + " in object " + getObjectId() + ": "  + e);
      return new NErrorType(BBacnetErrorClass.PROPERTY,
                            BBacnetErrorCode.INVALID_DATA_TYPE);
    }
    catch (PermissionException e)
    {
      log.warning("PermissionException writing property " + pId + " in object " + getObjectId() + ": "  + e);
      return new NErrorType(BBacnetErrorClass.PROPERTY,
                            BBacnetErrorCode.WRITE_ACCESS_DENIED);
    }
  }

  /**
   * Set the value of a property.
   * Subclasses with additional properties override this to check for
   * their properties.  If no match is found, call this superclass
   * method to check these properties.
   *
   * @param pId the requested property-identifier.
   * @param ndx the property array index (-1 if not specified).
   * @param val the Asn-encoded value for the property.
   * @param pri the priority level (only used for commandable properties).
   * @param sourceAddress the address of the BACnet device that originally sent the write request.
   * @return null if everything goes OK, or
   * an ErrorType describing the error if not.
   * @since Niagara 4.15
   */
  @Override
  protected ErrorType writeProperty(int pId, int ndx, byte[] val, int pri, BBacnetAddress sourceAddress)
    throws BacnetException
  {
    BStringWritable pt = (BStringWritable)getPoint();
    if (pt == null)
    {
      return new NErrorType(BBacnetErrorClass.OBJECT,
        BBacnetErrorCode.TARGET_NOT_CONFIGURED);
    }

    // Check for array index on non-array property.
    if (ndx >= 0)
    {
      if (!isArray(pId))
      {
        return new NErrorType(BBacnetErrorClass.PROPERTY,
          BBacnetErrorCode.PROPERTY_IS_NOT_AN_ARRAY);
      }
    }

    try
    {
      switch (pId)
      {
        case BBacnetPropertyIdentifier.PRESENT_VALUE:
          return writePriorityArray(pri, val, sourceAddress);

        case BBacnetPropertyIdentifier.VALUE_SOURCE:
          return writeValueSource(pri, val, sourceAddress);

        case BBacnetPropertyIdentifier.CURRENT_COMMAND_PRIORITY:
        case BBacnetPropertyIdentifier.VALUE_SOURCE_ARRAY:
        case BBacnetPropertyIdentifier.LAST_COMMAND_TIME:
        case BBacnetPropertyIdentifier.COMMAND_TIME_ARRAY:
          return new NErrorType(property, writeAccessDenied);

        default:
          return super.writeProperty(pId, ndx, val, pri, sourceAddress);
      }
    }
    catch (AsnException e)
    {
      log.warning("AsnException writing property " + pId + " in object " + getObjectId() + ": "  + e);
      return new NErrorType(BBacnetErrorClass.PROPERTY,
        BBacnetErrorCode.INVALID_DATA_TYPE);
    }
    catch (PermissionException e)
    {
      log.warning("PermissionException writing property " + pId + " in object " + getObjectId() + ": "  + e);
      return new NErrorType(BBacnetErrorClass.PROPERTY,
        BBacnetErrorCode.WRITE_ACCESS_DENIED);
    }
  }

  @Override
  protected ErrorType writeOptionalProperty(int pId, int ndx, byte[] val, int pri)
    throws BacnetException
  {
    switch (pId)
    {
      case BBacnetPropertyIdentifier.ALARM_VALUES:
        if (getAlarmExt() != null)
        {
          return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.WRITE_ACCESS_DENIED);
        }
        else
        {
          return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.UNKNOWN_PROPERTY);
        }
    }

    return super.writeOptionalProperty(pId, ndx, val, pri);
  }

  /**
   * Subclass override method to add required properties.
   * NOTE: You MUST call super.addRequiredProps(v) first!
   *
   * @param v Vector containing required propertyIds.
   */
  @Override
  @SuppressWarnings({"rawtypes", "unchecked"})
  protected void addRequiredProps(Vector v)
  {
    v.add(BBacnetPropertyIdentifier.presentValue);
    v.add(BBacnetPropertyIdentifier.statusFlags);
  }

  /**
   * Subclass override method to add optional properties.
   * NOTE: You MUST call super.addOptionalProps(v) first!
   *
   * @param v Vector containing optional propertyIds.
   */
  @Override
  @SuppressWarnings({"rawtypes", "unchecked"})
  protected void addOptionalProps(Vector v)
  {
    v.add(BBacnetPropertyIdentifier.description);
    v.add(BBacnetPropertyIdentifier.eventState);
    v.add(BBacnetPropertyIdentifier.reliability);
    v.add(BBacnetPropertyIdentifier.outOfService);
    v.add(BBacnetPropertyIdentifier.priorityArray);
    v.add(BBacnetPropertyIdentifier.relinquishDefault);
    v.add(BBacnetPropertyIdentifier.currentCommandPriority);
    v.add(BBacnetPropertyIdentifier.valueSource);
    v.add(BBacnetPropertyIdentifier.valueSourceArray);
    v.add(BBacnetPropertyIdentifier.lastCommandTime);
    v.add(BBacnetPropertyIdentifier.commandTimeArray);

    BAlarmSourceExt alarmExt = getAlarmExt();
    if (alarmExt != null)
    {
      v.add(BBacnetPropertyIdentifier.timeDelay);
      v.add(BBacnetPropertyIdentifier.notificationClass);
      v.add(BBacnetPropertyIdentifier.alarmValues);
      v.add(BBacnetPropertyIdentifier.eventEnable);
      v.add(BBacnetPropertyIdentifier.ackedTransitions);
      v.add(BBacnetPropertyIdentifier.notifyType);
      v.add(BBacnetPropertyIdentifier.eventTimeStamps);
      v.add(BBacnetPropertyIdentifier.eventMessageTexts);
      v.add(BBacnetPropertyIdentifier.eventMessageTextsConfig);
      v.add(BBacnetPropertyIdentifier.eventDetectionEnable);
      v.add(BBacnetPropertyIdentifier.eventAlgorithmInhibitRef);
      v.add(BBacnetPropertyIdentifier.eventAlgorithmInhibit);
      v.add(BBacnetPropertyIdentifier.timeDelayNormal);
    }
  }

  /**
   * Override point for subclasses to validate their exposed point's
   * current state.  Default implementation does nothing.  Some points may
   * set the BACnet status flags to fault if the Niagara value is disallowed
   * for the exposed BACnet object type.
   */
  @Override
  protected void validate()
  {
    // Do not clear the status, faultCause, or reliability for a configuration fault, which prevents
    // interacting with this object anyway.
    if (!configOk.get())
    {
      return;
    }

    if (getOosExt().getOutOfService())
    {
      // Clear any internal faults.
      setFaultCause("");
      setStatus(BStatus.makeFault(getStatus(), false));
      setReliability(noFaultDetected);
      return;
    }

    BStatusString sn = ((BStringPoint)getPoint()).getOut();
    BStatus s = sn.getStatus();
    if (s.isNull())
    {
      setFaultCause("Invalid value for BACnet Object: " + sn);
      setStatus(BStatus.makeFault(getStatus(), true));
      // TODO Add support for internal reliability evaluation
      //setReliability(BBacnetReliability.unreliableOther);
      setReliability(noFaultDetected);
      return;
    }

    // The out.status is not null.
    setFaultCause("");
    setStatus(BStatus.makeFault(getStatus(), false));

    // TODO Add support for internal reliability evaluation
    //if (s.isDown())
    //{
    //  setReliability(BBacnetReliability.communicationFailure);
    //  return;
    //}
    //
    //if (s.isFault())
    //{
    //  setReliability(BBacnetReliability.unreliableOther);
    //  return;
    //}
    //
    setReliability(noFaultDetected);
  }

  @Override
  protected int getPresentValueAsnType()
  {
    return ASN_CHARACTER_STRING;
  }

  @Override
  protected BValue readAsnPriorityValue(AsnInput in, BControlPoint point)
    throws AsnException
  {
    return BString.make(in.readCharacterString());
  }

  @Override
  protected boolean isPriorityValueOutOfRange(BValue value, BControlPoint point)
  {
    return false;
  }

  /**
   * Get the current statusValue to use in checking for COVs.
   * Subclasses must override this to return the correct statusValue,
   * taking into account the value of outOfService, and using the
   * getStatusFlags() method to incorporate the appropriate status
   * information to report to BACnet.
   */
  @Override
  BStatusValue getCurrentStatusValue()
  {
    BStatusValue sv = new BStatusString(((BStringPoint)getPoint()).getOut().getValue());
    sv.setStatus(getStatusFlags());
    return sv;
  }

  /**
   * Check to see if the current value requires a COV notification.
   *
   * @return true if the currentValue is different than the cov value, or the status bits have changed
   */
  @Override
  public boolean checkCov(BStatusValue currentValue, BStatusValue covValue)
  {
    return !currentValue.toString().equals(covValue.toString());
  }

  @Override
  protected void writeAsnPriorityValue(AsnOutput out, BValue value)
  {
    out.writeCharacterString(((BString) value).getString());
  }

  @Override
  protected byte[] convertToAsn(BValue value)
  {
    return AsnUtil.toAsnCharacterString(((BString) value).getString());
  }

  private void resetBacnetWritable()
  {
    StringBuilder sb = new StringBuilder();
    Knob[] knobs = getKnobs();
    for (int i = 0; i < knobs.length; i++)
    {
      BObject tgt = knobs[i].getTargetOrd().get(this);
      BObject pt = getPoint();
      if (knobs[i].getTargetSlotName().startsWith("in") &&
          tgt == pt)
      {
        sb.append(knobs[i].getTargetSlotName()).append(',');
      }
    }
    setBacnetWritable((sb.length() > 0) ? sb.substring(0, sb.length() - 1) :
                                          lexNotWritable);
  }

  @Override
  public void knobAdded(Knob knob, Context cx)
  {
    resetBacnetWritable();
  }

  @Override
  public void knobRemoved(Knob knob, Context cx)
  {
    resetBacnetWritable();
  }

////////////////////////////////////////////////////////////////
// Presentation
////////////////////////////////////////////////////////////////

  @Override
  public BIcon getIcon()
  {
    return icon;
  }

  private static final BIcon icon = BIcon.make(BIcon.std("control/stringPoint.png"), BIcon.std("badges/export.png"));
}
