/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.export;

import java.util.logging.Level;

import javax.baja.bacnet.BacnetConst;
import javax.baja.bacnet.BacnetException;
import javax.baja.bacnet.datatypes.BBacnetCovSubscription;
import javax.baja.bacnet.datatypes.BBacnetDate;
import javax.baja.bacnet.datatypes.BBacnetDateTime;
import javax.baja.bacnet.datatypes.BBacnetTime;
import javax.baja.bacnet.enums.BBacnetErrorClass;
import javax.baja.bacnet.enums.BBacnetErrorCode;
import javax.baja.bacnet.enums.BBacnetPolarity;
import javax.baja.bacnet.enums.BBacnetPropertyIdentifier;
import javax.baja.bacnet.enums.BBacnetReliability;
import javax.baja.bacnet.io.AsnException;
import javax.baja.bacnet.io.ErrorType;
import javax.baja.bacnet.io.OutOfRangeException;
import javax.baja.bacnet.io.PropertyValue;
import javax.baja.control.BBooleanPoint;
import javax.baja.control.BControlPoint;
import javax.baja.control.ext.BAbstractProxyExt;
import javax.baja.control.ext.BDiscreteTotalizerExt;
import javax.baja.driver.point.BDefaultProxyConversion;
import javax.baja.driver.point.BProxyExt;
import javax.baja.driver.point.conv.BReversePolarityConversion;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.security.PermissionException;
import javax.baja.status.BStatus;
import javax.baja.status.BStatusBoolean;
import javax.baja.status.BStatusValue;
import javax.baja.sys.BFacets;
import javax.baja.sys.BIBoolean;
import javax.baja.sys.BIcon;
import javax.baja.sys.BRelTime;
import javax.baja.sys.BString;
import javax.baja.sys.BValue;
import javax.baja.sys.Property;
import javax.baja.sys.SlotCursor;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.BFormat;

import com.tridium.bacnet.asn.AsnUtil;
import com.tridium.bacnet.asn.NErrorType;
import com.tridium.bacnet.asn.NReadPropertyResult;

/**
 * BBacnetBinaryPointDescriptor is the superclass for binary-type
 * point extensions exposing BooleanPoints to Bacnet.
 *
 * @author Craig Gemmill on 19 Feb 02
 * @since Niagara 3 Bacnet 1.0
 */
@NiagaraType
abstract public class BBacnetBinaryPointDescriptor
  extends BBacnetPointDescriptor
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.export.BBacnetBinaryPointDescriptor(2979906276)1.0$ @*/
/* Generated Thu Dec 16 19:44:32 CST 2021 by Slot-o-Matic (c) Tridium, Inc. 2012-2021 */

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetBinaryPointDescriptor.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// Overrides
////////////////////////////////////////////////////////////////

  /**
   * BBacnetBinaryPointDescriptor may only expose BBooleanPoint.
   *
   * @param pt the exposed point
   * @return true if the Niagara point type is legal for this point type.
   */
  @Override
  protected boolean isPointTypeLegal(BControlPoint pt)
  {
    return pt instanceof BBooleanPoint;
  }

  @Override
  protected int getPresentValueAsnType()
  {
    return ASN_ENUMERATED;
  }

  @Override
  protected byte[] convertToAsn(BValue value)
  {
    return AsnUtil.toAsnEnumerated(((BIBoolean) value).getBoolean() ? 1 : 0);
  }

  /**
   * Read the value of an optional property.
   * Subclasses with additional properties override this to check for
   * their properties.  If no match is found, call this superclass
   * method to check these properties.
   *
   * @param pId the requested property-identifier.
   * @param ndx the property array index (-1 if not specified).
   * @return a PropertyValue containing either the encoded value or the error.
   */
  @Override
  protected PropertyValue readOptionalProperty(int pId, int ndx)
  {
    BDiscreteTotalizerExt totExt = getTotalizerExt();
    if (totExt != null)
    {
      switch (pId)
      {
        case BBacnetPropertyIdentifier.CHANGE_OF_STATE_TIME:
          return new NReadPropertyResult(pId, ndx, AsnUtil.toBacnetDateTime(totExt.getChangeOfStateTime()));

        case BBacnetPropertyIdentifier.CHANGE_OF_STATE_COUNT:
          return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnUnsigned(totExt.getChangeOfStateCount()));

        case BBacnetPropertyIdentifier.TIME_OF_STATE_COUNT_RESET:
          return new NReadPropertyResult(pId, ndx, AsnUtil.toBacnetDateTime(totExt.getTimeOfStateCountReset()));

        case BBacnetPropertyIdentifier.ELAPSED_ACTIVE_TIME:
          return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnUnsigned(totExt.getElapsedActiveTime().getSeconds()));

        case BBacnetPropertyIdentifier.TIME_OF_ACTIVE_TIME_RESET:
          return new NReadPropertyResult(pId, ndx, AsnUtil.toBacnetDateTime(totExt.getTimeOfActiveTimeReset()));
      }
    }
    BBooleanPoint pt = (BBooleanPoint)getPoint();
    switch (pId)
    {
      case BBacnetPropertyIdentifier.ACTIVE_TEXT:
        BString tt = (BString)pt.getFacets().getFacet(BFacets.TRUE_TEXT);
        if (tt != null)
        {
          String trueText = BFormat.format(tt.toString(), null, null);
          return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnCharacterString(trueText));
        }
        break;

      case BBacnetPropertyIdentifier.INACTIVE_TEXT:
        BString ft = (BString)pt.getFacets().getFacet(BFacets.FALSE_TEXT);
        if (ft != null)
        {
          String falseText = BFormat.format(ft.toString(), null, null);
          return new NReadPropertyResult(pId, ndx, AsnUtil.toAsnCharacterString(falseText));
        }
    }

    return super.readOptionalProperty(pId, ndx);
  }

  /**
   * Encode the proxyValue boolean value to an ASN Enumerated BBacnetBinaryPv.
   * @since Niagara 4.14
   */
  @Override
  protected byte[] makeInterfaceValue(BStatusValue proxyValue)
  {
    return AsnUtil.toAsnEnumerated(((BStatusBoolean)proxyValue).getValue());
  }

  /**
   * Set the value of an optional property.
   * Subclasses with additional properties override this to check for
   * their properties.  If no match is found, call this superclass
   * method to check these properties.
   *
   * @param pId the requested property-identifier.
   * @param ndx the property array index (-1 if not specified).
   * @param val the Asn-encoded value for the property.
   * @param pri the priority level (only used for commandable properties).
   * @return null if everything goes OK, or
   * an ErrorType describing the error if not.
   */
  @Override
  protected ErrorType writeOptionalProperty(int pId,
                                            int ndx,
                                            byte[] val,
                                            int pri)
    throws BacnetException
  {
    BDiscreteTotalizerExt totExt = getTotalizerExt();
    try
    {
      if (totExt != null)
      {
        switch (pId)
        {
          case BBacnetPropertyIdentifier.CHANGE_OF_STATE_TIME:
            return new NErrorType(BBacnetErrorClass.PROPERTY,
                                  BBacnetErrorCode.WRITE_ACCESS_DENIED);

          case BBacnetPropertyIdentifier.CHANGE_OF_STATE_COUNT:
            long changeOfStateCount = AsnUtil.fromAsnUnsignedInteger(val);
            if (changeOfStateCount == 0)
            {
              totExt.invoke(BDiscreteTotalizerExt.resetChangeOfStateCount, null, BLocalBacnetDevice.getBacnetContext());
            }
            else if (changeOfStateCount > Integer.MAX_VALUE)
            {
              return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE);
            }
            else
            {
              // Since protocol 16, supports setting to a non-zero value.
              totExt.setInt(BDiscreteTotalizerExt.changeOfStateCount, (int) changeOfStateCount, BLocalBacnetDevice.getBacnetContext());
            }
            return null;

          case BBacnetPropertyIdentifier.TIME_OF_STATE_COUNT_RESET:
            BBacnetDateTime timeOfStateCountReset = new BBacnetDateTime();
            AsnUtil.fromAsn(BacnetConst.ASN_ANY, val, timeOfStateCountReset);
            checkDateTime(timeOfStateCountReset);
            totExt.set(BDiscreteTotalizerExt.timeOfStateCountReset, timeOfStateCountReset.toBAbsTime(), BLocalBacnetDevice.getBacnetContext());
            return null;

          case BBacnetPropertyIdentifier.ELAPSED_ACTIVE_TIME:
            long elapsedActiveTime = AsnUtil.fromAsnUnsignedInteger(val);
            if (elapsedActiveTime == 0)
            {
              totExt.invoke(BDiscreteTotalizerExt.resetElapsedActiveTime, null, BLocalBacnetDevice.getBacnetContext());
            }
            else if (elapsedActiveTime > Integer.MAX_VALUE)
            {
              // Elapsed Active Time property is of type Unsigned32, but framework supports till Unsigned31.
              return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE);
            }
            else
            {
              // Since protocol 16, supports setting to a non-zero value.
              totExt.set(BDiscreteTotalizerExt.elapsedActiveTime, BRelTime.makeSeconds((int) elapsedActiveTime), BLocalBacnetDevice.getBacnetContext());
            }
            return null;

          case BBacnetPropertyIdentifier.TIME_OF_ACTIVE_TIME_RESET:
            BBacnetDateTime timeOfActiveTimeReset = new BBacnetDateTime();
            AsnUtil.fromAsn(BacnetConst.ASN_ANY, val, timeOfActiveTimeReset);
            checkDateTime(timeOfActiveTimeReset);
            totExt.set(BDiscreteTotalizerExt.timeOfActiveTimeReset, timeOfActiveTimeReset.toBAbsTime(), BLocalBacnetDevice.getBacnetContext());
            return null;
        }
      }
      BBooleanPoint pt = (BBooleanPoint)getPoint();
      switch (pId)
      {
        case BBacnetPropertyIdentifier.ACTIVE_TEXT:
          BString tt = (BString)pt.getFacets().getFacet(BFacets.TRUE_TEXT);
          if (tt != null)
          {
            pt.set(BControlPoint.facets,
                   BFacets.make(pt.getFacets(), BFacets.TRUE_TEXT, BString.make(AsnUtil.fromAsnCharacterString(val))),
                   BLocalBacnetDevice.getBacnetContext());
            return null;
          }
          break;

        case BBacnetPropertyIdentifier.INACTIVE_TEXT:
          BString ft = (BString)pt.getFacets().getFacet(BFacets.FALSE_TEXT);
          if (ft != null)
          {
            pt.set(BControlPoint.facets,
                   BFacets.make(pt.getFacets(), BFacets.FALSE_TEXT, BString.make(AsnUtil.fromAsnCharacterString(val))),
                   BLocalBacnetDevice.getBacnetContext());
            return null;
          }
      }

      return super.writeOptionalProperty(pId, ndx, val, pri);
    }
    catch (OutOfRangeException e)
    {
      log.warning("OutOfRangeException writing property " + pId + " in object " + getObjectId() + ": " + e);
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.VALUE_OUT_OF_RANGE);
    }
    catch (AsnException e)
    {
      log.warning("AsnException writing property " + pId + " in object " + getObjectId() + ": " + e);
      return new NErrorType(BBacnetErrorClass.PROPERTY,
                            BBacnetErrorCode.INVALID_DATA_TYPE);
    }
    catch (PermissionException e)
    {
      log.warning("PermissionException writing property " + pId + " in object " + getObjectId() + ": " + e);
      return new NErrorType(BBacnetErrorClass.PROPERTY,
                            BBacnetErrorCode.WRITE_ACCESS_DENIED);
    }
  }

  private static void checkDateTime(BBacnetDateTime dateTime)
    throws OutOfRangeException
  {
    BBacnetDate date = dateTime.getDate();
    if (date.isYearUnspecified() ||
        date.isMonthUnspecified() ||
        date.isMonthSpecial() ||
        date.isDayOfMonthUnspecified() ||
        date.isDayOfMonthSpecial())
    {
      throw new OutOfRangeException("Date contains unspecified or special values for the year, month, or day-of-month: " + dateTime);
    }

    BBacnetTime time = dateTime.getTime();
    if (time.isHourUnspecified() ||
        time.isMinuteUnspecified())
    {
      throw new OutOfRangeException("Time contains unspecified values for the hour or minute: " + dateTime);
    }
  }

  /**
   * Override point for subclasses to validate their exposed point's
   * current state.  Default implementation does nothing.  Some points may
   * set the BACnet status flags to fault if the Niagara value is disallowed
   * for the exposed BACnet object type.
   */
  @Override
  protected void validate()
  {
    // Do not clear the status, faultCause, or reliability for a configuration fault, which prevents
    // interacting with this object anyway.
    if (!configOk.get())
    {
      return;
    }

    if (getOosExt().getOutOfService())
    {
      // Clear any internal faults.
      setFaultCause("");
      setStatus(BStatus.makeFault(getStatus(), false));
      setReliability(BBacnetReliability.noFaultDetected);
      return;
    }

    BStatusBoolean sb = ((BBooleanPoint)getPoint()).getOut();
    BStatus s = sb.getStatus();
    if (s.isNull())
    {
      setFaultCause("Invalid value for BACnet Object: " + sb);
      setStatus(BStatus.makeFault(getStatus(), true));
      // TODO Add support for internal reliability evaluation
      //setReliability(BBacnetReliability.unreliableOther);
      setReliability(BBacnetReliability.noFaultDetected);
      return;
    }

    // The out.status is not null.
    setFaultCause("");
    setStatus(BStatus.makeFault(getStatus(), false));

    // TODO Add support for internal reliability evaluation
    //if (s.isDown())
    //{
    //  setReliability(BBacnetReliability.communicationFailure);
    //  return;
    //}
    //
    //if (s.isFault())
    //{
    //  setReliability(BBacnetReliability.unreliableOther);
    //  return;
    //}

    setReliability(BBacnetReliability.noFaultDetected);
  }

  /**
   * Get the current statusValue to use in checking for COVs.
   * Subclasses must override this to return the correct statusValue,
   * taking into account the value of outOfService, and using the
   * getStatusFlags() method to incorporate the appropriate status
   * information to report to BACnet.
   */
  @Override
  BStatusValue getCurrentStatusValue()
  {
    BStatusValue sv = new BStatusBoolean(((BBooleanPoint)getPoint()).getOut().getValue());
    sv.setStatus(this.getStatusFlags());
    return sv;
  }

  /**
   * Check to see if the current value requires a COV notification.
   */
  @Override
  boolean checkCov(BStatusValue currentValue, BStatusValue covValue)
  {
    if (currentValue.getStatus().getBits() != covValue.getStatus().getBits())
    {
      return true;
    }

    return ((BStatusBoolean)currentValue).getBoolean() != ((BStatusBoolean)covValue).getBoolean();
  }

  /**
   * Check for Cov notification.
   * Binary points check if the point's current value is different
   * than the last Cov value.
   *
   * @deprecated
   */
  @Deprecated
  boolean checkCov(BControlPoint pt, BBacnetCovSubscription covSub)
  {
    if (pt.getStatus().getBits() != covSub.getLastValue().getStatus().getBits())
    {
      return true;
    }

    boolean currentValue = ((BBooleanPoint)pt).getBoolean();
    boolean covValue = ((BIBoolean)covSub.getLastValue()).getBoolean();
    return currentValue != covValue;
  }

  protected BDiscreteTotalizerExt getTotalizerExt()
  {
    BControlPoint pt = getPoint();
    if (pt == null)
    {
      return null;
    }

    SlotCursor<Property> c = pt.getProperties();
    if (c.next(BDiscreteTotalizerExt.class))
    {
      return (BDiscreteTotalizerExt) c.get();
    }

    return null;
  }

  PropertyValue readPolarityProperty(BBooleanPoint pt)
  {
    BAbstractProxyExt proxyExt = pt.getProxyExt();
    if (proxyExt instanceof BProxyExt &&
        ((BProxyExt) proxyExt).getConversion() instanceof BReversePolarityConversion)
    {
      return new NReadPropertyResult(BBacnetPropertyIdentifier.POLARITY, AsnUtil.toAsnEnumerated(BBacnetPolarity.REVERSE));
    }

    return new NReadPropertyResult(BBacnetPropertyIdentifier.POLARITY, AsnUtil.toAsnEnumerated(BBacnetPolarity.NORMAL));
  }

  protected ErrorType writePolarityProperty(BBooleanPoint pt, byte[] val)
    throws BacnetException
  {
    BAbstractProxyExt proxyExt = pt.getProxyExt();
    if (proxyExt instanceof BProxyExt)
    {
      if (AsnUtil.fromAsnEnumerated(val) == BBacnetPolarity.REVERSE)
      {
        ((BProxyExt) proxyExt).setConversion(BReversePolarityConversion.DEFAULT);
      }
      else
      {
        ((BProxyExt) proxyExt).setConversion(BDefaultProxyConversion.DEFAULT);
      }
      return null;
    }
    else
    {
      if (log.isLoggable(Level.FINE))
      {
        log.fine("Cannot write the Polarity property when the associated point's proxy ext is not" +
          " instanceof BProxyExt; object ID: " + getObjectId() + ", object name: " + getObjectName());
      }
      return new NErrorType(BBacnetErrorClass.PROPERTY, BBacnetErrorCode.WRITE_ACCESS_DENIED);
    }
  }

////////////////////////////////////////////////////////////////
// Presentation
////////////////////////////////////////////////////////////////

  @Override
  public BIcon getIcon()
  {
    return icon;
  }

  private static final BIcon icon = BIcon.make(BIcon.std("control/booleanPoint.png"), BIcon.std("badges/export.png"));
}
