/*
 * Copyright 2023 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.datatypes;

import static com.tridium.bacnet.stack.link.ip.util.BacnetIpLinkUtil.containsDotsAndDigitsOnly;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.StringTokenizer;

import javax.baja.bacnet.io.AsnException;
import javax.baja.bacnet.io.AsnInput;
import javax.baja.bacnet.io.AsnOutput;
import javax.baja.nre.annotations.Facet;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.sys.BFacets;
import javax.baja.sys.BStruct;
import javax.baja.sys.Context;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.bacnet.stack.link.ip.util.BacnetIpLinkUtil;

/**
 * Represents the BACnetHostNPort sequence, which contains a host and port.
 *
 * @author Uday Rapuru on 10-Oct-2023
 * @since Niagara 4.15
 */
@NiagaraType
@NiagaraProperty(
  name = "hostAddress",
  type = "BBacnetHostAddress",
  defaultValue = "new BBacnetHostAddress()"
)
@NiagaraProperty(
  name = "port",
  type = "int",
  defaultValue = "0",
  facets = @Facet("BFacets.makeInt(null, (int)BBacnetUnsigned.MIN_UNSIGNED_VALUE, (int)BBacnetUnsigned.MAX_UNSIGNED16_VALUE)")
)
public final class BBacnetHostNPort
  extends BStruct
  implements BIBacnetDataType
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.datatypes.BBacnetHostNPort(1283617051)1.0$ @*/
/* Generated Fri Oct 13 14:13:04 CDT 2023 by Slot-o-Matic (c) Tridium, Inc. 2012-2023 */

  //region Property "hostAddress"

  /**
   * Slot for the {@code hostAddress} property.
   * @see #getHostAddress
   * @see #setHostAddress
   */
  @Generated
  public static final Property hostAddress = newProperty(0, new BBacnetHostAddress(), null);

  /**
   * Get the {@code hostAddress} property.
   * @see #hostAddress
   */
  @Generated
  public BBacnetHostAddress getHostAddress() { return (BBacnetHostAddress)get(hostAddress); }

  /**
   * Set the {@code hostAddress} property.
   * @see #hostAddress
   */
  @Generated
  public void setHostAddress(BBacnetHostAddress v) { set(hostAddress, v, null); }

  //endregion Property "hostAddress"

  //region Property "port"

  /**
   * Slot for the {@code port} property.
   * @see #getPort
   * @see #setPort
   */
  @Generated
  public static final Property port = newProperty(0, 0, BFacets.makeInt(null, (int)BBacnetUnsigned.MIN_UNSIGNED_VALUE, (int)BBacnetUnsigned.MAX_UNSIGNED16_VALUE));

  /**
   * Get the {@code port} property.
   * @see #port
   */
  @Generated
  public int getPort() { return getInt(port); }

  /**
   * Set the {@code port} property.
   * @see #port
   */
  @Generated
  public void setPort(int v) { setInt(port, v, null); }

  //endregion Property "port"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetHostNPort.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  public BBacnetHostNPort()
  {
  }

  public BBacnetHostNPort(BBacnetHostAddress hostAddress, int port)
  {
    setHostAddress(hostAddress);
    setPort(port);
  }

  public static BBacnetHostNPort makeWithBacnetIpBytes(byte[] bacnetIpBytes)
  {
    BacnetIpLinkUtil.validateAddressLength(bacnetIpBytes, 6);
    return new BBacnetHostNPort(
      BBacnetHostAddress.makeWithIpAddress(BBacnetOctetString.make(Arrays.copyOf(bacnetIpBytes, 4))),
      BacnetIpLinkUtil.getPortFromBacnetIpBytes(bacnetIpBytes));
  }

  public static BBacnetHostNPort parseBacnetIpAddress(String string)
  {
    if (string == null || string.isEmpty() || string.equalsIgnoreCase("null"))
    {
      // none
      return new BBacnetHostNPort();
    }

    if (string.contains("."))
    {
      String[] hostNPort = string.split(":");
      if (hostNPort.length != 2)
      {
        throw new IllegalArgumentException("String that contains dots must have host and port separated by a colon; string: " + string);
      }

      int port = Integer.decode(hostNPort[1]);
      if (port < 0 || port > 65535)
      {
        throw new IllegalArgumentException("Port value of a string that contains dots must be between 0 and 65535 inclusive; string: " + string);
      }

      // IPAddressUtil.isIpv4Address will correctly return false for an address missing values
      // such as "10.16.36". However, InetAddress.getByName will interpret the last number (36) as
      // a 16-bit value and return the byte[] { 10, 16, 0, 36 }. Similar problems will occur with
      // two and one part addresses.  IPAddressUtil.isHostname will return true even if all
      // elements are numbers. Here, if all the elements are numbers, there must be 4 of them and
      // each must be 0-255.
      if (containsDotsAndDigitsOnly(hostNPort[0]))
      {
        String[] ipOctets = hostNPort[0].split("\\.");
        if (ipOctets.length != 4)
        {
          throw new IllegalArgumentException("When string contains digits and dots only, it must contain 4 parts; string: " + string);
        }

        byte[] ipBytes = new byte[4];
        for (int i = 0; i < 4; i++)
        {
          int octet = Integer.decode(ipOctets[i]);
          if (octet < 0 || octet > 255)
          {
            throw new IllegalArgumentException("When string contains digits and dots only, numbers must be between 0 and 255 inclusive; string: " + string);
          }
          ipBytes[i] = (byte)octet;
        }

        // ipAddress
        return new BBacnetHostNPort(BBacnetHostAddress.makeWithIpAddress(ipBytes), port);
      }
      else
      {
        // name
        return new BBacnetHostNPort(BBacnetHostAddress.makeWithHostName(hostNPort[0]), port);
      }
    }
    else
    {
      // Contains no dots
      StringTokenizer tokenizer = new StringTokenizer(string, " :");
      if (tokenizer.countTokens() < 6)
      {
        throw new IllegalArgumentException("When string contains no dots, it must contain 6 parts; string: " + string);
      }

      byte[] bacnetIpBytes = new byte[6];
      for (int i = 0; i < 6; i++)
      {
        int value = Integer.parseInt(tokenizer.nextToken(), 16);
        if (value < 0 || value > 255)
        {
          throw new IllegalArgumentException("When string contains no dots, numbers must be between 0 and 255 inclusive; string: " + string);
        }
        bacnetIpBytes[i] = (byte)value;
      }

      // ipAddress
      return makeWithBacnetIpBytes(bacnetIpBytes);
    }
  }

  /**
   * Returns a byte array that concatenates the host IP address and port. Returns null if the host
   * is not an IPv4 address.
   */
  public byte[] getBacnetIpAddressBytes()
  {
    byte[] hostAddressBytes;
    switch (getHostAddress().getChoice().getOrdinal())
    {
      case BBacnetHostAddressChoice.NAME:
        try
        {
          hostAddressBytes = InetAddress.getByName(getHostAddress().getHostName()).getAddress();
        }
        catch (UnknownHostException e)
        {
          return null;
        }
        break;
      case BBacnetHostAddressChoice.IP_ADDRESS:
        hostAddressBytes = getHostAddress().getIpAddress().getBytes();
        break;
      default:
        return null;
    }

    if (hostAddressBytes == null || hostAddressBytes.length != 4)
    {
      return null;
    }

    byte[] macBytes = new byte[6];
    System.arraycopy(hostAddressBytes, 0, macBytes, 0, 4);

    int port = getPort();
    if (port < 0 || port > 65535)
    {
      return null;
    }
    macBytes[4] = (byte)((port >> 8) & 0xFF);
    macBytes[5] = (byte)(port & 0xFF);
    return macBytes;
  }

  /**
   * Read the value from the Asn input stream.
   */
  @Override
  public void readAsn(AsnInput in)
    throws AsnException
  {
    in.skipOpeningTag(0);
    BBacnetHostAddress hostAddress = new BBacnetHostAddress();
    hostAddress.readAsn(in);
    in.skipClosingTag(0);
    int port = in.readUnsignedInt(1);

    set(BBacnetHostNPort.hostAddress, hostAddress, noWrite);
    setInt(BBacnetHostNPort.port, port, noWrite);
  }

  /**
   * Write the value to the Asn output stream.
   */
  @Override
  public void writeAsn(AsnOutput out)
  {
    out.writeOpeningTag(0);
    getHostAddress().writeAsn(out);
    out.writeClosingTag(0);
    out.writeUnsignedInteger(1, getPort());
  }

  @Override
  public String toString(Context context)
  {
    return getHostAddress().toString(context) + ":" + getPort();
  }
}
