/*
 * Copyright 2024 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.datatypes;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Supplier;

import javax.baja.bacnet.io.AsnException;
import javax.baja.bacnet.io.AsnInput;
import javax.baja.bacnet.io.AsnOutput;
import javax.baja.nre.annotations.Facet;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BComponent;
import javax.baja.sys.BFacets;
import javax.baja.sys.BRelTime;
import javax.baja.sys.Context;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.units.BUnit;

/**
 * Represents the BACnetCOVMultipleSubscription sequence that is used by the
 * SubscribeCOVPropertyMultiple service to subscribe for the receipt of notifications of changes
 * that may occur to multiple properties of multiple objects.
 * <p>
 * BACnetCOVMultipleSubscription ::= SEQUENCE {
 *     recipient                              [0] BACnetRecipientProcess,
 *     issueConfirmedNotifications            [1] BOOLEAN,
 *     timeRemaining                          [2] Unsigned,
 *     maxNotificationDelay                   [3] Unsigned,
 *     listOfCOVSubscriptionSpecifications    [4] SEQUENCE OF SEQUENCE {
 *         monitoredObject                        [0] BACnetObjectIdentifier,
 *         listOfCOVReferences                    [1] SEQUENCE OF SEQUENCE {
 *             monitoredProperty                      [0] BACnetPropertyReference,
 *             covIncrement                           [1] REAL OPTIONAL,
 *             timestamped                            [2] BOOLEAN
 *             }
 *         }
 *     }
 *
 *
 * @author Tim Urenda on Oct 18, 2024
 * @since Niagara 4.15
 */
@NiagaraType
@NiagaraProperty(
  name = "recipient",
  type = "BBacnetRecipientProcess",
  defaultValue = "new BBacnetRecipientProcess()"
)
@NiagaraProperty(
  name = "issueConfirmedNotifications",
  type = "boolean",
  defaultValue = "false"
)
/*
 The time at which this subscriber's subscription will end.
 This is an estimate; the actual timing is handled through an internal timer.
 */
@NiagaraProperty(
  name = "subscriptionEndTime",
  type = "BAbsTime",
  defaultValue = "BAbsTime.NULL",
  facets = {
    @Facet(name = "BFacets.SHOW_SECONDS", value = "true"),
    @Facet(name = "BFacets.SHOW_MILLISECONDS", value = "true")
  }
)
@NiagaraProperty(
  name = "maxNotificationDelay",
  type = "BBacnetUnsigned",
  defaultValue = "BBacnetUnsigned.DEFAULT",
  facets = @Facet("BFacets.makeInt(BUnit.getUnit(\"second\"), 0, 3600)")
)
public class BBacnetCovMultipleSubscription
  extends BComponent
  implements BIBacnetDataType
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.datatypes.BBacnetCovMultipleSubscription(1738509668)1.0$ @*/
/* Generated Fri Oct 25 23:46:12 CDT 2024 by Slot-o-Matic (c) Tridium, Inc. 2012-2024 */

  //region Property "recipient"

  /**
   * Slot for the {@code recipient} property.
   * @see #getRecipient
   * @see #setRecipient
   */
  @Generated
  public static final Property recipient = newProperty(0, new BBacnetRecipientProcess(), null);

  /**
   * Get the {@code recipient} property.
   * @see #recipient
   */
  @Generated
  public BBacnetRecipientProcess getRecipient() { return (BBacnetRecipientProcess)get(recipient); }

  /**
   * Set the {@code recipient} property.
   * @see #recipient
   */
  @Generated
  public void setRecipient(BBacnetRecipientProcess v) { set(recipient, v, null); }

  //endregion Property "recipient"

  //region Property "issueConfirmedNotifications"

  /**
   * Slot for the {@code issueConfirmedNotifications} property.
   * @see #getIssueConfirmedNotifications
   * @see #setIssueConfirmedNotifications
   */
  @Generated
  public static final Property issueConfirmedNotifications = newProperty(0, false, null);

  /**
   * Get the {@code issueConfirmedNotifications} property.
   * @see #issueConfirmedNotifications
   */
  @Generated
  public boolean getIssueConfirmedNotifications() { return getBoolean(issueConfirmedNotifications); }

  /**
   * Set the {@code issueConfirmedNotifications} property.
   * @see #issueConfirmedNotifications
   */
  @Generated
  public void setIssueConfirmedNotifications(boolean v) { setBoolean(issueConfirmedNotifications, v, null); }

  //endregion Property "issueConfirmedNotifications"

  //region Property "subscriptionEndTime"

  /**
   * Slot for the {@code subscriptionEndTime} property.
   * The time at which this subscriber's subscription will end.
   * This is an estimate; the actual timing is handled through an internal timer.
   * @see #getSubscriptionEndTime
   * @see #setSubscriptionEndTime
   */
  @Generated
  public static final Property subscriptionEndTime = newProperty(0, BAbsTime.NULL, BFacets.make(BFacets.make(BFacets.SHOW_SECONDS, true), BFacets.make(BFacets.SHOW_MILLISECONDS, true)));

  /**
   * Get the {@code subscriptionEndTime} property.
   * The time at which this subscriber's subscription will end.
   * This is an estimate; the actual timing is handled through an internal timer.
   * @see #subscriptionEndTime
   */
  @Generated
  public BAbsTime getSubscriptionEndTime() { return (BAbsTime)get(subscriptionEndTime); }

  /**
   * Set the {@code subscriptionEndTime} property.
   * The time at which this subscriber's subscription will end.
   * This is an estimate; the actual timing is handled through an internal timer.
   * @see #subscriptionEndTime
   */
  @Generated
  public void setSubscriptionEndTime(BAbsTime v) { set(subscriptionEndTime, v, null); }

  //endregion Property "subscriptionEndTime"

  //region Property "maxNotificationDelay"

  /**
   * Slot for the {@code maxNotificationDelay} property.
   * @see #getMaxNotificationDelay
   * @see #setMaxNotificationDelay
   */
  @Generated
  public static final Property maxNotificationDelay = newProperty(0, BBacnetUnsigned.DEFAULT, BFacets.makeInt(BUnit.getUnit("second"), 0, 3600));

  /**
   * Get the {@code maxNotificationDelay} property.
   * @see #maxNotificationDelay
   */
  @Generated
  public BBacnetUnsigned getMaxNotificationDelay() { return (BBacnetUnsigned)get(maxNotificationDelay); }

  /**
   * Set the {@code maxNotificationDelay} property.
   * @see #maxNotificationDelay
   */
  @Generated
  public void setMaxNotificationDelay(BBacnetUnsigned v) { set(maxNotificationDelay, v, null); }

  //endregion Property "maxNotificationDelay"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetCovMultipleSubscription.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  public void writeAsn(AsnOutput out)
  {
    out.writeOpeningTag(0);
    getRecipient().writeAsn(out);
    out.writeClosingTag(0);
    out.writeBoolean(1, getIssueConfirmedNotifications());
    out.writeUnsignedInteger(2, getTimeRemaining());
    out.writeUnsigned(3, getMaxNotificationDelay());

    out.writeOpeningTag(4);
    for (BBacnetCovMultipleSubscriptionSpecification entry : getChildren(BBacnetCovMultipleSubscriptionSpecification.class))
    {
      entry.writeAsn(out);
    }
    out.writeClosingTag(4);
  }

  @Override
  public void readAsn(AsnInput in)
    throws AsnException
  {
    in.skipOpeningTag(0);
    BBacnetRecipientProcess recipient = new BBacnetRecipientProcess();
    recipient.readAsn(in);
    in.skipClosingTag(0);

    boolean issueConfirmedNotifications = in.readBoolean(1);
    int timeRemaining = in.readUnsignedInt(2);
    int maxNotificationDelay = in.readUnsignedInt(3);

    List<BBacnetCovMultipleSubscriptionSpecification> specifications = new ArrayList<>();
    in.skipOpeningTag(4);
    in.peekTag();
    while (!in.isClosingTag(4))
    {
      BBacnetCovMultipleSubscriptionSpecification specification = new BBacnetCovMultipleSubscriptionSpecification();
      specification.readAsn(in);
      specifications.add(specification);
      in.peekTag();
    }
    in.skipClosingTag(4);

    set(BBacnetCovMultipleSubscription.recipient, recipient, noWrite);
    setBoolean(BBacnetCovMultipleSubscription.issueConfirmedNotifications, issueConfirmedNotifications, noWrite);

    BAbsTime now = nowSupplier.get();
    BAbsTime subscriptionEndTime = now.add(BRelTime.make(timeRemaining * BRelTime.MILLIS_IN_SECOND));
    set(BBacnetCovMultipleSubscription.subscriptionEndTime, subscriptionEndTime, noWrite);
    set(BBacnetCovMultipleSubscription.maxNotificationDelay, BBacnetUnsigned.make(maxNotificationDelay), noWrite);

    removeAll(noWrite);
    for (int i = 0; i < specifications.size(); i++)
    {
      add("entry" + (i + 1), specifications.get(i), noWrite);
    }
  }

  @Override
  public String toString(Context context)
  {
    return getRecipient().toString(context) +
      (getIssueConfirmedNotifications() ? " C until " : " U until ") +
      getSubscriptionEndTime();
  }

  /*
   * Get the remaining life of this subscription in seconds.
   * This estimated calculation is done by determining the time
   * until the end time, which is set at the time when the termination
   * ticket is generated.
   *
   * @return the estimated remaining lifetime in seconds.
   */
  private int getTimeRemaining()
  {
    if (getSubscriptionEndTime().equals(BAbsTime.NULL))
    {
      return 0;
    }
    long now = nowSupplier.get().getMillis();
    int timeRemaining = (int)((getSubscriptionEndTime().getMillis() - now) / 1000);
    return timeRemaining > 0 ? timeRemaining : -1;
  }

  // Do not make final to allow test classes to override.
  @SuppressWarnings("FieldMayBeFinal")
  private static Supplier<BAbsTime> nowSupplier = BAbsTime::now;
}
