/*
Updates and diffing
===================

When a new value is loaded, spandrel will not just wipe the entire structure of
the widget and start over. Instead, it will apply incremental updates, only
applying what changes are needed to bring the widget up to date. This is a
similar approach to React's diffing process but spandrel does not use any kind
of virtual DOM - it works directly on Widget instances.

Point this example widget at a Ramp in your station. As it changes value,
inspect the element in the Chrome developer tools and observe that spandrel
continually updates the DOM structure to reflect the current values. Also, the
latest value will be loaded into the NumericEditor, and if you use another
browser window to change the Ramp's facets (such as the "units" facet) then
those facets changes will also be applied in real time.

What triggers the continual updates is the use of subscriberMixIn, which ensures
that the loaded Ramp gets subscribed, and attaching its "changed" event to a
call to this.rerender().

Point this example at a component of type kitControl:Ramp, such as
station:|slot:/SpandrelExamples/Components/Ramp.
*/

/** @jsx spandrel.jsx */

define([
  'bajaux/spandrel',
  'bajaux/mixin/subscriberMixIn',
  'nmodule/webEditors/rc/fe/baja/NumericEditor' ], function (
  spandrel,
  subscriberMixIn,
  NumericEditor) {

  'use strict';

  // please note, NumericEditor is private API just used for this example. see
  // upcoming examples for more info about constructing field editors with
  // spandrel.

  class HorizontalGauge extends spandrel((ramp) => {
    const facets = ramp.get('facets');
    const value = ramp.get('out').get('value');
    const percent = calculatePercent(ramp);

    return (
      <div>
        <div style="border: 1px solid black; width: 100px; height: 10px; position: relative">
          <div style={{ height: '10px', width: `${ percent }px`, background: 'red' }} />
        </div>
        <br/>
        <NumericEditor value={ value } properties={ facets.toObject() } readonly="true" />
      </div>
    );
  }) {
    constructor() {
      super(...arguments);
      subscriberMixIn(this);
      this.getSubscriber().attach('changed', () => this.rerender());
    }
  }

  /**
   * @param {baja.Component} ramp
   * @returns {number} a number between 0 and 100 indicating the ramp's current
   * distance between the bottom and top of its waveform
   */
  function calculatePercent(ramp) {
    const amplitude = ramp.get('amplitude');
    const offset = ramp.get('offset');
    const value = ramp.get('out').get('value');

    const min = offset - amplitude;
    const max = offset + amplitude;

    return (value - min) / (max - min) * 100;
  }

  return HorizontalGauge;
});
