/*
  This widget demonstrates how to create powerful visualizations using open web technology.
  Please note, this isn't a complete example nor is this part of Tridium's commercial 
  offering to be used onsite. It's merely a code example to show what can be done.

  This gauge uses a really powerful open source visualization library called 'd3'...

  http://d3js.org/

  There are a plethora of great web technologies out there. We look forward to seeing how
  you incorporate them into Niagara.
*/
define([
  'baja!',
  'baja!baja:StatusNumeric,control:ControlPoint,gx:Color',
  'bajaux/Widget',
  'bajaux/mixin/subscriberMixIn',
  'd3',
  'css!nmodule/docDeveloper/examples/bajaux/LinearGaugeStyle' ], function (
  baja,
  types,
  Widget,
  subscriberMixIn,
  d3) {

  'use strict';

  const options = {
    left: 20,
    top: 10
  };

  class LinearGauge extends Widget {
    constructor(params) {
      super({
        params,
        defaults: {
          properties: {
            rootCssClass: 'example-linear-gauge-outer',
            barColor: baja.$('gx:Color', 'steelblue'),
            autoScale: true,
            autoScaleStep: 10,
            max: 100,
            min: 0
          }
        }
      });

      subscriberMixIn(this);
    }

    doInitialize(jq) {
      d3.select(jq[0])
        .append('svg')
        .attr('top', 0)
        .attr('left', 0)
        .attr('width', "100%")
        .attr('height', "100%")
        .attr('class', 'example-linear-gauge')
        .append('g')
        .attr('transform', 'translate(' + options.left + ',' + options.top + ')');

      this.getSubscriber().attach("changed", () => render(this));

      return render(this);
    }

    doLoad(value) {
      return render(this);
    }

    doLayout() {
      return render(this);
    }

    doChanged() {
      return render(this);
    }
  }

  function makeModel(widget) {
    // getNumberFromINumeric allows this widget to support many kinds of numeric values.
    const value = baja.Double.getNumberFromINumeric(widget.value() || 0);
    const props = widget.properties();
    let min = props.getValue("min");
    let max = props.getValue("max");
    const autoScale = props.getValue("autoScale");
    const autoScaleStep = props.getValue("autoScaleStep");

    if (autoScale) {
      // Figure out the minimum scale.
      while (value < min) {
        min -= autoScaleStep;
      }

      // Figure out the maximum scale.
      while (value > max) {
        max += autoScaleStep;
      }
    }

    return { max, min, data: [ { value } ] };
  }

  function render(widget) {
    const jq = widget.jq();
    const width = (jq.width() || 800) - options.left * 2;
    const height = 50;
    const { min, max, data } = makeModel(widget);

    //const x = d3.scale.linear()() //Notice how this has been changed from d3 v3 with Niagara 4.14-
    const x = d3.scaleLinear() //d3 v7 with Niagara 4.15+. For more info see https://github.com/d3/d3/blob/main/CHANGES.md
      .domain([ min, max ])
      .range([ 0, width ]);

    const gauge = d3.select(widget.jq()[0])
      .select(".example-linear-gauge")
      .select("g");

    // Render the background
    const bk = gauge.selectAll(".example-linear-grid-background")
      .data(data);

    bk.enter()
      .append("rect")
      .attr("class", "example-linear-grid-background")
      .attr("height", height);

    bk.attr("width", width);

    // The background has a grid on top.
    const grid = gauge.selectAll(".example-linear-grid")
      .data(data);

    grid.enter()
      .append("g")
      .attr("class", "example-linear-grid");

    grid.attr("transform", "translate(0," + height + ")")
      //.call(d3.svg.axis().scale(x).ticks(50).tickSize(-height)) //d3 v3
      .call(d3.axisBottom(x).ticks(50).tickSize(-height)) //d3 v7
      .selectAll(".example-linear-tick")
      .data(x.ticks(10), function (d) { return d; })
      .exit()
      .classed("example-linear-minor", true);

    // The background has an axis drawn at the bottom of it.
    const axis = gauge.selectAll(".example-linear-axis")
      .data(data);

    axis.enter()
      .append("g")
      .attr("class", "example-linear-axis");

    axis.attr("transform", "translate(0," + height + ")")
      //.call(d3.svg.axis().scale(x).ticks(10)); //d3 v3
      .call(d3.axisBottom(x).ticks(10)); //d3 v7

    // The data bar is drawn on top.
    const bar = gauge.selectAll(".example-linear-bar")
      .data(data);

    bar.enter()
      .append("g")
      .append("rect")
      .attr("class", "example-linear-bar")
      .attr("x", 0)
      .attr("y", 0);

    bar.attr("height", height / 2)
      .transition()
      .attr("fill", String(widget.properties().getValue("barColor")))
      .attr("width", function (d) {
        let w = x(d.value);
        if (w > width) {
          w = width;
        } else if (w < 0) {
          w = 0;
        }
        return w;
      });
  }

  return LinearGauge;
});
