/*
  Dashboards in Niagara 4 enable users to modify a Px page without having to use
  a full graphical editor. The data for a Dashboard is stored on the Station for
  the user and the position of the Widget on the Px page.

  In order to use this feature, a user must add a DashboardPane (from the Dashboard
  palette) to a Px page via the Px Editor. A Web Widget is then added to the content of 
  the Dashboard Pane.

  To make a widget dashboardable...

  - Add a widget property with the 'dashboard' attribute set to true.
  - When a dashboard is saved, all properties marked with a truthy dashboard
    attribute are saved.
  - When a dashboard is loaded, all properties are restored.
  - Remember to use the 'readonly' and 'hidden' attributes if you don't want a user
    to see dashboard properties in an editor.
  - If you need to know whether your widget is running in a dashboard, a special 'dashboard'
    property is added to the widget when it's loaded. If this property is true, the widget is
    running in a dashboard.
  - When a widget is loaded on a Dashboard, two commands are dynamically added onto the
    dashboard...
    - Clear Dashboard Data: wipes the widget of any dashboard data. This effectively resets
      the widget back to its default state without any dashboard data.
    - Save: the standard bajaux save command. If the widget is modified, the save command
      is enabled.

  This widget also demonstrates how drag and drop works! Try dragging and dropping anything
  onto the widget.          
*/
define([
  'baja!',
  'log!nmodule.docDeveloper.examples.bajaux.DashboardWidget',
  'bajaux/Widget',
  'bajaux/dragdrop/dragDropUtils',
  'css!nmodule/docDeveloper/examples/bajaux/DashboardWidgetStyle' ], function (
  baja,
  log,
  Widget,
  dragDropUtils) {

  'use strict';

  const MAX_LINES = 10;
  const logSevere = log.severe.bind(log);
  const exampleDbClass = "example-dashboard-widget";
  const highlightClass = "example-dashboard-widget-drag-over";
  const dashboardOrdPropName = "dashboardOrd";

  class DashboardWidget extends Widget {
    constructor(params) {
      super({
        params,
        defaults: {
          properties: {
            rootCssClass: exampleDbClass,
            // 'dashboard' is a special property that can be added to see if a widget is running on a dashboard
            // or not. A lot of the time this property shouldn't be needed as a developer should just work
            // with the properties that have been loaded into the widget.
            dashboard: {
              value: false, hidden: true, readonly: true, transient: true
            },
            // A property marked with the `dashboard` attribute. When the dashboard is saved, this
            // property information will be saved away to the Station for the user.
            [dashboardOrdPropName]: {
              value: '', dashboard: true, readonly: true
            }
          }
        } });
    }

    /**
     * @param {JQuery} dom
     */
    doInitialize(dom) {
      dom.html("<pre></pre>");

      dom.addClass(exampleDbClass);

      this
        .$append("Initialized Dashboard Widget...")
        .$append("Is the Widget on a Dashboard Pane: " + this.properties().getValue("dashboard"))
        .$append("")
        .$append("Dashboard ORD: " + this.properties().getValue(dashboardOrdPropName))
        .$append("Please drag and drop something from the\nNav Tree onto this Widget...");

      // Set up drag and drop.
      dom
        .on("dragover", (e) => {
          dom.addClass(highlightClass);
          e.preventDefault();
        })
        .on("dragleave", (e) => {
          dom.removeClass(highlightClass);
          e.preventDefault();
        })
        .on("drop", (e) => {
          dom.removeClass(highlightClass);

          // Do the drop...
          this.$updateFromDrop(e.originalEvent.dataTransfer)
            .catch(logSevere);

          e.preventDefault();
          e.stopPropagation();
        });
    }

    doDestroy() {
      this.jq().removeClass(highlightClass);
    }

    /**
     * @private
     * @param {string} str
     * @returns {module:bajaux/Widget} this
     */
    $append(str) {
      // Append some new text to the widget's 'pre' DOM element...
      const pre = this.jq().find("pre");
      const lines = pre.text().split('\n');
      pre.text(lines.concat(str).slice(-MAX_LINES).join('\n'));
      return this;
    }

    /**
     * @private
     * @param {DataTransfer} dataTransfer
     * @returns {Promise}
     */
    $updateFromDrop(dataTransfer) {
      return dragDropUtils.fromClipboard(dataTransfer)
        .then((envelope) => {
          switch (envelope.getMimeType()) {
            case 'niagara/navnodes':
              return envelope.toJson()
                .then((json) => {
                  // Although multiple items could have been dragged and dropped on,
                  // we're just going to use the first item in the list.
                  const obj = json && json[0];

                  if (obj && obj.ord) {
                    // Setting this property will automatically mark the widget as modified
                    // due to the fact it's in a Dashboard Pane.
                    this.properties().setValue(dashboardOrdPropName, obj.ord);

                    this.$append("")
                      .$append("New Dashboard ORD...")
                      .$append(obj.ord);
                  }
                });
          }
        });
    }
  }

  return DashboardWidget;
});
