/**
 * @copyright 2023 Tridium, Inc. All Rights Reserved.
 */

define(['bajaux/Widget', 'underscore'], function (Widget, _) {
  'use strict';

  function addMixinName(target, mixinName) {
    var mixins = target.$mixins || (target.$mixins = []);
    if (_.contains(mixins, mixinName)) {
      return false;
    }
    mixins.push(mixinName);
    return true;
  }
  function softExtend(target, source) {
    for (var prop in source) {
      if (source.hasOwnProperty(prop) && !target[prop]) {
        target[prop] = source[prop];
      }
    }
    return true;
  }

  /**
   * API Status: **Private**
   *
   * Shared utility functions for mixins.
   *
   * @exports bajaux/mixin/mixinUtils
   */
  var exports = {};

  /**
   * Add the mixin to the given widget.
   *
   * @param {module:bajaux/Widget|Object} target if a widget, hasMixin will
   * return true after this.
   * @param {String} mixinName
   * @param {Object} [mixin] soft-extend the target with this object (existing
   * keys will not be overwritten).
   *
   * @returns {Boolean} true if the mixin was successfully applied
   */
  exports.applyMixin = function (target, mixinName, mixin) {
    return addMixinName(target, mixinName) && (!mixin || softExtend(target, mixin));
  };

  /**
   * Check to see if an object was previously given a mixin by name.
   *
   * @param {module:bajaux/Widget|Object} obj
   * @param {String} mixinName
   * @returns {Boolean}
   */
  exports.hasMixin = function (obj, mixinName) {
    if (!obj) {
      throw new Error('target required');
    }
    if (obj instanceof Widget) {
      return obj.hasMixIn(mixinName);
    }
    return _.contains(obj.$mixins, mixinName);
  };
  return exports;
});
