/**
 * @copyright 2023 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module bajaux/mixin/UxModelSupport
 */
define(['bajaux/Widget', 'underscore', 'bajaux/model/UxModel', 'bajaux/mixin/mixinUtils'], function (Widget, _, UxModel, mixinUtils) {
  'use strict';

  var applyMixin = mixinUtils.applyMixin;
  var MIXIN_NAME = 'UxModelSupport';

  /**
   * Applies support for `UxModel` to a Widget. A widget with this mixin will
   * support passing `UxModel` instances as well as regular values to `load()`.
   * If the value is not a `UxModel`, it will be applied to the current
   * `UxModel`, so that the value passed to `doLoad` is always a `UxModel` with
   * the current value.
   *
   * The motivation for this mixin is that it enables building a `spandrel`
   * widget that supports both loading in a value as a main content view, and
   * building out its structure in a `UxMedia` page. Without this mixin, a Widget
   * could only work as a main content view or appearing in a `UxMedia` page -
   * not both.
   *
   * By implementing this, you don't need two separate code paths for loading
   * a value and building the structure. Just call `widget.load(uxModel)` or
   * `widget.load(value)`.
   *
   * After applying this mixin, `value()` will return the loaded value, not the
   * `UxModel` itself. To retrieve the `UxModel` directly, call `getModel()`.
   *
   * @mixin
   * @alias module:bajaux/mixin/UxModelSupport
   * @param {module:bajaux/Widget} target
   *
   * @example
   * <caption>A spandrel widget with UxModel support</caption>
   * class FieldEditorWrapper extends spandrel((model, { properties }) => {
   *   const value = model.getValue();
   *   return {
   *     content: {
   *       dom: '<div class="fieldEditorGoesHere"/>',
   *       formFactor: 'mini',
   *       properties,
   *       value
   *     }
   *   };
   * });
   *
   * const ed = new FieldEditorWrapper();
   * return ed.initialize($('#wrapperGoesHere'))
   *   .then(() => UxModel.make({ value: 'hello' }))
   *   .then((model) => ed.load(model)) // builds out structure
   *   .then(() => ed.load('world')); // just updates value
   */
  var UxModelSupport = function UxModelSupport(target) {
    if (!(target instanceof Widget)) {
      throw new Error('Widget required');
    }
    if (!applyMixin(target, MIXIN_NAME, UxModelSupport.prototype)) {
      return;
    }
    var _load = target.load;
    target.load = function load() {
      for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
        args[_key] = arguments[_key];
      }
      return applyWithUxModel.apply(void 0, [this, _load].concat(args));
    };
    var _render = target.render;
    if (_render) {
      target.render = function render() {
        for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
          args[_key2] = arguments[_key2];
        }
        return applyWithUxModel.apply(void 0, [this, _render].concat(args));
      };
    }
    target.value = function () {
      var model = this.getModel();
      return model ? model.getValue() : null;
    };
  };

  /**
   * @returns {module:bajaux/model/UxModel|null}
   */
  UxModelSupport.prototype.getModel = function () {
    return this.$value;
  };
  function applyWithUxModel(ed, func, value) {
    for (var _len3 = arguments.length, args = new Array(_len3 > 3 ? _len3 - 3 : 0), _key3 = 3; _key3 < _len3; _key3++) {
      args[_key3 - 3] = arguments[_key3];
    }
    if (value instanceof UxModel) {
      return func.call.apply(func, [ed, value].concat(args));
    } else {
      var model = ed.getModel();
      return (model instanceof UxModel ? model.clone({
        value: value
      }) : UxModel.make({
        value: value
      })).then(function (model) {
        return func.call.apply(func, [ed, model].concat(args));
      });
    }
  }
  return UxModelSupport;
});
