function _typeof(o) { "@babel/helpers - typeof"; return _typeof = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function (o) { return typeof o; } : function (o) { return o && "function" == typeof Symbol && o.constructor === Symbol && o !== Symbol.prototype ? "symbol" : typeof o; }, _typeof(o); }
/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Utility methods used for managing bajaux containers.
 * 
 * @module bajaux/container/util
 * @private
 */
define(["bajaux/Widget", "bajaux/util/CommandButtonGroup", "Promise", "underscore"], function (Widget, CommandButtonGroup, Promise, _) {
  "use strict";

  var isBoolean = _.isBoolean,
    isNumber = _.isNumber,
    isString = _.isString;
  var exports = {},
    hideCommandBar = "hideCommandBar",
    baja;

  /**
   * Decode a value that's been passed in as a parameter.
   * Please note, this will only work if this Properties instance contains
   * the Property of the specified name and the Property has the 'parameter' attribute
   * set to true. If these conditions aren't met, the return value will be null.
   *
   * @param {module:bajaux/Properties} properties
   * @param {String} name The name of the Property being used to decode.
   * @param {String} value The value currently being decoded.
   * @returns {number|boolean|string|null} The decoded Property value or null.
   */
  exports.decodeValueFromParameter = function (properties, name, value) {
    var to;
    if (properties.has(name) && properties.get(name, "parameter", false)) {
      // Decode the parameter value (it's easier doing this in JavaScript).
      value = decodeURIComponent(value);

      // Check the value of the current Property as we need to try and match it.
      to = _typeof(properties.getValue(name));
      if (to === "number") {
        value = parseFloat(value);
      } else if (to === "boolean") {
        if (typeof value === "string") {
          value = value === "true";
        } else {
          value = !!value;
        }
      }
    } else {
      value = null;
    }
    return value;
  };

  /**
   * The hide command bar property name.
   * 
   * @type {String}
   */
  exports.hideCommandBar = hideCommandBar;

  /**
   * Toggle the Command Bar's visibility on/off.
   * 
   * @param  widget The Widget instance.
   * @param  {JQuery} toolbarJq The jQuery wrapped toolbar DOM.
   */
  exports.toggleCommandBar = function (widget, toolbarJq) {
    var hide = widget.properties().getValue(hideCommandBar);

    // If a property hasn't been specified then base the command bar showing
    // on whether there are any commands in the widget.
    if (typeof hide !== "boolean") {
      hide = widget.getCommandGroup().isEmpty();
    }
    toolbarJq.toggle(!hide).toggleClass('hidden', hide);
  };

  /**
   * @param {JQuery} containerJq
   * @returns {Promise}
   */
  exports.destroyCommandBar = function (containerJq) {
    var headerJq = containerJq.children(".bajaux-toolbar-outer"),
      toolbarJq = headerJq.children(".bajaux-toolbar"),
      cg = toolbarJq.data("widget");
    return Promise.resolve(cg && typeof cg.destroy === "function" && cg.destroy());
  };
  function $initCommandBar(widget, containerJq) {
    var headerJq = containerJq.children(".bajaux-toolbar-outer"),
      toolbarJq = headerJq.children(".bajaux-toolbar"),
      contentJq = containerJq.children(".bajaux-widget-container");
    return exports.destroyCommandBar(containerJq).then(function () {
      exports.toggleCommandBar(widget, toolbarJq);
      var group = new CommandButtonGroup({
        properties: {
          toolbar: true
        }
      });
      return group.initialize(toolbarJq).then(function () {
        return group.load(widget.getCommandGroup());
      }).then(function () {
        function layoutContent() {
          contentJq.css('margin-top', headerJq.height());
        }
        if (containerJq.is('.ux-fullscreen-support, .ux-fullscreen-support .ux-fullscreen') && !widget.properties().getValue(hideCommandBar, false)) {
          if (!widget.$layoutContent) {
            var layout = widget.layout;
            widget.layout = widget.$layoutContent = function () {
              // Layout the content first and then perform the widget layout.
              layoutContent();
              return layout.apply(this, arguments);
            };
          }
        }
        layoutContent();
      });
    });
  }

  /**
   * Initializes the Command Bar.
   *
   * Please note, this will destroy any old Command Bar it finds before
   * creating the new one.
   *
   * @param {module:bajaux/Widget} widget
   * @param  {JQuery} containerJq
   * @returns {Promise} The promise that's resolved once the 
   * Command Bar has been initialized.
   */
  exports.initCommandBar = function (widget, containerJq) {
    var promise = containerJq.data("initCommandBarPromise") || Promise.resolve(),
      initFunc = function initFunc() {
        return $initCommandBar(widget, containerJq);
      };

    // Only allow the command bar to initialize one at a time.    
    promise = promise.then(initFunc, initFunc);
    containerJq.data("initCommandBarPromise", promise);
    return promise;
  };

  /**
   * Pre-destroy timeout. This has been exposed for unit testing.
   * 
   * @type {Number}
   */
  exports.$preDestroyTimeout = 10000;

  /**
   * Called just before a widget is destroyed. This method prevents 'unsubscribe'
   * network calls from happening when a widget is destroyed. Instead it delays the
   * unsubscription network calls until a later time. Providing the widget in question
   * is using its subscriber for all subscriptions, this should speed up destroying
   * this widget.
   * 
   * @param {module:bajaux/Widget} widget A widget that's about to be destroyed.
   */
  exports.preDestroy = function (widget) {
    var sub;
    var baja = getBaja();
    if (widget.hasMixIn("subscriber") && baja && typeof widget.getSubscriber === "function" && typeof widget.getSubscriber().getComponents === "function") {
      sub = new baja.Subscriber();
      sub.subscribe(widget.getSubscriber().getComponents());
      baja.clock.schedule(function () {
        sub.unsubscribeAll();
      }, exports.$preDestroyTimeout);
    }
  };

  /**
   * When bajaux Properties are sent down from the station (as used in BWebWidget or BHxWebWidget),
   * they are sent down using string encoding: the property value is the result of `encodeToString`,
   * and the `typeSpec` metadata informs the client of what Type the value is supposed to be.
   *
   * However, many Widget types actually use the BajaScript values as their Properties. For
   * instance, a Label uses an actual `Brush` instance as the `foreground` property value, not the
   * string encoding of the Brush.
   *
   * This function will take a `WebProperty` as string-encoded by the station, decode it to the
   * desired type of the receiving Widget, and add it to the Widget's properties. If the Widget is
   * already using the old-style string/primitive encoding of Properties, the value will still be
   * string-encoded to respect that.
   *
   * @param {module:bajaux/Properties~PropertyDefinition} webProp
   * @param {module:bajaux/Widget} widget
   * @param {object} [params]
   * @param {boolean} [params.wipeModified] by default, properties with the `userModified` (set when
   * a user changes a dashboardable property) will not be overwritten. Set this to true to overwrite
   * the value even if modified.
   * @param {*} [params.setValueOptions] if given, will be passed to the `setValue()` call.
   * @param {*} [params.setMetadataOptions] if given, will be passed to the `setMetadata()` call.
   * @returns {Promise}
   * @since Niagara 4.14
   */
  exports.applyWebPropertyToWidget = function (webProp, widget) {
    var params = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
    var name = webProp.name,
      value = webProp.value;
    var baja = getBaja();
    var widgetProps = widget.properties();
    if (!baja) {
      // what are type specs?
      return Promise.resolve(applyProp(widgetProps, webProp, params));
    }
    if (widgetProps.get(name, 'userModified') && !params.wipeModified) {
      return Promise.resolve();
    }
    var widgetPropValue = widgetProps.getValue(name);
    if (!baja.hasType(widgetPropValue) || isPrimitive(widgetPropValue)) {
      return Promise.resolve(applyProp(widgetProps, exports.encodeToWebPropertyDefinition(webProp), params));
    } else if (isString(value)) {
      return Promise.resolve(widgetPropValue.getType().getInstance().decodeAsync(value)).then(function (value) {
        return applyProp(widgetProps, Object.assign({}, webProp, {
          value: value
        }), params);
      });
    } else {
      return Promise.resolve(applyProp(widgetProps, webProp, params));
    }
  };
  function applyProp(properties, prop, _ref) {
    var setValueOptions = _ref.setValueOptions,
      setMetadataOptions = _ref.setMetadataOptions;
    var name = prop.name,
      value = prop.value,
      metadata = prop.metadata;
    var existing = properties.get(name);
    if (!existing) {
      return properties.add(prop);
    }
    if (value !== undefined) {
      properties.setValue(name, value, setValueOptions);
    }
    if (metadata !== undefined) {
      properties.setMetadata(name, metadata, setMetadataOptions);
    }
  }
  function getBaja() {
    baja = baja || require.defined("baja!") ? require("baja!") : null;
    return baja;
  }
  function isPrimitive(val) {
    return isString(val) || isBoolean(val) || isNumber(val);
  }

  /**
   * @param {string} typeSpec
   * @returns {boolean} true if the typeSpec is "unspecified" - not present, or set to the default
   * value of `baja:String` in which case we will string-decode as normal
   */
  function isTypeSpecSpecified(typeSpec) {
    return typeSpec && typeSpec !== 'baja:String';
  }

  /**
   * The WebProperty style of Properties encoding always has `value` as a JavaScript primitive
   * (string, number, or boolean). If `value` is not a primitive already, it gets
   * encodeToString()ed. This function examines the Property as given (which may have an
   * actual Baja value such as a Facets) and converts it to the WebProperty-style definition with
   * a primitive `value`.
   *
   * @param {module:bajaux/Properties~PropertyDefinition} webProp
   * @returns {module:bajaux/Properties~PropertyDefinition}
   */
  exports.encodeToWebPropertyDefinition = function (webProp) {
    var baja = getBaja();
    if (!baja) {
      return webProp; //give up
    }
    var typeSpec = webProp.typeSpec,
      value = webProp.value;
    var type = !isTypeSpecSpecified(typeSpec) && baja.hasType(value) ? value.getType() : baja.lt(typeSpec);
    if (!type) {
      return webProp;
    }
    var primitive = value;
    if (type.is('baja:Boolean')) {
      primitive = isBoolean(value) ? value : value === 'true';
    } else if (type.is('baja:Number')) {
      primitive = baja.$(type, value).valueOf();
    } else if (type.is('baja:Enum')) {
      primitive = (baja.hasType(value, 'baja:Enum') ? value : baja.$(type, value)).getTag();
    } else if (baja.hasType(value, 'baja:Simple')) {
      primitive = value.encodeToString();
    }
    return Object.assign({}, webProp, {
      value: primitive,
      typeSpec: String(type)
    });
  };
  return exports;
});
