/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

define(['baja!', 'baja!alarm:AlarmAlgorithm,alarm:AlarmSourceExt,' + 'control:ControlPoint', 'lex!alarm', 'underscore', 'nmodule/alarm/rc/Alarm', 'nmodule/webEditors/rc/util/textUtils', 'Promise'], function (baja, types, lexs, _, Alarm, textUtils, Promise) {
  'use strict';

  var alarmLex = lexs[0],
    DEFAULT_ALARM_CLASS = 'defaultAlarmClass';

  /**
   * Utility functions for alarm editors and views.
   *
   * API Status: **Private**
   * @exports nmodule/alarm/rc/util/alarmUtils
   */
  var exports = {};

  /**
   * Get the parent `AlarmSourceExt` of an `AlarmAlgorithm`.
   *
   * @param {baja.Component} algorithm an `alarm:AlarmAlgorithm` component
   * @returns {baja.Component|null} the parent `alarm:AlarmSourceExt` component,
   * or `null` if not found
   * @see BAlarmAlgorithm#getParentExt
   */
  exports.getParentExt = function (algorithm) {
    var parent = baja.hasType(algorithm, 'alarm:AlarmAlgorithm') && algorithm.getParent();
    if (!baja.hasType(parent, 'alarm:AlarmSourceExt')) {
      return null;
    }
    return parent;
  };

  /**
   * Get the grandparent `control:ControlPoint` of an `alarm:AlarmAlgorithm`
   * component.
   *
   * @param {baja.Component} algorithm an `alarm:AlarmAlgorithm` component
   * @returns {baja.Component|null} the grandparent `control:ControlPoint`
   * component, or `null` if not found
   * @see BAlarmAlgorithm#getParentPoint
   */
  exports.getParentPoint = function (algorithm) {
    var ext = exports.getParentExt(algorithm),
      point = ext && ext.getParent();
    if (!baja.hasType(point, 'control:ControlPoint')) {
      return null;
    }
    return point;
  };

  /**
   * Get the facets of an `alarm:AlarmAlgorithm`'s containing control point.
   *
   * @param {baja.Component} algorithm an `alarm:AlarmAlgorithm` component
   * @returns {Promise} promise to be resolved with the `facets` slot of the
   * containing control point, or with `baja.Facets.DEFAULT` if not found
   * @see BAlarmAlgorithm#getPointFacets
   */
  exports.getPointFacets = function (algorithm) {
    var point = exports.getParentPoint(algorithm);
    if (point) {
      return Promise.resolve(point.isMounted() && point.lease()).then(function () {
        return point.get('facets');
      });
    }
    return Promise.resolve(baja.Facets.DEFAULT);
  };

  /**
   * Returns the display name for the alarm data column.
   *
   * If a display name can't be found in a lexicon, the key will
   * attempt to be converted to a friendly name.
   *
   * @since Niagara 4.6
   *
   * @param  {String} key The key for the display name.
   * @returns {String} The display name.
   */
  exports.getAlarmDataDisplayName = function (key) {
    return alarmLex.get('alarmData.' + key) || textUtils.toFriendly(key);
  };

  /**
   * Checks whether the user has operatorWrite access to all Alarm Classes in the provided AlarmRecords.
   *
   * @param {Array.<module:nmodule/webEditors/rc/wb/table/model/Row|baja.Struct>} rows An Array of Table Model Rows or
   * an Array of `baja.Struct` records which are each a `alarm:AlarmRecord`.
   * @returns {Promise.<boolean>}
   * @since Niagara 4.12
   */
  exports.hasOperatorWrite = function (rows) {
    var hasRequiredPermissions = true;
    var subjects = rows.map(function (row) {
      return row.getSubject ? row.getSubject() : row;
    });
    var uniqueAlarmClassNames = _.chain(subjects).map(function (subject) {
      return subject._alarmClass || DEFAULT_ALARM_CLASS;
    }).uniq().value();
    return Alarm.getPermissionsMap(uniqueAlarmClassNames).then(function (permissionsMap) {
      var localPermissionsMap = _.mapObject(permissionsMap, baja.Permissions.make);
      subjects.forEach(function (subject) {
        var alarmClass = subject._alarmClass || DEFAULT_ALARM_CLASS;
        var permissions = localPermissionsMap[alarmClass];
        hasRequiredPermissions = hasRequiredPermissions && alarmClass && permissions && permissions.hasOperatorWrite();
      });
      return hasRequiredPermissions;
    });
  };
  return exports;
});
