/*jshint browser: true */ /* eslint-env browser */
/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/db/DatabaseMaintenance
 */
define(['bajaux/Widget', 'baja!', 'baja!alarm:AlarmRecord,bql:DynamicTimeRange,bql:AbsTimeFilter,alarm:AlarmService,alarm:ArchiveAlarmProvider,alarm:AlarmArchive', 'jquery', 'underscore', 'Promise', 'lex!alarm', 'log!nmodule.alarm.rc.db.DatabaseMaintenance', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/BaseWidget', 'bajaux/events', 'nmodule/alarm/rc/db/editors/Pager', 'nmodule/alarm/rc/db/editors/RunCommand', 'nmodule/alarm/rc/db/commands/NotesCommand', 'nmodule/alarm/rc/db/commands/RefreshCommand', 'nmodule/alarm/rc/db/commands/FilterCommand', 'nmodule/alarm/rc/db/commands/AlarmDetailsCommand', 'nmodule/alarm/rc/db/commands/PlayCommand', 'hbs!nmodule/alarm/rc/db/templates/view', 'css!nmodule/alarm/rc/db/templates/view'], function (Widget, baja, types, $, _, Promise, lexs, log, fe, BaseWidget, events, Pager, RunCommand, NotesCommand, RefreshCommand, FilterCommand, AlarmDetailsCommand, PlayCommand, viewTemplate) {
  'use strict';

  var lex = lexs[0],
    logSevere = log.severe.bind(log),
    CLEAR_OLD_ALARMS_LABEL = lex.get('command.clearOldRecords.label'),
    CLEAR_SELECTED_ALARMS_LABEL = lex.get('command.clearSelectedRecords.label'),
    CLEAR_BEFORE_SELECTED_ALARM_LABEL = lex.get('command.clearRecordsBeforeSelected.label'),
    CLEAR_ALL_ALARMS_LABEL = lex.get('command.clearAllRecords.label'),
    RESULTS_PER_PAGE_LABEL = lex.get('alarm.db.resultsPerPage'),
    DATE_EDITOR_CLASS = 'dateEditor',
    DATE_EDITOR_SELECTOR = '.' + DATE_EDITOR_CLASS,
    TIME_RANGE_EDITOR_SELECTOR = '.timeRangeEditor',
    MAINTENANCE_GROUP_SELECTOR = '.alarmMaintenance',
    RUN_BTN_SELECTOR = '.run',
    PAGING_SELECTOR = '.paging',
    ALARM_TABLE_SELECTOR = '.alarmTable';
  function initCommandGroup(view) {
    var cmdGroup = view.getCommandGroup();
    _.each([PlayCommand.makeFor(view), FilterCommand.makeFor(view), AlarmDetailsCommand.makeFor(view), new NotesCommand(view), RefreshCommand.makeFor(view)], function (cmd) {
      cmdGroup.add(cmd);
    });
    return view;
  }
  function initTimeRangeEditor(view) {
    return baja.TimeZoneDatabase.get().then(function (database) {
      var timezoneId = view.$getBrowserTimeZone(),
        timezone = timezoneId && database.getTimeZone(timezoneId) || database.getTimeZone("UTC");
      return fe.buildFor({
        dom: $(TIME_RANGE_EDITOR_SELECTOR, view.jq()),
        value: baja.$('bql:DynamicTimeRange', 'today'),
        formFactor: 'mini',
        properties: {
          TimeZone: timezone.encodeToString()
        }
      });
    }).then(function (editor) {
      view.timeRangeEditor = editor;
      editor.jq().on(events.MODIFY_EVENT, function (e, ed) {
        var pager = view.pager;
        if (pager) {
          ed.read().then(function (timeRange) {
            pager.setTimeRange(timeRange);
          })["catch"](logSevere);
        }
        return false;
      });
      return view;
    });
  }
  function initDateEditor(view) {
    if (!view.isMaintenanceView()) {
      return view;
    }
    var today = new Date(),
      year = today.getFullYear() - 5,
      month = today.getMonth(),
      day = today.getDate();
    return fe.buildFor({
      dom: $(DATE_EDITOR_SELECTOR, view.jq()),
      value: baja.AbsTime.make({
        jsDate: new Date(year, month, day)
      }),
      //NCCB-14801
      formFactor: 'mini'
    }).then(function (editor) {
      editor.jq().on(events.MODIFY_EVENT, function () {
        return false;
      });
      view.dateEditor = editor;
      return view;
    });
  }
  function initRunMaintenanceButton(view, isAlarmArchiveSpace) {
    if (!view.isMaintenanceView()) {
      return view;
    }
    return RunCommand.makeFor(view, isAlarmArchiveSpace).then(function (button) {
      view.runBtn = button;
      return view;
    });
  }
  function initPaging(view, isAlarmArchiveSpace) {
    return Pager.makeFor(view, isAlarmArchiveSpace).then(function (pager) {
      //set bottom margin if db view
      if (!view.isMaintenanceView()) {
        $(PAGING_SELECTOR, view.jq()).css('margin-bottom', '20px');
      }
      view.pager = pager;
      return view;
    });
  }
  function getWidget(view, selector) {
    return $(selector, view.jq()).data('widget');
  }

  /**
   * A view that handles the management details for alarms in
   * a station Alarm Database
   *
   * @class
   * @alias module:nmodule/alarm/rc/db/DatabaseMaintenance
   * @extends module:nmodule/webEditors/rc/fe/BaseWidget
   * @param params
   */
  var DatabaseMaintenance = function DatabaseMaintenance(params) {
    BaseWidget.call(this, _.extend({
      moduleName: 'alarm',
      keyName: 'DatabaseMaintenance'
    }, params || {}));
  };

  //extend and set up prototype chain
  DatabaseMaintenance.prototype = Object.create(BaseWidget.prototype);
  DatabaseMaintenance.prototype.constructor = DatabaseMaintenance;
  DatabaseMaintenance.prototype.$getTimeRangeEditor = function () {
    return getWidget(this, TIME_RANGE_EDITOR_SELECTOR);
  };
  DatabaseMaintenance.prototype.$getDateEditor = function () {
    return getWidget(this, DATE_EDITOR_SELECTOR);
  };
  DatabaseMaintenance.prototype.$getPager = function () {
    return this.pager;
  };
  DatabaseMaintenance.prototype.$getRunBtn = function () {
    return getWidget(this, RUN_BTN_SELECTOR);
  };

  /**
   * Return the local time zone
   *
   * @private
   * @returns {String} Time zone id
   */
  DatabaseMaintenance.prototype.$getBrowserTimeZone = function () {
    return Intl.DateTimeFormat().resolvedOptions().timeZone;
  };

  /**
   * Return an array of the currently selected alarm records.
   *
   * @private
   * @returns {Array} An array of alarm records.
   */
  DatabaseMaintenance.prototype.$getCurrentRecords = function () {
    var table = getWidget(this, ALARM_TABLE_SELECTOR),
      rows = table.getSelectedRows();
    return _.invoke(rows, 'getSubject');
  };
  DatabaseMaintenance.prototype.isMaintenanceView = function () {
    return true;
  };

  /**
   * Initial DOM setup
   *
   * @param {JQuery} dom the DOM element into which to load this View
   * @return {Promise} Promise
   */
  DatabaseMaintenance.prototype.doInitialize = function (dom) {
    var that = this,
      widget = dom.data('widget');

    //NCCB-15117 - Destroy existing widget if present in DOM data
    return Promise.resolve(baja.hasType(widget, Widget) && !widget.isDestroyed() && widget.destroy()).then(function () {
      //load the view template
      dom.html(viewTemplate({
        maintenanceView: that.isMaintenanceView(),
        clearOldId: that.generateId(),
        clearOld: CLEAR_OLD_ALARMS_LABEL,
        clearSelectedId: that.generateId(),
        clearSelected: CLEAR_SELECTED_ALARMS_LABEL,
        clearBeforeSelectedId: that.generateId(),
        clearBeforeSelected: CLEAR_BEFORE_SELECTED_ALARM_LABEL,
        clearAllId: that.generateId(),
        clearAll: CLEAR_ALL_ALARMS_LABEL,
        resultsPerPage: RESULTS_PER_PAGE_LABEL
      }));

      //NCCB-14806: Disable text-highlighting in IE
      dom.on('selectstart', false);
      return Promise.all([initCommandGroup(that), initTimeRangeEditor(that), initDateEditor(that)]);
    });
  };
  DatabaseMaintenance.prototype.doLoad = function (value) {
    var that = this,
      isAlarmArchiveSpace = false;
    if (baja.hasType(value, 'alarm:ArchiveAlarmProvider') || baja.hasType(value, 'alarm:AlarmArchive')) {
      isAlarmArchiveSpace = true;
    }
    if (!baja.hasType(value, 'alarm:AlarmService')) {
      // get the alarmService and operate on that
      return baja.Ord.make('service:alarm:AlarmService').get().then(function (service) {
        that.jq().find(MAINTENANCE_GROUP_SELECTOR).toggle(service.getPermissions().hasAdminInvoke());
      }).then(function () {
        return Promise.all([initRunMaintenanceButton(that, isAlarmArchiveSpace), initPaging(that, isAlarmArchiveSpace)]);
      });
    } else {
      that.jq().find(MAINTENANCE_GROUP_SELECTOR).toggle(value.getPermissions().hasAdminInvoke());
      return Promise.all([initRunMaintenanceButton(that, isAlarmArchiveSpace), initPaging(that, isAlarmArchiveSpace)]);
    }
  };

  /**
   * Destroy the relevant field editors
   */
  DatabaseMaintenance.prototype.doDestroy = function () {
    return this.getChildWidgets().destroyAll();
  };
  return DatabaseMaintenance;
});
