/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/commands/DetailsCommand
 */
define(['lex!alarm,bajaui', 'log!nmodule.alarm.rc.console.commands.DetailsCommand', 'jquery', 'dialogs', 'Promise', 'bajaux/commands/Command', 'nmodule/alarm/rc/console/commands/AcknowledgeCommand', 'nmodule/alarm/rc/console/commands/HyperlinkCommand', 'nmodule/alarm/rc/console/commands/NotesCommand', 'nmodule/alarm/rc/console/commands/SequenceCommand', 'nmodule/alarm/rc/console/fe/DetailsEditor', 'nmodule/js/rc/log/Log', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/alarm/rc/console/commands/alarmConsoleCommandMixIn', 'nmodule/alarm/rc/util/alarmUtils'], function (lexs, log, $, dialogs, Promise, Command, AcknowledgeCommand, HyperlinkCommand, NotesCommand, SequenceCommand, DetailsEditor, Log, fe, feDialogs, alarmConsoleCommandMixIn, alarmUtils) {
  'use strict';

  var lex = lexs[0],
    bajauiLex = lexs[1];
  var logSevere = log.severe.bind(log);
  function details(cmd) {
    var support = cmd.$support,
      ackCmd = cmd.$ackCmd,
      hyperlinkCmd = cmd.$hyperlinkCmd,
      notesCmd = cmd.$notesCmd,
      sequenceCmd = cmd.$sequenceCmd;
    return sequenceCmd.init().then(function (subject) {
      if (!subject) {
        return;
      }
      return alarmUtils.hasOperatorWrite([subject]).then(function (hasPermission) {
        var needsAck = hasPermission && subject._ackState !== 'acked';
        var hasHyperlink = !!(subject.alarmData && subject.alarmData.hyperlinkOrd);
        return feDialogs.showFor({
          value: subject,
          type: DetailsEditor,
          formFactor: 'compact',
          properties: {
            support: support
          },
          buttons: [{
            name: 'previous',
            displayName: bajauiLex.get('commands.back.label'),
            handler: function handler(dialog, e, ed) {
              dialog.enableButton('next');
              sequenceCmd.previous().then(function (subject) {
                if (subject) {
                  return ed.load(subject);
                } else {
                  dialog.disableButton('previous');
                }
              })["catch"](Log.error);
              dialog.keepOpen();
            }
          }, {
            name: 'next',
            displayName: bajauiLex.get('commands.forward.label'),
            handler: function handler(dialog, e, ed) {
              dialog.enableButton('previous');
              sequenceCmd.next().then(function (subject) {
                if (subject) {
                  return ed.load(subject);
                } else {
                  dialog.disableButton('next');
                }
              })["catch"](Log.error);
              dialog.keepOpen();
            }
          }, {
            name: 'acknowledge',
            displayName: lex.get('commands.acknowledge.displayName'),
            disable: !needsAck,
            handler: function handler(dialog, e, ed) {
              ackCmd.invoke({
                subject: ed.value()
              }).then(function (failures) {
                if (!failures || failures.length === 0) {
                  dialog.disableButton('acknowledge');
                }
              })["catch"](Log.error);
              dialog.keepOpen();
            }
          }, {
            name: 'hyperlink',
            displayName: lex.get('commands.hyperlink.displayName'),
            disable: !hasHyperlink,
            handler: function handler(dialog, e, ed) {
              hyperlinkCmd.invoke({
                subject: ed.value()
              })["catch"](Log.error);
              dialog.keepOpen();
            }
          }, {
            name: 'notes',
            displayName: lex.get('commands.notes.displayName'),
            handler: function handler(dialog, e, ed) {
              cmd.$support.getNotes({
                uuid: ed.value().uuid
              }).then(function (notes) {
                var subject = ed.value();
                subject.alarmData.notes = notes;
                return notesCmd.invoke({
                  callThrough: true,
                  subject: subject
                })["catch"](Log.error);
              })["catch"](logSevere);
              dialog.keepOpen();
            }
          }, {
            name: 'close',
            displayName: lex.get('command.closeNoteDialog.label'),
            handler: function handler(dialog, e, ed) {
              return Promise.all([sequenceCmd.reset(), ed && ed.destroy()]);
            },
            esc: true
          }]
        });
      });
    });
  }

  /**
   * Shows the currently selected alarm's details in a pop up dialog box.
   *
   * @constructor
   * @param {module:nmodule/alarm/rc/console/AlarmConsole} view The alarm console view.
   * @extends {module:bajaux/commands/Command}
   * @mixes {module:nmodule/alarm/rc/console/commands/alarmConsoleCommandMixIn}
   */
  var DetailsCommand = function DetailsCommand(view) {
    var that = this;
    Command.apply(that, [{
      module: 'alarm',
      lex: 'alarm.console.commands.details',
      func: function func() {
        return details(that);
      }
    }]);
    alarmConsoleCommandMixIn(that);
    that.$view = view;
    that.$viewModel = view.$getViewModel();
    that.$support = that.$viewModel.$getSupport();
    that.$ackCmd = new AcknowledgeCommand(view, {
      details: true
    });
    that.$hyperlinkCmd = new HyperlinkCommand(view, {
      details: true
    });
    that.$notesCmd = new NotesCommand(view, {
      details: true
    });
    that.$sequenceCmd = new SequenceCommand(view);
    that.$showInMenu = true;
  };
  DetailsCommand.prototype = Object.create(Command.prototype);
  DetailsCommand.prototype.constructor = DetailsCommand;
  DetailsCommand.prototype.$getAcknowledgeCommand = function () {
    return this.$ackCmd;
  };
  DetailsCommand.prototype.$getHyperlinkCommand = function () {
    return this.$hyperlinkCmd;
  };
  DetailsCommand.prototype.$getNotesCommand = function () {
    return this.$notesCmd;
  };
  DetailsCommand.prototype.$getSequenceCommand = function () {
    return this.$sequenceCmd;
  };
  DetailsCommand.prototype.selectionChanged = function () {
    this.setEnabled(this.$view.getSelectedAlarmRecords().length === 1);
  };
  return DetailsCommand;
});
