/* jshint browser:true */
/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/AlarmConsoleViewModel
 */
define(['lex!alarm', 'Promise', 'underscore', 'nmodule/js/rc/tinyevents/tinyevents', 'nmodule/alarm/rc/console/support/SupportFactory', 'nmodule/alarm/rc/console/audio/AlarmAudioHandler', 'nmodule/alarm/rc/console/audio/AlarmAudioQueue', 'dialogs'], function (lexs, Promise, _, tinyevents, SupportFactory, AlarmAudioHandler, AlarmAudioQueue, dialogs) {
  'use strict';

  var alarmLex = lexs[0],
    highSourceCount = 500,
    RECIPIENT_UPDATE_EVENT = 'recipientUpdate',
    SINGLE_SOURCE_UPDATE_EVENT = 'singleSourceUpdate',
    OPTIONS_CHANGED_EVENT = 'options',
    COUNTS_CHANGED_EVENT = 'countsChanged',
    ALARM_AUDIO_QUEUE = new AlarmAudioQueue(SupportFactory.getAudioSupport('baja'));
  function restartAndLoadAlarmSummary(viewModel) {
    return dialogs.showLoading(0, loadAlarmSummary(viewModel, /*restart*/true)).promise();
  }
  function loadAlarmSummary(viewModel, restart) {
    var support = viewModel.$support,
      params = {
        source: viewModel.$source,
        timeRange: viewModel.$timeRange,
        filterSet: viewModel.$filterSet,
        offset: viewModel.$pageOffset,
        limit: viewModel.$pageSize,
        restart: !!restart,
        column: viewModel.$column,
        sortDesc: viewModel.$sortDesc
      };
    return support && support.loadAlarmSummary(params);
  }
  function processSummary(viewModel, summary, update) {
    var tableModel = viewModel.$tableModel,
      records = summary.records || [],
      rows = tableModel.getRows(),
      sc = summary.sourceCount,
      ac = summary.alarmCount,
      pc = summary.pageCount,
      idx = viewModel.$pageIndex,
      sources = {},
      newSources = {},
      toInsert = [],
      toChanged = [],
      toRemove = [];
    if (!update) {
      toRemove = rows;
      toInsert = records;
    } else {
      // Convert the row to a sources map (source -> row).
      rows.forEach(function (row) {
        sources[row.getSubject().source] = row;
      });

      // For each record determine whether to insert a new row or trigger
      // a changed event.
      records.forEach(function (record) {
        newSources[record.source] = record;
        var row = sources[record.source];
        if (row) {
          // Perform a deep comparison against the record to see if there's any differences.
          if (!_.isEqual(row.getSubject(), record)) {
            // If there's a difference then update the subject and mark the row for change.
            row.$setSubject(record);
            toChanged.push(row);
          }
        } else {
          // We need to insert a row if it doesn't exist yet.
          toInsert.push(record);
        }
      });

      // Find out what sources are missing to determine what needs to be removed.
      rows.forEach(function (row) {
        if (!newSources[row.getSubject().source]) {
          toRemove.push(row);
        }
      });

      // If any rows have changed then emit an event.
      if (toChanged.length) {
        tableModel.emit('rowsChanged', toChanged);
      }
    }

    // Set some data to indicate whether this row was changed or inserted from
    // an update. This used by the row animation to indicate to the user something has
    // changed or been added.
    toInsert = toInsert.map(function (record) {
      var row = tableModel.makeRow(record);
      row.data("fromUpdate", update);
      return row;
    });
    toChanged.forEach(function (row) {
      row.data("fromUpdate", update);
    });
    return Promise.resolve(toRemove.length && tableModel.removeRows(toRemove)).then(function () {
      if (toInsert.length > highSourceCount && !viewModel.$isSingleSourceModel() && !update) {
        return dialogs.showYesNo(alarmLex.get('alarm.console.multiSource.highSourceCount')).yes(function () {
          toInsert = toInsert.splice(0, highSourceCount);
        }).promise();
      }
    }).then(function () {
      return toInsert.length && tableModel.insertRows(toInsert);
    }).then(function () {
      viewModel.$sourceCount = sc;
      viewModel.$alarmCount = ac;
      viewModel.$pageCount = pc;
      var params = {
        sourceCount: sc,
        alarmCount: ac,
        currentIdx: idx,
        noOfPages: pc
      };
      if (viewModel.$isSingleSourceModel()) {
        params.sourceName = records[0] && records[0].alarmData && records[0].alarmData.sourceName;
      }
      viewModel.emit(COUNTS_CHANGED_EVENT, params);
    });
  }
  function checkFailures(viewModel, failures) {
    failures = _.isArray(failures) || [];
    return Promise.resolve(failures.length && failures || viewModel.$singleSource && loadAlarmSummary(viewModel));
  }
  function updateAlarmAudioHandler(viewModel, summary) {
    viewModel.$audioHandler.update(summary);
  }
  function optionsChangedHandler(viewModel, options) {
    ALARM_AUDIO_QUEUE.setEnabled(!options.soundOff);
    ALARM_AUDIO_QUEUE.setContinuous(!!options.soundContinuous);
    if (!viewModel.$isSingleSourceModel()) {
      ALARM_AUDIO_QUEUE.setDelay(options.multiSourceView.soundDelay || AlarmAudioQueue.DEFAULT_ALARM_DELAY);
      viewModel.$audioHandler.setDefaultSound(options.multiSourceView.soundFile || AlarmAudioHandler.DEFAULT_ALARM_SOUND);
    }
  }

  /**
   * AlarmConsoleViewModel is the view model for an AlarmConsole instance.
   * The view model is responsible for maintaining view state and handling
   * command delegation.
   *
   * @class
   * @alias module:nmodule/alarm/rc/console/AlarmConsoleViewModel
   * @see {module:nmodule/alarm/rc/console/AlarmConsole}
   * @param {Object} [params] - object literal specifying view model params
   * @param {String} [params.source] - String encoding of the baja.Ord for the source in a single
   * source model. If this parameter is not specified the model defaults to a multi-source view
   * model
   */
  var AlarmConsoleViewModel = function AlarmConsoleViewModel(params) {
    var that = this;
    tinyevents(that);

    // Support
    that.$support = SupportFactory.getSupport('baja');

    // State
    that.$source = params && params.source || 'null';
    that.$singleSource = that.$source !== 'null';
    that.$tableModel = that.$support.getDefaultTableModel();
    that.$timeRange = that.$support.getDefaultTimeRange();
    that.$filterSet = that.$support.getDefaultFilterSet();
    that.$pageIndex = that.$support.getDefaultPageIndex();
    that.$sourceCount = 0;
    that.$alarmCount = 0;
    that.$checkFailures = _.partial(checkFailures, that);
    that.$audioHandler = new AlarmAudioHandler(ALARM_AUDIO_QUEUE);

    // Event handling.
    that.$support.on(SINGLE_SOURCE_UPDATE_EVENT, _.partial(processSummary, that));
    that.$support.on(RECIPIENT_UPDATE_EVENT, function (summary, update) {
      if (!that.$isSingleSourceModel()) {
        processSummary(that, summary, update);
      }
    });
    that.$support.on(RECIPIENT_UPDATE_EVENT, function (summary, update) {
      that.emit(RECIPIENT_UPDATE_EVENT, summary, update);
    });
    that.$support.on(RECIPIENT_UPDATE_EVENT, _.partial(updateAlarmAudioHandler, that));
    that.$support.on(OPTIONS_CHANGED_EVENT, _.partial(optionsChangedHandler, that));
  };

  /**
   * The alarm audio queue.
   *
   * @type {module:nmodule/alarm/rc/console/audio/AlarmAudioQueue}
   */
  AlarmConsoleViewModel.ALARM_AUDIO_QUEUE = ALARM_AUDIO_QUEUE;
  AlarmConsoleViewModel.$processSummary = processSummary;

  //////////////////////////////////////////////////////////////////////////
  //Lifecycle
  //////////////////////////////////////////////////////////////////////////

  /**
   * Loads the view model with the target recipient.
   *
   * @param {Object} recipient - remote recipient with which model will interface
   * @return {Promise} Promise resolved when recipient has been loaded into the model
   */
  AlarmConsoleViewModel.prototype.load = function (recipient) {
    var that = this,
      support = that.$support;
    that.$timeRange = recipient.getDefaultTimeRange();
    return that.$support.load(recipient, {
      timeRange: that.$timeRange
    }).then(function () {
      that.$pageSize = support.getDefaultPageSize();
      that.$pageOffset = (that.$pageIndex - 1) * that.$pageSize;
      return Promise.all([that.loadAlarmDataColumns(), that.loadHiddenColumns()]);
    });
  };

  /**
   * De-registers event listeners and destroys view model support instance.
   *
   * @return {Promise} Promise resolved when view model has been destroyed
   */
  AlarmConsoleViewModel.prototype.destroy = function () {
    this.$audioHandler.destroy();
    this.removeAllListeners();
    return this.$support.destroy();
  };

  //////////////////////////////////////////////////////////////////////////
  //Column Support
  //////////////////////////////////////////////////////////////////////////

  /**
   * Adds alarm data columns to the table based on alarm console options.
   *
   * @returns {Promise}
   */
  AlarmConsoleViewModel.prototype.loadAlarmDataColumns = function () {
    var that = this,
      support = that.$support,
      options = support.getAlarmConsoleOptions();
    return support && support.getAlarmFields().then(function (fields) {
      that.$defaultAlarmDataColumnNames = fields;
      that.$addableColumnNames = fields;
      that.$removableColumnNames = [];
      that.$alarmDataColumns = [];
      if (that.$isSingleSourceModel()) {
        return Promise.all(_.map(options.singleSourceView.alarmDataColumns, function (name) {
          return that.addAlarmDataColumn(name);
        }));
      } else {
        return Promise.all(_.map(options.multiSourceView.alarmDataColumns, function (name) {
          return that.addAlarmDataColumn(name);
        }));
      }
    });
  };

  /**
   * Hides columns based on alarm console options.
   *
   * @returns {Promise}
   */
  AlarmConsoleViewModel.prototype.loadHiddenColumns = function () {
    var that = this,
      tableModel = that.$tableModel,
      support = that.$support,
      options = support.getAlarmConsoleOptions();
    if (that.$isSingleSourceModel()) {
      _.each(tableModel.getColumns(), function (column) {
        column.setUnseen(_.contains(options.singleSourceView.hiddenColumns, column.getName()));
      });
    } else {
      _.each(tableModel.getColumns(), function (column) {
        column.setUnseen(_.contains(options.multiSourceView.hiddenColumns, column.getName()));
      });
    }
    return Promise.resolve();
  };

  /**
   * Updates the alarm console options with the current list of hidden columns.
   *
   * @returns {Promise}
   */
  AlarmConsoleViewModel.prototype.setHiddenColumns = function () {
    var hiddenColumns = _.filter(this.$tableModel.getColumns(), function (column) {
      return column.isUnseen();
    });
    hiddenColumns = _.map(hiddenColumns, function (column) {
      return column.getName();
    });
    return this.$support.setAlarmConsoleOption('hiddenColumns', hiddenColumns, this.$isSingleSourceModel());
  };

  /**
   * Get a list of default AlarmData column names that don't
   * currently exist in the alarm summary table.
   *
   * @returns {Array.<String>}
   */
  AlarmConsoleViewModel.prototype.getAddableColumnNames = function () {
    return this.$addableColumnNames;
  };

  /**
   * Get a list of AlarmData column names that
   * currently exist in the alarm summary table.
   *
   * @returns {Array.<String>}
   */
  AlarmConsoleViewModel.prototype.getRemovableColumnNames = function () {
    return this.$removableColumnNames;
  };

  /**
   * Reset the alarm table settings on client and server.
   *
   * @returns {Promise}
   */
  AlarmConsoleViewModel.prototype.resetAlarmTableSettings = function () {
    return this.$support.resetAlarmTableSettings(this.$isSingleSourceModel());
  };

  /**
   * Reloads alarm data columns and shows or hides columns based on the
   * current alarm console options for the current view.
   *
   * @returns {Promise}
   */
  AlarmConsoleViewModel.prototype.reloadAlarmTableColumns = function () {
    var that = this;
    return that.removeAlarmDataColumns().then(function () {
      return that.loadAlarmDataColumns();
    }).then(function () {
      return that.loadHiddenColumns();
    });
  };

  //////////////////////////////////////////////////////////////////////////
  //Command Handlers / Event delegation
  //////////////////////////////////////////////////////////////////////////
  /**
   * Adds an AlarmData column to the alarm summary table if it doesn't
   * already exist in the table.
   *
   * @param {String} name column name
   * @returns {Promise}
   */
  AlarmConsoleViewModel.prototype.addAlarmDataColumn = function (name) {
    var that = this;
    var columnNames = _.map(that.$alarmDataColumns, function (column) {
      return column.getName();
    });
    if (name && !_.contains(columnNames, name)) {
      var column = that.$support.createAlarmDataColumn(name);
      return that.$tableModel.insertColumns([column], name === 'msgText' ? 4 : undefined).then(function () {
        that.$alarmDataColumns.push(column);
        that.$addableColumnNames = _(that.$addableColumnNames).without(name);
        that.$removableColumnNames.push(name);
      });
    } else {
      return Promise.resolve();
    }
  };

  /**
   * Removes an AlarmData column from the alarm summary table.
   *
   * @param {String} name column name
   * @returns {Promise}
   */
  AlarmConsoleViewModel.prototype.removeAlarmDataColumn = function (name) {
    var that = this;
    if (name) {
      var column = _.find(that.$alarmDataColumns, function (column) {
        return column.getName() === name;
      });
      return that.$tableModel.removeColumns([column]).then(function () {
        that.$alarmDataColumns = _.filter(that.$alarmDataColumns, function (column) {
          return column.getName() !== name;
        });
        if (_.contains(that.$defaultAlarmDataColumnNames, name)) {
          that.$addableColumnNames.push(name);
          that.$addableColumnNames.sort(function (a, b) {
            return a.toLowerCase().localeCompare(b.toLowerCase());
          });
        }
        that.$removableColumnNames = _.filter(that.$removableColumnNames, function (columnName) {
          return columnName !== name;
        });
      });
    } else {
      return Promise.resolve();
    }
  };

  /**
   * Removes all alarm data columns from the table.
   *
   * @returns {Promise}
   */
  AlarmConsoleViewModel.prototype.removeAlarmDataColumns = function () {
    var that = this,
      promises = [];
    _.each(that.getRemovableColumnNames(), function (name) {
      promises.push(that.removeAlarmDataColumn(name));
    });
    return Promise.all(promises);
  };

  /**
   * Called when the page index is changed on the pagination widget.
   *
   * @param {Number} pageIndex
   * @return {Promise}
   */
  AlarmConsoleViewModel.prototype.pageIndexChanged = function (pageIndex) {
    if (this.$pageIndex === pageIndex) {
      return Promise.resolve();
    }
    if (pageIndex < 1) {
      this.$pageIndex = 1;
    } else if (pageIndex > this.$pageCount) {
      this.$pageIndex = this.$pageCount;
    } else {
      this.$pageIndex = pageIndex;
    }
    this.$pageOffset = (this.$pageIndex - 1) * this.$pageSize;
    return Promise.resolve(dialogs.showLoading(0, restartAndLoadAlarmSummary(this)));
  };

  /**
   * Called when the page size is changed in the alarm console options.
   *
   * @param {Number} pageSize
   * @return {Promise}
   */
  AlarmConsoleViewModel.prototype.pageSizeChanged = function (pageSize) {
    if (pageSize >= 0 && pageSize !== this.$pageSize) {
      this.$pageSize = pageSize;
      this.$pageIndex = 1;
      this.$pageOffset = 0;
      return Promise.resolve(dialogs.showLoading(0, restartAndLoadAlarmSummary(this)));
    }
    return Promise.resolve();
  };

  /**
   * Called when the time range setting on the view is modified.
   *
   * @param {baja.Component} timeRange - the new time range
   * @return {Promise} Promise resolved when model has been updated based on new time range
   */
  AlarmConsoleViewModel.prototype.timeRangeChanged = function (timeRange) {
    this.$timeRange = timeRange;
    return Promise.resolve(dialogs.showLoading(0, restartAndLoadAlarmSummary(this)));
  };

  /**
   * Called when the filter setting on the view is modified.
   *
   * @param {baja.Component} filterSet - the new filter set
   * @return {Promise} Promise resolved when model has been updated based on new filter setting
   */
  AlarmConsoleViewModel.prototype.filterChanged = function (filterSet) {
    this.$filterSet = filterSet;
    return Promise.resolve(dialogs.showLoading(0, restartAndLoadAlarmSummary(this)));
  };

  /**
   * Called when sort column or sort column direction is modified
   *
   * @param {String} column - column name
   * @param {Boolean} sortDescending - flag indicating sort direction
   * @return {Promise} Promise resolved when model has been updated based on sorted column
   */
  AlarmConsoleViewModel.prototype.sortColumnChanged = function (column, sortDescending) {
    this.$column = column;
    this.$sortDesc = sortDescending || false;
    return Promise.resolve(dialogs.showLoading(0, loadAlarmSummary(this)));
  };

  /**
   * Called when one or more alarms are acknowledged by source or id.
   *
   * @param {Object} params
   * @return {Promise.<Array.<String>>} Promise that resolves with an array of alarm classes
   * for which alarm acknowledgment failed
   */
  AlarmConsoleViewModel.prototype.ackAlarms = function (params) {
    return Promise.resolve(this.$support && this.$support.ackAlarms(params)).then(this.$checkFailures);
  };

  /**
   * Called when one or more alarms are force cleared by source or id.
   *
   * @param {Object} params
   * @return {Promise.<Array.<String>>} Promise that resolves with an array of alarm classes
   * for which alarm acknowledgment failed
   */
  AlarmConsoleViewModel.prototype.forceClearAlarms = function (params) {
    return Promise.resolve(this.$support && this.$support.forceClearAlarms(params)).then(this.$checkFailures);
  };

  /**
   * Called when notes are added to one or more alarms by source or id.
   *
   * @param {Object} params
   * @return {Promise.<Array.<String>>} Promise that resolves with an array of alarm classes
   * for which note addition failed
   */
  AlarmConsoleViewModel.prototype.addNoteToAlarms = function (params) {
    return Promise.resolve(this.$support && this.$support.addNoteToAlarms(params)).then(this.$checkFailures);
  };

  /**
   * Loads the alarm summary for an optional list of alarm sources. If a source array is specified this
   * method currently only loads the summary for the first source in the array
   *
   * @param {Object} params
   * @param {Array.<String>} [params.sources] array of string encoded baja.Ords representing alarm sources
   * @return {Promise} Promise resolved when the summary has been loaded
   */
  AlarmConsoleViewModel.prototype.loadAlarmSummary = function (params) {
    var that = this,
      sources = params && params.sources,
      load = false;
    if (_.isArray(sources)) {
      if (sources.length) {
        that.$source = sources[0];
        that.$singleSource = true;
        that.$pageIndex = 1;
        that.$pageOffset = 0;
        load = true;
      }
    } else {
      that.$source = 'null';
      that.$singleSource = false;
      that.$pageIndex = 1;
      that.$pageOffset = 0;
      load = true;
    }
    return Promise.resolve(load && loadAlarmSummary(that));
  };

  //////////////////////////////////////////////////////////////////////////
  //Private
  //////////////////////////////////////////////////////////////////////////
  AlarmConsoleViewModel.prototype.$getSupport = function () {
    return this.$support;
  };
  AlarmConsoleViewModel.prototype.$getAlarmConsoleOptions = function () {
    return this.$support && this.$support.getAlarmConsoleOptions() || {};
  };
  AlarmConsoleViewModel.prototype.$getTableModel = function () {
    return this.$tableModel;
  };
  AlarmConsoleViewModel.prototype.$getTimeRange = function () {
    return this.$timeRange;
  };
  AlarmConsoleViewModel.prototype.$getFilterSet = function () {
    return this.$filterSet;
  };
  AlarmConsoleViewModel.prototype.$getSourceCount = function () {
    return this.$sourceCount;
  };
  AlarmConsoleViewModel.prototype.$getAlarmCount = function () {
    return this.$alarmCount;
  };
  AlarmConsoleViewModel.prototype.$getPageIndex = function () {
    return this.$pageIndex;
  };
  AlarmConsoleViewModel.prototype.$getPageOffset = function () {
    return this.$pageOffset;
  };
  AlarmConsoleViewModel.prototype.$getPageSize = function () {
    return this.$pageSize;
  };
  AlarmConsoleViewModel.prototype.$isSingleSourceModel = function () {
    return this.$singleSource;
  };
  AlarmConsoleViewModel.prototype.$getAudioHandler = function () {
    return this.$audioHandler;
  };
  AlarmConsoleViewModel.prototype.$getAudioQueue = function () {
    return ALARM_AUDIO_QUEUE;
  };
  AlarmConsoleViewModel.prototype.$getSortColumn = function () {
    return this.$column;
  };
  AlarmConsoleViewModel.prototype.$getSortDescending = function () {
    return this.$sortDesc;
  };
  return AlarmConsoleViewModel;
});
