/*
 * Decompiled with CFR 0.152.
 */
package com.tridium.cloudLink.auth;

import com.tridium.cloudLink.auth.IKeyStoreHelper;
import com.tridium.nre.security.SecretChars;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.util.Calendar;
import java.util.Date;
import java.util.function.Supplier;
import java.util.logging.Logger;
import javax.baja.nre.security.IKeyPurpose;
import javax.baja.nre.security.IX509CertificateEntry;
import javax.baja.security.BPassword;
import javax.baja.security.crypto.IKeyStore;
import javax.baja.security.crypto.X509CertificateFactory;

public abstract class AbstractKeyStoreHelper
implements IKeyStoreHelper {
    protected static final Logger log = Logger.getLogger("cloudLink.auth.key");
    protected static final int KEY_DURATION_YEARS = 20;
    protected boolean initialized;
    protected PublicKey publicKey;
    protected IKeyStore keyStore;
    protected String alias;
    protected BPassword password;
    protected String subject;
    protected Supplier<KeyPairGenerator> keyPairGenSupplier;

    protected AbstractKeyStoreHelper(IKeyStore keyStore, String alias, BPassword password, String subject, Supplier<KeyPairGenerator> keyPairGenSupplier) {
        this.keyStore = keyStore;
        this.alias = alias;
        this.password = password;
        this.subject = subject;
        this.keyPairGenSupplier = keyPairGenSupplier;
    }

    @Override
    public synchronized PublicKey getPublicKey() throws Exception {
        this.initialize();
        return this.publicKey;
    }

    @Override
    public synchronized void deleteKeyEntry() throws Exception {
        log.config(() -> String.format("Deleting key entry with alias %s", this.alias));
        if (this.initialized && this.keyStore.isKeyEntry(this.alias)) {
            this.keyStore.deleteEntry(this.alias);
            this.keyStore.save();
        } else {
            log.config(() -> String.format("Unable to delete, key entry with alias %s", this.alias));
        }
        this.publicKey = null;
        this.initialized = false;
    }

    @Override
    public synchronized void changeKeyEntryPassword(BPassword currentPassword, BPassword newPassword) throws Exception {
        if (!this.initialized) {
            throw new Exception("The key store helper has not been initialized.");
        }
        if (!this.password.validate(currentPassword)) {
            throw new Exception("The current password parameter does not match the key entry password.");
        }
        log.config(() -> String.format("Changing the certificate password for key with alias %s", this.alias));
        try {
            AbstractKeyStoreHelper.changePassword(this.keyStore, this.alias, currentPassword, newPassword);
        }
        catch (Exception certChangeExcept) {
            throw new Exception("Certificate password change failed.", certChangeExcept);
        }
        try {
            this.keyStore.save();
            this.password = newPassword;
            this.keysChanged();
        }
        catch (Exception saveExcept) {
            log.fine("Reverting the certificate password change.");
            AbstractKeyStoreHelper.changePassword(this.keyStore, this.alias, newPassword, currentPassword);
            throw new Exception("Keystore save failed.", saveExcept);
        }
    }

    protected final void finalize() throws Throwable {
    }

    protected void initialize() throws Exception {
        boolean hasKeyEntry = this.keyStore.isKeyEntry(this.alias);
        if (hasKeyEntry && !this.initialized) {
            Key privateKey;
            log.config(() -> String.format("Loading existing key entry for alias %s and subject %s", this.alias, this.subject));
            this.publicKey = AbstractKeyStoreHelper.getPublicKeyFromCertificate(this.keyStore, this.alias);
            try (SecretChars secretChars = this.password.getSecretChars();){
                privateKey = this.keyStore.getKey(this.alias, secretChars.get());
            }
            AbstractKeyStoreHelper.validateKeyEntry(this.alias, this.publicKey, privateKey);
            this.keysChanged();
        } else if (!hasKeyEntry) {
            Key privateKey;
            AbstractKeyStoreHelper.createKeyEntry(this.keyStore, this.keyPairGenSupplier, this.alias, this.password, this.subject);
            this.publicKey = AbstractKeyStoreHelper.getPublicKeyFromCertificate(this.keyStore, this.alias);
            try (SecretChars secretChars = this.password.getSecretChars();){
                privateKey = this.keyStore.getKey(this.alias, secretChars.get());
            }
            AbstractKeyStoreHelper.validateKeyEntry(this.alias, this.publicKey, privateKey);
            this.keysChanged();
        }
        this.initialized = true;
    }

    protected void keysChanged() throws Exception {
    }

    protected static PublicKey getPublicKeyFromCertificate(IKeyStore keyStore, String alias) throws Exception {
        X509Certificate cert = keyStore.getCertificate(alias);
        return cert.getPublicKey();
    }

    protected static void validateKeyEntry(String alias, PublicKey publicKey, Key privateKey) {
        if (publicKey == null) {
            throw new IllegalStateException(String.format("The certificate for key entry %s does not contain a valid public key", alias));
        }
        if (!(privateKey instanceof PrivateKey)) {
            throw new IllegalStateException(String.format("The keystore does not contain a valid private key for keystore entry %s", alias));
        }
    }

    protected static KeyPair createKeyEntry(IKeyStore keyStore, Supplier<KeyPairGenerator> kpGenSupplier, String alias, BPassword password, String subject) throws Exception {
        log.config("Creating a new key pair...");
        KeyPairGenerator keyPairGen = kpGenSupplier.get();
        KeyPair keyPair = keyPairGen.generateKeyPair();
        try (SecretChars secretChars = password.getSecretChars();){
            keyStore.setKeyEntry(alias, (Key)keyPair.getPrivate(), secretChars.get(), AbstractKeyStoreHelper.makeCertificateChain(keyPair, subject));
            keyStore.save();
        }
        log.config(() -> String.format("A new key pair was created for alias %s", alias));
        return keyPair;
    }

    protected static X509Certificate[] makeCertificateChain(KeyPair keyPair, String id) throws Exception {
        Calendar cal = Calendar.getInstance();
        Date notBefore = cal.getTime();
        cal.set(1, cal.get(1) + 20);
        Date notAfter = cal.getTime();
        IX509CertificateEntry entry = X509CertificateFactory.getInstance().generateSelfSignedCert(keyPair, "cloudLinkIdentity", String.format("CN=%s,O=Tridium,C=US", id), String.format("CN=%s,O=Tridium,C=US", id), notBefore, notAfter, IKeyPurpose.CLIENT_CERT, null, null);
        return entry.getCertificates();
    }

    private static void changePassword(IKeyStore keyStore, String alias, BPassword currentPassword, BPassword newPassword) throws Exception {
        try (SecretChars currentSecret = currentPassword.getSecretChars();
             SecretChars newSecret = newPassword.getSecretChars();){
            Key privateKey = keyStore.getKey(alias, currentSecret.get());
            X509Certificate[] certChain = keyStore.getCertificateChain(alias);
            keyStore.setKeyEntry(alias, privateKey, newSecret.get(), certChain);
        }
    }
}

